/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.model;

import com.google.common.collect.ImmutableList;
import de.fhlintstone.accessors.IAccessorCache;
import java.util.Optional;
import lombok.AccessLevel;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import org.hl7.fhir.r4b.model.CodeSystem.ConceptDefinitionComponent;

/**
 * Implementation of {@link IConceptDefinitionComponentAccessor} for FHIR
 * version R4B. i.e. wrapping {@link ConceptDefinitionComponent}.
 */
@EqualsAndHashCode(callSuper = true)
public final class ConceptDefinitionComponentAccessorR4B extends BackboneElementAccessorR4B
        implements IConceptDefinitionComponentAccessor {

    @Getter(AccessLevel.PRIVATE)
    private final ConceptDefinitionComponent concept;

    /**
     * Creates a new wrapper.
     *
     * @param cache the cache this accessor is managed by
     * @param concept the concept to wrap
     */
    public ConceptDefinitionComponentAccessorR4B(IAccessorCache cache, ConceptDefinitionComponent concept) {
        super(cache, concept);
        this.concept = concept;
    }

    @Override
    public Optional<String> getDisplay() {
        if (this.concept.hasDisplay()) {
            return Optional.of(this.concept.getDisplay());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<String> getCode() {
        if (this.concept.hasCode()) {
            return Optional.of(this.concept.getCode());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public ImmutableList<IConceptPropertyComponentAccessor> getProperty() {
        return this.concept.getProperty().stream()
                .map(component -> getCache().getAccessor(component, IConceptPropertyComponentAccessor.class))
                .collect(ImmutableList.toImmutableList());
    }

    @Override
    public ImmutableList<IConceptDefinitionComponentAccessor> getChildConcept() {
        return this.concept.getConcept().stream()
                .map(component -> getCache().getAccessor(component, IConceptDefinitionComponentAccessor.class))
                .collect(ImmutableList.toImmutableList());
    }
}
