/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.model;

import com.google.common.collect.ImmutableList;
import de.fhlintstone.accessors.IAccessorCache;
import java.util.Optional;
import lombok.AccessLevel;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import org.hl7.fhir.r4.model.CodeSystem;

/**
 * Implementation of {@link ICodeSystemAccessor} for FHIR version R4. i.e.
 * wrapping {@link CodeSystem}.
 */
@EqualsAndHashCode(callSuper = true)
public final class CodeSystemAccessorR4 extends MetadataResourceAccessorR4 implements ICodeSystemAccessor {

    @Getter(AccessLevel.PRIVATE)
    private final CodeSystem codeSystem;

    /**
     * Creates a new wrapper.
     *
     * @param cache the cache this accessor is managed by
     * @param codeSystem the CodeSystem instance to wrap
     */
    public CodeSystemAccessorR4(IAccessorCache cache, CodeSystem codeSystem) {
        super(cache, codeSystem);
        this.codeSystem = codeSystem;
    }

    @Override
    public Optional<String> getName() {
        if (this.codeSystem.hasName()) {
            return Optional.of(this.codeSystem.getName());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<String> getUrl() {
        if (this.codeSystem.hasUrl()) {
            return Optional.of(this.codeSystem.getUrl());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<String> getDescription() {
        if (this.codeSystem.hasDescription()) {
            return Optional.of(this.codeSystem.getDescription());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public ImmutableList<IConceptDefinitionComponentAccessor> getConcept() {
        return this.codeSystem.getConcept().stream()
                .map(component -> getCache().getAccessor(component, IConceptDefinitionComponentAccessor.class))
                .collect(ImmutableList.toImmutableList());
    }
}
