/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.model;

import com.google.common.collect.ImmutableList;
import de.fhlintstone.accessors.Accessor;
import de.fhlintstone.accessors.IAccessorCache;
import java.util.Optional;
import lombok.AccessLevel;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import org.hl7.fhir.r4b.model.Base;

/**
 * Implementation of {@link IBaseAccessor} for FHIR release R4B.
 */
@EqualsAndHashCode(callSuper = true)
public class BaseAccessorR4B extends Accessor implements IBaseAccessor {

    @Getter(AccessLevel.PRIVATE)
    private final Base base;

    /**
     * Creates a new wrapper.
     *
     * @param cache the cache this accessor is managed by
     * @param base the FHIR Base to wrap
     */
    public BaseAccessorR4B(IAccessorCache cache, Base base) {
        super(cache);
        this.base = base;
    }

    @Override
    public String toString() {
        return this.base.toString();
    }

    @Override
    public Optional<String> getId() {
        return Optional.ofNullable(this.base.getIdBase());
    }

    @Override
    public Optional<String> getFhirType() {
        return Optional.ofNullable(this.base.fhirType());
    }

    @Override
    public ImmutableList<IPropertyAccessor> children() {
        return this.base.children().stream()
                .map(child -> getCache().getAccessor(child, IPropertyAccessor.class))
                .collect(ImmutableList.toImmutableList());
    }

    @Override
    public Optional<IPropertyAccessor> getChildByName(String name) {
        final var child = this.base.getChildByName(name);
        return child == null ? Optional.empty() : Optional.of(getCache().getAccessor(child, IPropertyAccessor.class));
    }
}
