/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.implementations;

import com.google.common.collect.ImmutableList;
import java.net.URI;
import java.util.stream.Collectors;
import lombok.Builder;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NonNull;
import lombok.Singular;
import lombok.extern.slf4j.XSlf4j;

/**
 * Default implementation of {@link ITypeSpecification}
 */
@EqualsAndHashCode
@Builder(setterPrefix = "with")
@XSlf4j
public final class TypeSpecification implements ITypeSpecification {

    @Getter
    @NonNull
    private final TypeReference typeCode;

    @Getter
    @Singular(value = "profile")
    private final ImmutableList<TypeReference> profiles;

    @Getter
    @Singular(value = "targetProfile")
    private final ImmutableList<TypeReference> targetProfiles;

    @Override
    public boolean hasProfiles() {
        return !this.profiles.isEmpty();
    }

    @Override
    public boolean hasTargetProfiles() {
        return !this.targetProfiles.isEmpty();
    }

    /**
     * Builder to create instances of {@link TypeSpecification}.
     */
    public static class TypeSpecificationBuilder {

        /**
         * Convenience method to add a typeCode.
         * This should be named withTypeCode, but that conflicts with the generated withTypeCode for some reason...
         * @param code the type code
         * @param canonical the canonical URL
         * @return the builder instance for method chaining
         */
        public TypeSpecificationBuilder withType(String code, URI canonical) {
            return this.withTypeCode(new TypeReference(code, canonical));
        }

        /**
         * Convenience method to add a profile.
         * @param code the type code
         * @param canonical the canonical URL
         * @return the builder instance for method chaining
         */
        public TypeSpecificationBuilder withProfile(String code, URI canonical) {
            return this.withProfile(new TypeReference(code, canonical));
        }

        /**
         * Convenience method to add a target profile.
         * @param code the type code
         * @param canonical the canonical URL
         * @return the builder instance for method chaining
         */
        public TypeSpecificationBuilder withTargetProfile(String code, URI canonical) {
            return this.withTargetProfile(new TypeReference(code, canonical));
        }
    }

    @Override
    public String toString() {
        final var builder = new StringBuilder();
        builder.append("TypeSpecification [typeCode=" + this.typeCode.code() + " (canonical="
                + this.typeCode.canonical() + ")");
        if (hasProfiles()) {
            builder.append(", profiles = [");
            builder.append(this.profiles.stream()
                    .map(p -> p.code() + " (canonical=" + p.canonical() + ")")
                    .collect(Collectors.joining(", ")));
            builder.append("]");
            ;
        }
        if (hasTargetProfiles()) {
            builder.append(", profiles = [");
            builder.append(this.targetProfiles.stream()
                    .map(p -> p.code() + " (canonical=" + p.canonical() + ")")
                    .collect(Collectors.joining(", ")));
            builder.append("]");
            ;
        }
        builder.append("]");
        return builder.toString();
    }
}
