/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.implementations;

import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableList;
import com.palantir.javapoet.TypeName;
import java.net.URI;
import java.util.ArrayList;
import java.util.Optional;
import lombok.Builder;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NonNull;
import lombok.extern.slf4j.XSlf4j;

/**
 * Default implementation of {@link IMappedType}
 */
@EqualsAndHashCode
@Builder(setterPrefix = "with")
@XSlf4j
public class MappedType implements IMappedType {

    @Getter
    @NonNull
    private final String typeCode;

    @Getter
    @NonNull
    private final URI typeCodeURI;

    @Getter
    @Builder.Default
    private final Optional<String> profile = Optional.empty();

    @Getter
    @Builder.Default
    private final Optional<URI> profileURI = Optional.empty();

    @Getter
    @NonNull
    private final TypeName type;

    @Override
    public String toString() {
        final var sb = new StringBuilder();
        sb.append("MappedType [typeCode=" + this.typeCode + ", typeCodeURI=" + this.typeCodeURI);
        if (this.profile.isPresent()) {
            sb.append(", profile=" + this.profile);
        }
        if (this.profileURI.isPresent()) {
            sb.append(", profileURI=" + this.profileURI);
        }
        sb.append(" --> type=" + this.type + "]");
        return sb.toString();
    }

    /**
     * Creates a collection of mapped types for the type and profile references of a {@link ITypeSpecification} and a given type name.
     * If only a type code is present, a single mapped type will be issued. If profiles are present, a separate mapped type is issued for each profile.
     *
     * @param typeSpecification the {@link ITypeSpecification} to extract the type data from
     * @param typeName the {@link TypeName} to add to the mapped type
     * @return a list of the mapped types
     */
    public static ImmutableCollection<IMappedType> forTypeSpecification(
            ITypeSpecification typeSpecification, TypeName typeName) {
        logger.entry(typeSpecification, typeName);
        final var result = new ArrayList<IMappedType>();
        final var typeCode = typeSpecification.getTypeCode();
        if (typeSpecification.hasProfiles()) {
            for (final var profile : typeSpecification.getProfiles()) {
                result.add(builder()
                        .withTypeCode(typeCode.code())
                        .withTypeCodeURI(typeCode.canonical())
                        .withProfile(Optional.of(profile.code()))
                        .withProfileURI(Optional.of(profile.canonical()))
                        .withType(typeName)
                        .build());
            }
        } else {
            result.add(builder()
                    .withTypeCode(typeCode.code())
                    .withTypeCodeURI(typeCode.canonical())
                    .withType(typeName)
                    .build());
        }
        return logger.exit(ImmutableList.copyOf(result));
    }
}
