/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.implementations;

import ca.uhn.fhir.context.FhirVersionEnum;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableList;
import com.palantir.javapoet.TypeName;
import de.fhlintstone.accessors.UnsupportedFHIRVersionException;
import de.fhlintstone.accessors.model.PrimitiveDatatype;
import de.fhlintstone.process.IContextProvider;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collection;
import java.util.Optional;
import javax.inject.Inject;
import javax.inject.Named;
import lombok.AccessLevel;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.extern.slf4j.XSlf4j;

/**
 * Release-independent implementation of {@link IFrameworkTypeLocator} that
 * can be injected into clients. It defers to the actual release-dependent
 * implementation that will be create at the latest possible point in time and
 * will be cached.
 */
@Named
@EqualsAndHashCode
@XSlf4j
public class LocatorSelector implements IFrameworkTypeLocator {

    @Getter(AccessLevel.PRIVATE)
    private final IContextProvider contextProvider;

    /**
     * Constructor for dependency injection.
     *
     * @param contextProvider the {@link IContextProvider} to use
     */
    @Inject
    public LocatorSelector(IContextProvider contextProvider) {
        super();
        this.contextProvider = contextProvider;
    }

    @SuppressWarnings("java:S4738") // Java supplier does not support memoization
    private final Supplier<IFrameworkTypeLocator> locatorSupplier = Suppliers.memoize(() -> {
        logger.entry();
        IFrameworkTypeLocator result;
        final FhirVersionEnum fhirVersion =
                getContextProvider().getFhirVersion().orElseThrow();
        switch (fhirVersion) {
            case R4:
                result = new FrameworkTypeLocatorR4(getContextProvider());
                break;
            // TODO #15 provide type locator for release R4B
            // TODO #15 provide type locator for release R5
            default:
                throw logger.throwing(new UnsupportedFHIRVersionException(fhirVersion));
        }
        return logger.exit(result);
    });

    @Override
    public Optional<TypeName> determineType(URI resourceURI) {
        return this.locatorSupplier.get().determineType(resourceURI);
    }

    @Override
    public boolean isFrameworkType(TypeName type) {
        return this.locatorSupplier.get().isFrameworkType(type);
    }

    @Override
    public ImmutableCollection<IMappedType> determineFrameworkTypes(ITypeSpecification typeSpecification) {
        return this.locatorSupplier.get().determineFrameworkTypes(typeSpecification);
    }

    @Override
    public ImmutableCollection<IMappedType> determineFrameworkTypes(Collection<ITypeSpecification> typeSpecifications) {
        return this.locatorSupplier.get().determineFrameworkTypes(typeSpecifications);
    }

    @Override
    public TypeName getPrimitiveType(PrimitiveDatatype type) {
        return this.locatorSupplier.get().getPrimitiveType(type);
    }

    @Override
    public TypeName getConfigurationType() {
        return this.locatorSupplier.get().getConfigurationType();
    }

    @Override
    public TypeName getBackboneElementType() {
        return this.locatorSupplier.get().getBackboneElementType();
    }

    @Override
    public TypeName getExtensionType() {
        return this.locatorSupplier.get().getExtensionType();
    }

    @Override
    public TypeName getBaseType() {
        return this.locatorSupplier.get().getBaseType();
    }

    @Override
    public TypeName getGenericType() {
        return this.locatorSupplier.get().getGenericType();
    }

    @Override
    public TypeName getCodingType() {
        return this.locatorSupplier.get().getCodingType();
    }

    @Override
    public TypeName getPropertyType() {
        return this.locatorSupplier.get().getPropertyType();
    }

    @Override
    public boolean isDerivedFromPrimitiveType(TypeName type) {
        return this.locatorSupplier.get().isDerivedFromPrimitiveType(type);
    }

    @Override
    public Optional<String> getBaseCastingMethod(TypeName targetType) {
        return this.locatorSupplier.get().getBaseCastingMethod(targetType);
    }

    @Override
    public URI makeAbsoluteStructureDefinitionReference(String reference) throws URISyntaxException {
        return this.locatorSupplier.get().makeAbsoluteStructureDefinitionReference(reference);
    }

    @Override
    public URI makeAbsoluteValueSetReference(String reference) throws URISyntaxException {
        return this.locatorSupplier.get().makeAbsoluteValueSetReference(reference);
    }

    @Override
    public Optional<TypeName> determinePrimitiveType(TypeName frameworkType) {
        return this.locatorSupplier.get().determinePrimitiveType(frameworkType);
    }

    @Override
    public ImmutableList<Constructor> determineConstructors(TypeName frameworkType) throws IllegalArgumentException {
        return this.locatorSupplier.get().determineConstructors(frameworkType);
    }
}
