/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.implementations;

import com.google.common.collect.ImmutableList;
import java.net.URI;

/**
 * A specification of the FHIR type (or rather types) that can be applied to an ElementDefinition.
 *
 * This consists of exactly one type code that may have one or multiple profiles applied to it
 * (e.g. an Address that must conform to a specific profile).
 *
 * If the type code specifies a Reference or a canonical, the profile is applied to the Reference or
 * canonical itself (e.g. to add an extension to the reference). In this case, a target profile can
 * be specified that then is applied to the referred resource.
 */
public interface ITypeSpecification {

    /**
     * A reference to a FHIR type that can be represented as a code (e.g. "string" or a canonical URI (e.g.
     * "http://hl7.org/fhir/StructureDefinition/string").
     * @param code the (original) type reference, e.g. "string"
     * @param canonical the canonical URI, e.g. e.g. "http://hl7.org/fhir/StructureDefinition/string"
     */
    record TypeReference(String code, URI canonical) {}

    /**
     * Returns the type code.
     *
     * @return the type code
     * @see <a href="https://hl7.org/fhir/R4/elementdefinition-definitions.html#ElementDefinition.type.code">ElementDefinition.type.code</a>
     */
    TypeReference getTypeCode();

    /**
     * Determines whether profiles for the type code have been specified.
     *
     * @return <code>true</code> if profiles for the type code have been specified
     */
    boolean hasProfiles();

    /**
     * Returns the profile identifiers, if any have been specified.
     *
     * @return the profile identifiers, if any have been specified
     * @see <a href="https://hl7.org/fhir/R4/elementdefinition-definitions.html#ElementDefinition.type.profile">ElementDefinition.type.profile</a>
     */
    ImmutableList<TypeReference> getProfiles();

    /**
     * Determines whether target profiles for the referred type have been specified.
     *
     * @return <code>true</code> if target profiles for the referred type have been specified
     */
    boolean hasTargetProfiles();

    /**
     * Returns the target profile identifiers, if any have been specified.
     *
     * @return the target profile identifiers, if any have been specified
     * @see <a href="https://hl7.org/fhir/R4/elementdefinition-definitions.html#ElementDefinition.type.profile">ElementDefinition.type.profile</a>
     */
    ImmutableList<TypeReference> getTargetProfiles();
}
