/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.implementations;

import com.palantir.javapoet.TypeName;
import java.net.URI;
import java.util.Optional;

/**
 * A representation of a FHIR type that has been identified using a profile or a
 * type code and mapped to a Java type.
 */
public interface IMappedType {

    /**
     * Returns the original value of the type code.
     *
     * @return the original value of the type code
     * @see <a href="https://hl7.org/fhir/R4/elementdefinition-definitions.html#ElementDefinition.type.code">ElementDefinition.type.code</a>
     */
    String getTypeCode();

    /**
     * Returns the expanded URI of the type code.
     *
     * @return the expanded URI of the type code
     * @see <a href="https://hl7.org/fhir/R4/elementdefinition-definitions.html#ElementDefinition.type.code">ElementDefinition.type.code</a>
     */
    URI getTypeCodeURI();

    /**
     * Returns the original value of the type profile, if the type has been
     * identified using the profile.
     *
     * @return the original value of the type profile, if the type has been
     *         identified using the profile
     * @see <a href="https://hl7.org/fhir/R4/elementdefinition-definitions.html#ElementDefinition.type.profile">ElementDefinition.type.profile</a>
     */
    Optional<String> getProfile();

    /**
     * Returns the expanded URI of the type profile, if the type has been identified
     * using the profile.
     *
     * @return the expanded URI of the type profile, if the type has been identified
     *         using the profile
     * @see <a href="https://hl7.org/fhir/R4/elementdefinition-definitions.html#ElementDefinition.type.profile">ElementDefinition.type.profile</a>
     */
    Optional<URI> getProfileURI();

    /**
     * Returns the Java type selected to represent the FHIR type.
     *
     * @return the Java type selected to represent the FHIR type
     */
    TypeName getType();
}
