/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.implementations;

import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableList;
import com.palantir.javapoet.TypeName;
import de.fhlintstone.accessors.model.PrimitiveDatatype;
import de.fhlintstone.generator.structuredefinition.intermediate.ITypeMapper;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collection;
import java.util.Optional;

/**
 * Utility to determine the HAPI framework class that represents a resource.
 */
public interface IFrameworkTypeLocator {

    /**
     * Determines the framework type (if any) to represent the resource
     *
     * @param resourceURI the URI (including the http prefix!)
     * @return the framework type
     */
    Optional<TypeName> determineType(URI resourceURI);

    /**
     * Determines whether the type name given represents a HAPI FHIR framework-supplied type.
     *
     * @param type the name of the type to examine
     * @return <code>true</code> if the type name given represents a HAPI FHIR framework-supplied type
     */
    boolean isFrameworkType(TypeName type);

    /**
     * Determines the HAPI types that represent the FHIR types specified as an argument. Note that this method might
     * return multiple types (if the specification lists multiple profiles) or an empty collection if no matching type
     * was found.
     *
     * Caution: This method will ONLY resolve the HAPI built-in types. The {@link ITypeMapper} is used to combine this
     * information with the configured names of the generated types.
     *
     * @param typeSpecification the type specification to evaluate
     * @return a collection of the HAPI types that can be used to represent the element
     */
    ImmutableCollection<IMappedType> determineFrameworkTypes(ITypeSpecification typeSpecification);

    /**
     * Determines the HAPI types that represent the FHIR types specified as an argument. Note that this method might
     * return multiple types (if multiple specifications are used or a specification lists multiple profiles) or an
     * empty collection if no matching type was found.
     *
     * Caution: This method will ONLY resolve the HAPI built-in types. The {@link ITypeMapper} is used to combine this
     * information with the configured names of the generated types.
     *
     * @param typeSpecifications the type specifications to evaluate
     * @return a collection of the HAPI types that can be used to represent the element
     */
    ImmutableCollection<IMappedType> determineFrameworkTypes(Collection<ITypeSpecification> typeSpecifications);

    /**
     * Determines the framework implementation of a {@link PrimitiveDatatype}.
     *
     * @param type the {@link PrimitiveDatatype}
     * @return the framework implementation of the type
     */
    TypeName getPrimitiveType(PrimitiveDatatype type);

    /**
     * Determines the framework configuration class.
     *
     * @return the framework configuration class
     */
    TypeName getConfigurationType();

    /**
     * Determines the framework type to be used as a backbone element.
     *
     * @return the framework type to be used as a backbone element
     */
    TypeName getBackboneElementType();

    /**
     * Determines the framework type to be used for a simple extension.
     *
     * @return the framework type to be used for a simple extension
     */
    TypeName getExtensionType();

    /**
     * Determines the framework type used as a base object type
     *
     * @return the framework type used as a base object type
     */
    TypeName getBaseType();

    /**
     * Determines the framework type to be used as a generic type (e.g. for multi-typed attributes).
     *
     * @return the framework type to be used as a generic type
     */
    TypeName getGenericType();

    /**
     * Determines the class name used to represent a FHIR Coding.
     *
     * @return the class name used to represent a FHIR Coding
     */
    TypeName getCodingType();

    /**
     * Determines the framework type to be used as a property type.
     *
     * @return the framework type to be used as a property type
     */
    TypeName getPropertyType();

    /**
     * Determines whether the type name provided is derived from the HAPI PrimititveType. Note that this method can only
     * be used for HAPI Framework classes since the class needs to be accessible via reflection.
     *
     * @param type the type name to check
     * @return <code>true</code> if the type name provided is derived from the HAPI PrimititveType
     * @throws IllegalArgumentException if the class is not a framework class, i.e. is not accessible via reflection
     */
    boolean isDerivedFromPrimitiveType(TypeName type) throws IllegalArgumentException;

    /**
     * Determines the name of the method in the Base implementation that takes a Base reference, performs a type check
     * and cast and returns the value of that type
     *
     * @param targetType the target type to cast to
     * @return the name of the casting method, or an empty value if no method exists
     */
    Optional<String> getBaseCastingMethod(TypeName targetType);

    /**
     * Creates a URI for the StructureDefinition reference, adding the prefix for FHIR default types if necessary.
     *
     * @param reference
     * @return a URI for the StructureDefinition reference
     * @throws URISyntaxException
     */
    URI makeAbsoluteStructureDefinitionReference(String reference) throws URISyntaxException;

    /**
     * Creates a URI for the ValueSet reference, adding the prefix for FHIR default ValueSets if necessary.
     *
     * @param reference
     * @return a URI for the ValueSet reference
     * @throws URISyntaxException
     */
    URI makeAbsoluteValueSetReference(String reference) throws URISyntaxException;

    /**
     * Determines the basic Java type (e.g. Integer, String) for a framework element type (e.g. IntegerType,
     * StringType).
     *
     * @param frameworkType the framework element type to check
     * @return the matching basic Java type, if any
     */
    Optional<TypeName> determinePrimitiveType(TypeName frameworkType);

    /**
     * Describes a parameter of a constructor.
     *
     * @see #determineConstructors(TypeName)
     */
    record ConstructorParameter(String name, TypeName type) {}

    /**
     * Describes a constructor of a framework class.
     *
     * @see #determineConstructors(TypeName)
     */
    record Constructor(ImmutableList<ConstructorParameter> parameters, boolean deprecated) {}

    /**
     * Determines the constructors of the framework type. This method checks for some special cases (like known
     * parameterized classes) and handles these. It will revert to a basic, reflection-based approach if no special
     * handling is implemented.
     *
     * @param frameworkType the type to examine
     * @return the constructors of the framework type
     * @throws IllegalArgumentException if the argument does not refer to a resolvable framework class
     *
     */
    ImmutableList<Constructor> determineConstructors(TypeName frameworkType) throws IllegalArgumentException;
}
