/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.implementations;

import com.google.common.collect.ImmutableList;
import com.palantir.javapoet.ArrayTypeName;
import com.palantir.javapoet.ClassName;
import com.palantir.javapoet.ParameterizedTypeName;
import com.palantir.javapoet.TypeName;
import de.fhlintstone.accessors.model.PrimitiveDatatype;
import de.fhlintstone.process.IContextProvider;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.Optional;
import lombok.extern.slf4j.XSlf4j;
import org.hl7.fhir.r4.model.Address;
import org.hl7.fhir.r4.model.Annotation;
import org.hl7.fhir.r4.model.Attachment;
import org.hl7.fhir.r4.model.BackboneElement;
import org.hl7.fhir.r4.model.Base;
import org.hl7.fhir.r4.model.Base64BinaryType;
import org.hl7.fhir.r4.model.BooleanType;
import org.hl7.fhir.r4.model.Bundle.BundleType;
import org.hl7.fhir.r4.model.Bundle.HTTPVerb;
import org.hl7.fhir.r4.model.Bundle.SearchEntryMode;
import org.hl7.fhir.r4.model.CanonicalType;
import org.hl7.fhir.r4.model.CodeType;
import org.hl7.fhir.r4.model.CodeableConcept;
import org.hl7.fhir.r4.model.Coding;
import org.hl7.fhir.r4.model.Composition.CompositionStatus;
import org.hl7.fhir.r4.model.Composition.DocumentConfidentiality;
import org.hl7.fhir.r4.model.Composition.DocumentRelationshipType;
import org.hl7.fhir.r4.model.Composition.SectionMode;
import org.hl7.fhir.r4.model.Configuration;
import org.hl7.fhir.r4.model.ContactDetail;
import org.hl7.fhir.r4.model.ContactPoint;
import org.hl7.fhir.r4.model.ContactPoint.ContactPointSystem;
import org.hl7.fhir.r4.model.ContactPoint.ContactPointUse;
import org.hl7.fhir.r4.model.Contributor;
import org.hl7.fhir.r4.model.Coverage.CoverageStatus;
import org.hl7.fhir.r4.model.DataRequirement;
import org.hl7.fhir.r4.model.DateTimeType;
import org.hl7.fhir.r4.model.DateType;
import org.hl7.fhir.r4.model.DecimalType;
import org.hl7.fhir.r4.model.Dosage;
import org.hl7.fhir.r4.model.Duration;
import org.hl7.fhir.r4.model.Element;
import org.hl7.fhir.r4.model.ElementDefinition;
import org.hl7.fhir.r4.model.EnumFactory;
import org.hl7.fhir.r4.model.Enumeration;
import org.hl7.fhir.r4.model.Enumerations.AdministrativeGender;
import org.hl7.fhir.r4.model.Expression;
import org.hl7.fhir.r4.model.Extension;
import org.hl7.fhir.r4.model.HumanName;
import org.hl7.fhir.r4.model.HumanName.NameUse;
import org.hl7.fhir.r4.model.IdType;
import org.hl7.fhir.r4.model.Identifier;
import org.hl7.fhir.r4.model.Identifier.IdentifierUse;
import org.hl7.fhir.r4.model.InstantType;
import org.hl7.fhir.r4.model.IntegerType;
import org.hl7.fhir.r4.model.MarkdownType;
import org.hl7.fhir.r4.model.MarketingStatus;
import org.hl7.fhir.r4.model.Meta;
import org.hl7.fhir.r4.model.Money;
import org.hl7.fhir.r4.model.Narrative;
import org.hl7.fhir.r4.model.OidType;
import org.hl7.fhir.r4.model.ParameterDefinition;
import org.hl7.fhir.r4.model.Patient.LinkType;
import org.hl7.fhir.r4.model.Period;
import org.hl7.fhir.r4.model.Population;
import org.hl7.fhir.r4.model.PositiveIntType;
import org.hl7.fhir.r4.model.PrimitiveType;
import org.hl7.fhir.r4.model.ProdCharacteristic;
import org.hl7.fhir.r4.model.ProductShelfLife;
import org.hl7.fhir.r4.model.Property;
import org.hl7.fhir.r4.model.Quantity;
import org.hl7.fhir.r4.model.Range;
import org.hl7.fhir.r4.model.Ratio;
import org.hl7.fhir.r4.model.Reference;
import org.hl7.fhir.r4.model.RelatedArtifact;
import org.hl7.fhir.r4.model.Resource;
import org.hl7.fhir.r4.model.SampledData;
import org.hl7.fhir.r4.model.Signature;
import org.hl7.fhir.r4.model.SimpleQuantity;
import org.hl7.fhir.r4.model.StringType;
import org.hl7.fhir.r4.model.SubstanceAmount;
import org.hl7.fhir.r4.model.TimeType;
import org.hl7.fhir.r4.model.Timing;
import org.hl7.fhir.r4.model.TriggerDefinition;
import org.hl7.fhir.r4.model.Type;
import org.hl7.fhir.r4.model.UnsignedIntType;
import org.hl7.fhir.r4.model.UriType;
import org.hl7.fhir.r4.model.UrlType;
import org.hl7.fhir.r4.model.UsageContext;
import org.hl7.fhir.r4.model.UuidType;
import org.hl7.fhir.r4.model.codesystems.CompositionAttestationMode;
import org.hl7.fhir.utilities.xhtml.XhtmlNode;

/**
 * Implementation of {@link IFrameworkTypeLocator} for FHIR Release R4.
 */
@XSlf4j
class FrameworkTypeLocatorR4 extends FrameworkTypeLocatorBase {

    private static final ClassName BACKBONE_ELEMENT = ClassName.get(BackboneElement.class);
    private static final ClassName BASE64_BINARY_TYPE = ClassName.get(Base64BinaryType.class);
    private static final ClassName BOOLEAN_TYPE = ClassName.get(BooleanType.class);
    private static final ClassName CANONICAL_TYPE = ClassName.get(CanonicalType.class);
    private static final ClassName CODE_TYPE = ClassName.get(CodeType.class);
    private static final ClassName DATE = ClassName.get(Date.class);
    private static final ClassName DATE_TIME_TYPE = ClassName.get(DateTimeType.class);
    private static final ClassName DATE_TYPE = ClassName.get(DateType.class);
    private static final ClassName DECIMAL_TYPE = ClassName.get(DecimalType.class);
    private static final ClassName ELEMENT = ClassName.get(Element.class);
    private static final ClassName ID_TYPE = ClassName.get(IdType.class);
    private static final ClassName INSTANT_TYPE = ClassName.get(InstantType.class);
    private static final ClassName INTEGER_TYPE = ClassName.get(IntegerType.class);
    private static final ClassName MARKDOWN_TYPE = ClassName.get(MarkdownType.class);
    private static final ClassName OID_TYPE = ClassName.get(OidType.class);
    private static final ClassName POSITIVE_INT_TYPE = ClassName.get(PositiveIntType.class);
    private static final ClassName RESOURCE = ClassName.get(Resource.class);
    private static final ClassName STRING = ClassName.get(String.class);
    private static final ClassName STRING_TYPE = ClassName.get(StringType.class);
    private static final ClassName TIME_TYPE = ClassName.get(TimeType.class);
    private static final ClassName UNSIGNED_INT_TYPE = ClassName.get(UnsignedIntType.class);
    private static final ClassName URI_TYPE = ClassName.get(UriType.class);
    private static final ClassName URL_TYPE = ClassName.get(UrlType.class);
    private static final ClassName UUID_TYPE = ClassName.get(UuidType.class);
    private static final ClassName ADDRESS = ClassName.get(Address.class);
    private static final ClassName ANNOTATION = ClassName.get(Annotation.class);
    private static final ClassName ATTACHMENT = ClassName.get(Attachment.class);
    private static final ClassName BIG_DECIMAL = ClassName.get(BigDecimal.class);
    private static final ClassName CODEABLE_CONCEPT = ClassName.get(CodeableConcept.class);
    private static final ClassName CODING = ClassName.get(Coding.class);
    private static final ClassName CONTACT_DETAIL = ClassName.get(ContactDetail.class);
    private static final ClassName CONTACT_POINT = ClassName.get(ContactPoint.class);
    private static final ClassName CONTRIBUTOR = ClassName.get(Contributor.class);
    private static final ClassName DATA_REQUIREMENT = ClassName.get(DataRequirement.class);
    private static final ClassName DOSAGE = ClassName.get(Dosage.class);
    private static final ClassName DURATION = ClassName.get(Duration.class);
    private static final ClassName ELEMENT_DEFINITION = ClassName.get(ElementDefinition.class);
    private static final ClassName EXPRESSION = ClassName.get(Expression.class);
    private static final ClassName EXTENSION = ClassName.get(Extension.class);
    private static final ClassName HUMAN_NAME = ClassName.get(HumanName.class);
    private static final ClassName IDENTIFIER = ClassName.get(Identifier.class);
    private static final ClassName MARKETING_STATUS = ClassName.get(MarketingStatus.class);
    private static final ClassName META = ClassName.get(Meta.class);
    private static final ClassName MONEY = ClassName.get(Money.class);
    private static final ClassName NARRATIVE = ClassName.get(Narrative.class);
    private static final ClassName PARAMETER_DEFINITION = ClassName.get(ParameterDefinition.class);
    private static final ClassName PERIOD = ClassName.get(Period.class);
    private static final ClassName POPULATION = ClassName.get(Population.class);
    private static final ClassName PROD_CHARACTERISTIC = ClassName.get(ProdCharacteristic.class);
    private static final ClassName PRODUCT_SHELF_LIFE = ClassName.get(ProductShelfLife.class);
    private static final ClassName QUANTITY = ClassName.get(Quantity.class);
    private static final ClassName RANGE = ClassName.get(Range.class);
    private static final ClassName RATIO = ClassName.get(Ratio.class);
    private static final ClassName REFERENCE = ClassName.get(Reference.class);
    private static final ClassName RELATED_ARTIFACT = ClassName.get(RelatedArtifact.class);
    private static final ClassName SAMPLED_DATA = ClassName.get(SampledData.class);
    private static final ClassName SIGNATURE = ClassName.get(Signature.class);
    private static final ClassName SIMPLE_QUANTITY = ClassName.get(SimpleQuantity.class);
    private static final ClassName SUBSTANCE_AMOUNT = ClassName.get(SubstanceAmount.class);
    private static final ClassName TIMING = ClassName.get(Timing.class);
    private static final ClassName TRIGGER_DEFINITION = ClassName.get(TriggerDefinition.class);
    private static final ClassName TYPE = ClassName.get(Type.class);
    private static final ClassName USAGE_CONTEXT = ClassName.get(UsageContext.class);
    private static final ClassName XHTML_NODE = ClassName.get(XhtmlNode.class);

    /**
     * Default constructor
     *
     * @param contextProvider the {@link IContextProvider} to use
     */
    public FrameworkTypeLocatorR4(IContextProvider contextProvider) {
        super(contextProvider);
    }

    @Override
    protected boolean isFrameworkClass(ClassName className) {
        logger.entry(className);
        if (!className.packageName().startsWith("org.hl7.fhir.r4.model")) {
            return logger.exit(false);
        }
        try {
            Class.forName(className.reflectionName());
        } catch (final ClassNotFoundException e) {
            return logger.exit(false);
        }
        return logger.exit(true);
    }

    @Override
    @SuppressWarnings({
        "java:S1541", // nothing to do at the moment, see comment below
        "java:S3776" // nothing to do at the moment, see comment below
    })
    protected Optional<TypeName> getWiredImplementation(String resourceURI) {
        logger.entry(resourceURI);

        // This is a manually curated list, built for a specific use case.
        // TODO #14 find a better way to create this mapping
        // TODO #15 also provide this for R4B and R5

        final TypeName className =
                switch (resourceURI) {

                    // HL4 FHIR primitive types
                    // see https://hl7.org/fhir/R4/datatypes.html#primitive
                    // see also
                    // https://hl7.org/fhir/R4/elementdefinition-definitions.html#ElementDefinition.type.code:
                    // "References are URLs that are relative to
                    // http://hl7.org/fhir/StructureDefinition
                    // e.g. "string" is a reference
                    // tohttp://hl7.org/fhir/StructureDefinition/string."
                    case "http://hl7.org/fhir/StructureDefinition/boolean" -> BOOLEAN_TYPE;
                    case "http://hl7.org/fhir/StructureDefinition/integer" -> INTEGER_TYPE;
                    case "http://hl7.org/fhir/StructureDefinition/string" -> STRING_TYPE;
                    case "http://hl7.org/fhir/StructureDefinition/decimal" -> DECIMAL_TYPE;
                    case "http://hl7.org/fhir/StructureDefinition/uri" -> URI_TYPE;
                    case "http://hl7.org/fhir/StructureDefinition/url" -> URL_TYPE;
                    case "http://hl7.org/fhir/StructureDefinition/canonical" -> CANONICAL_TYPE;
                    case "http://hl7.org/fhir/StructureDefinition/base64Binary" -> BASE64_BINARY_TYPE;
                    case "http://hl7.org/fhir/StructureDefinition/instant" -> INSTANT_TYPE;
                    case "http://hl7.org/fhir/StructureDefinition/date" -> DATE_TYPE;
                    case "http://hl7.org/fhir/StructureDefinition/dateTime" -> DATE_TIME_TYPE;
                    case "http://hl7.org/fhir/StructureDefinition/time" -> TIME_TYPE;
                    case "http://hl7.org/fhir/StructureDefinition/code" -> CODE_TYPE;
                    case "http://hl7.org/fhir/StructureDefinition/oid" -> OID_TYPE;
                    case "http://hl7.org/fhir/StructureDefinition/id" -> ID_TYPE;
                    case "http://hl7.org/fhir/StructureDefinition/markdown" -> MARKDOWN_TYPE;
                    case "http://hl7.org/fhir/StructureDefinition/unsignedInt" -> UNSIGNED_INT_TYPE;
                    case "http://hl7.org/fhir/StructureDefinition/positiveInt" -> POSITIVE_INT_TYPE;
                    case "http://hl7.org/fhir/StructureDefinition/uuid" -> UUID_TYPE;

                    // FHIRPath types
                    // see https://build.fhir.org/fhirpath.html#types
                    case "http://hl7.org/fhirpath/System.Boolean" -> BOOLEAN_TYPE;
                    case "http://hl7.org/fhirpath/System.String" -> STRING_TYPE;
                    case "http://hl7.org/fhirpath/System.Integer" -> INTEGER_TYPE;
                    case "http://hl7.org/fhirpath/System.Decimal" -> DECIMAL_TYPE;
                    case "http://hl7.org/fhirpath/System.DateTime" -> DATE_TIME_TYPE;
                    case "http://hl7.org/fhirpath/System.Time" -> TIME_TYPE;

                    // ===== StructureDefinitions =====
                    case "http://hl7.org/fhir/StructureDefinition/BackboneElement" -> BACKBONE_ELEMENT;
                    case "http://hl7.org/fhir/StructureDefinition/Element" -> ELEMENT;
                    case "http://hl7.org/fhir/StructureDefinition/Resource" -> RESOURCE;

                    // ===== ValueSets =====
                    case "http://hl7.org/fhir/ValueSet/administrative-gender" ->
                        ParameterizedTypeName.get(Enumeration.class, AdministrativeGender.class);
                    case "http://hl7.org/fhir/ValueSet/bundle-type" ->
                        ParameterizedTypeName.get(Enumeration.class, BundleType.class);
                    case "http://hl7.org/fhir/ValueSet/composition-attestation-mode" ->
                        ParameterizedTypeName.get(Enumeration.class, CompositionAttestationMode.class);
                    case "http://hl7.org/fhir/ValueSet/composition-status" ->
                        ParameterizedTypeName.get(Enumeration.class, CompositionStatus.class);
                    case "http://hl7.org/fhir/ValueSet/contact-point-system" ->
                        ParameterizedTypeName.get(Enumeration.class, ContactPointSystem.class);
                    case "http://hl7.org/fhir/ValueSet/contact-point-use" ->
                        ParameterizedTypeName.get(Enumeration.class, ContactPointUse.class);
                    // "http://hl7.org/fhir/ValueSet/coverage-class" -> just a CodeableConcept
                    // "http://hl7.org/fhir/ValueSet/coverage-copay-type" -> just a CodeableConcept
                    // "http://hl7.org/fhir/ValueSet/doc-typecodes" -> just a CodeableConcept
                    case "http://hl7.org/fhir/ValueSet/document-relationship-type" ->
                        ParameterizedTypeName.get(Enumeration.class, DocumentRelationshipType.class);
                    case "http://hl7.org/fhir/ValueSet/fm-status" ->
                        ParameterizedTypeName.get(Enumeration.class, CoverageStatus.class);
                    case "http://hl7.org/fhir/ValueSet/http-verb" ->
                        ParameterizedTypeName.get(Enumeration.class, HTTPVerb.class);
                    // "http://hl7.org/fhir/ValueSet/identifier-type" -> just a CodeableConcept
                    case "http://hl7.org/fhir/ValueSet/identifier-use" ->
                        ParameterizedTypeName.get(Enumeration.class, IdentifierUse.class);
                    // "http://hl7.org/fhir/ValueSet/iso3166-1-2" -> apparently not mapped
                    // "http://hl7.org/fhir/ValueSet/languages" -> just a CodeableConcept
                    case "http://hl7.org/fhir/ValueSet/link-type" ->
                        ParameterizedTypeName.get(Enumeration.class, LinkType.class);
                    // "http://hl7.org/fhir/ValueSet/list-empty-reason" -> just a CodeableConcept
                    case "http://hl7.org/fhir/ValueSet/list-mode" ->
                        ParameterizedTypeName.get(Enumeration.class, SectionMode.class);
                    // "http://hl7.org/fhir/ValueSet/list-order" -> just a CodeableConcept
                    // "http://hl7.org/fhir/ValueSet/marital-status" -> just a CodeableConcept
                    // "http://hl7.org/fhir/ValueSet/mimetypes" -> just a CodeType
                    // "http://hl7.org/fhir/ValueSet/name-part-qualifier" -> apparently not mapped
                    case "http://hl7.org/fhir/ValueSet/name-use" ->
                        ParameterizedTypeName.get(Enumeration.class, NameUse.class);
                    // "http://hl7.org/fhir/ValueSet/patient-contactrelationship" -> just a
                    // CodeableConcept
                    case "http://hl7.org/fhir/ValueSet/resource-types" -> URI_TYPE;
                    case "http://hl7.org/fhir/ValueSet/search-entry-mode" ->
                        ParameterizedTypeName.get(Enumeration.class, SearchEntryMode.class);
                    // "http://hl7.org/fhir/ValueSet/security-labels" -> just a Coding
                    // "http://hl7.org/fhir/ValueSet/signature-type" -> just a Coding
                    // "http://hl7.org/fhir/ValueSet/subscriber-relationship" -> just a
                    // CodeableConcept
                    case "http://terminology.hl7.org/ValueSet/v3-ConfidentialityClassification" ->
                        ParameterizedTypeName.get(Enumeration.class, DocumentConfidentiality.class);

                    // =====
                    default -> null;
                };
        final Optional<TypeName> result = className == null ? Optional.empty() : Optional.of(className);
        return logger.exit(result);
    }

    @Override
    public TypeName getPrimitiveType(PrimitiveDatatype type) {
        logger.entry(type);
        TypeName result;
        switch (type) {
            case BASE64BINARY -> result = ClassName.get(Base64BinaryType.class);
            case BOOLEAN -> result = ClassName.get(BooleanType.class);
            case CANONICAL -> result = ClassName.get(CanonicalType.class);
            case CODE -> result = ClassName.get(CodeType.class);
            case DATE -> result = ClassName.get(DateType.class);
            case DATETIME -> result = ClassName.get(DateTimeType.class);
            case DECIMAL -> result = ClassName.get(DecimalType.class);
            case ID -> result = ClassName.get(IdType.class);
            case INSTANT -> result = ClassName.get(InstantType.class);
            case INTEGER -> result = ClassName.get(IntegerType.class);
            case MARKDOWN -> result = ClassName.get(MarkdownType.class);
            case OID -> result = ClassName.get(OidType.class);
            case POSITIVEINT -> result = ClassName.get(PositiveIntType.class);
            case STRING -> result = ClassName.get(StringType.class);
            case TIME -> result = ClassName.get(TimeType.class);
            case UNSIGNEDINT -> result = ClassName.get(UnsignedIntType.class);
            case URI -> result = ClassName.get(UriType.class);
            case URL -> result = ClassName.get(UrlType.class);
            case UUID -> result = ClassName.get(UuidType.class);
            default -> throw new IllegalArgumentException(String.format("Type %s is currently not supported.", type));
        }
        return logger.exit(result);
    }

    @Override
    public TypeName getConfigurationType() {
        return ClassName.get(Configuration.class);
    }

    @Override
    public TypeName getBackboneElementType() {
        return BACKBONE_ELEMENT;
    }

    @Override
    public TypeName getExtensionType() {
        return ClassName.get(Extension.class);
    }

    @Override
    public TypeName getBaseType() {
        return ClassName.get(Base.class);
    }

    @Override
    public TypeName getGenericType() {
        return ClassName.get(Type.class);
    }

    @Override
    public TypeName getCodingType() {
        return ClassName.get(Coding.class);
    }

    @Override
    public TypeName getPropertyType() {
        return ClassName.get(Property.class);
    }

    @Override
    public boolean isDerivedFromPrimitiveType(TypeName typeName) throws IllegalArgumentException {
        logger.entry(typeName);
        final ClassName classNameToCheck =
                switch (typeName) {
                    case final ClassName className -> className;
                    case final ParameterizedTypeName parameterizedTypeName -> parameterizedTypeName.rawType();
                    default ->
                        throw logger.throwing(new IllegalArgumentException(String.format(
                                "Invalid parameter type %s", typeName.getClass().getCanonicalName())));
                };

        try {
            final Class<?> actualClass = Class.forName(classNameToCheck.reflectionName());
            return logger.exit(PrimitiveType.class.isAssignableFrom(actualClass));
        } catch (final ClassNotFoundException e) {
            throw logger.throwing(new IllegalArgumentException(
                    String.format(
                            "Unable to load class %s to check for PrimitiveType inheritance - is this a HAPI framework class?",
                            classNameToCheck.reflectionName()),
                    e));
        }
    }

    @Override
    @SuppressWarnings({
        "java:S1541", // this method isn't complex, it's just long
        "java:S3776" // this method isn't complex, it's just long
    })
    public Optional<TypeName> determinePrimitiveType(TypeName frameworkType) {
        logger.entry(frameworkType);
        if (frameworkType == BASE64_BINARY_TYPE) {
            return logger.exit(Optional.of(ArrayTypeName.of(TypeName.BYTE)));
        } else if (frameworkType == BOOLEAN_TYPE) {
            return logger.exit(Optional.of(TypeName.BOOLEAN));
        } else if (frameworkType == CANONICAL_TYPE) {
            return logger.exit(Optional.of(STRING));
        } else if (frameworkType == CODE_TYPE) {
            return logger.exit(Optional.of(STRING));
        } else if (frameworkType == DATE_TIME_TYPE) {
            return logger.exit(Optional.of(DATE));
        } else if (frameworkType == DATE_TYPE) {
            return logger.exit(Optional.of(DATE));
        } else if (frameworkType == DECIMAL_TYPE) {
            return logger.exit(Optional.of(BIG_DECIMAL));
        } else if (frameworkType == ID_TYPE) {
            return logger.exit(Optional.of(STRING));
        } else if (frameworkType == INSTANT_TYPE) {
            return logger.exit(Optional.of(DATE));
        } else if (frameworkType == INTEGER_TYPE) {
            return logger.exit(Optional.of(TypeName.INT));
        } else if (frameworkType == MARKDOWN_TYPE) {
            return logger.exit(Optional.of(STRING));
        } else if (frameworkType == OID_TYPE) {
            return logger.exit(Optional.of(STRING));
        } else if (frameworkType == POSITIVE_INT_TYPE) {
            return logger.exit(Optional.of(TypeName.INT));
        } else if (frameworkType == STRING_TYPE) {
            return logger.exit(Optional.of(STRING));
        } else if (frameworkType == TIME_TYPE) {
            return logger.exit(Optional.of(STRING));
        } else if (frameworkType == UNSIGNED_INT_TYPE) {
            return logger.exit(Optional.of(TypeName.INT));
        } else if (frameworkType == URI_TYPE) {
            return logger.exit(Optional.of(STRING));
        } else if (frameworkType == UUID_TYPE) {
            return logger.exit(Optional.of(STRING));
        } else {
            return logger.exit(Optional.empty());
        }
    }

    @Override
    @SuppressWarnings({
        "java:S1541", // this method isn't complex, it's just long
        "java:S3776" // this method isn't complex, it's just long
    })
    public Optional<String> getBaseCastingMethod(TypeName targetType) {
        logger.entry(targetType);
        if (targetType == ADDRESS) {
            return logger.exit(Optional.of("castToAddress"));
        } else if (targetType == ANNOTATION) {
            return logger.exit(Optional.of("castToAnnotation"));
        } else if (targetType == ATTACHMENT) {
            return logger.exit(Optional.of("castToAttachment"));
        } else if (targetType == BASE64_BINARY_TYPE) {
            return logger.exit(Optional.of("castToBase64Binary"));
        } else if (targetType == BOOLEAN_TYPE) {
            return logger.exit(Optional.of("castToBoolean"));
        } else if (targetType == CANONICAL_TYPE) {
            return logger.exit(Optional.of("castToCanonical"));
        } else if (targetType == CODEABLE_CONCEPT) {
            return logger.exit(Optional.of("castToCodeableConcept"));
        } else if (targetType == CODE_TYPE) {
            return logger.exit(Optional.of("castToCode"));
        } else if (targetType == CODING) {
            return logger.exit(Optional.of("castToCoding"));
        } else if (targetType == CONTACT_DETAIL) {
            return logger.exit(Optional.of("castToContactDetail"));
        } else if (targetType == CONTACT_POINT) {
            return logger.exit(Optional.of("castToContactPoint"));
        } else if (targetType == CONTRIBUTOR) {
            return logger.exit(Optional.of("castToContributor"));
        } else if (targetType == DATA_REQUIREMENT) {
            return logger.exit(Optional.of("castToDataRequirement"));
        } else if (targetType == DATE_TIME_TYPE) {
            return logger.exit(Optional.of("castToDateTime"));
        } else if (targetType == DATE_TYPE) {
            return logger.exit(Optional.of("castToDate"));
        } else if (targetType == DECIMAL_TYPE) {
            return logger.exit(Optional.of("castToDecimal"));
        } else if (targetType == DOSAGE) {
            return logger.exit(Optional.of("castToDosage"));
        } else if (targetType == DURATION) {
            return logger.exit(Optional.of("castToDuration"));
        } else if (targetType == ELEMENT_DEFINITION) {
            return logger.exit(Optional.of("castToElementDefinition"));
        } else if (targetType == EXPRESSION) {
            return logger.exit(Optional.of("castToExpression"));
        } else if (targetType == EXTENSION) {
            return logger.exit(Optional.of("castToExtension"));
        } else if (targetType == HUMAN_NAME) {
            return logger.exit(Optional.of("castToHumanName"));
        } else if (targetType == IDENTIFIER) {
            return logger.exit(Optional.of("castToIdentifier"));
        } else if (targetType == ID_TYPE) {
            return logger.exit(Optional.of("castToId"));
        } else if (targetType == INSTANT_TYPE) {
            return logger.exit(Optional.of("castToInstant"));
        } else if (targetType == INTEGER_TYPE) {
            return logger.exit(Optional.of("castToInteger"));
        } else if (targetType == MARKDOWN_TYPE) {
            return logger.exit(Optional.of("castToMarkdown"));
        } else if (targetType == MARKETING_STATUS) {
            return logger.exit(Optional.of("castToMarketingStatus"));
        } else if (targetType == META) {
            return logger.exit(Optional.of("castToMeta"));
        } else if (targetType == MONEY) {
            return logger.exit(Optional.of("castToMoney"));
        } else if (targetType == NARRATIVE) {
            return logger.exit(Optional.of("castToNarrative"));
        } else if (targetType == OID_TYPE) {
            return logger.exit(Optional.of("castToOid"));
        } else if (targetType == PARAMETER_DEFINITION) {
            return logger.exit(Optional.of("castToParameterDefinition"));
        } else if (targetType == PERIOD) {
            return logger.exit(Optional.of("castToPeriod"));
        } else if (targetType == POPULATION) {
            return logger.exit(Optional.of("castToPopulation"));
        } else if (targetType == POSITIVE_INT_TYPE) {
            return logger.exit(Optional.of("castToPositiveInt"));
        } else if (targetType == PROD_CHARACTERISTIC) {
            return logger.exit(Optional.of("castToProdCharacteristic"));
        } else if (targetType == PRODUCT_SHELF_LIFE) {
            return logger.exit(Optional.of("castToProductShelfLife"));
        } else if (targetType == QUANTITY) {
            return logger.exit(Optional.of("castToQuantity"));
        } else if (targetType == RANGE) {
            return logger.exit(Optional.of("castToRange"));
        } else if (targetType == RATIO) {
            return logger.exit(Optional.of("castToRatio"));
        } else if (targetType == REFERENCE) {
            return logger.exit(Optional.of("castToReference"));
        } else if (targetType == RELATED_ARTIFACT) {
            return logger.exit(Optional.of("castToRelatedArtifact"));
        } else if (targetType == RESOURCE) {
            return logger.exit(Optional.of("castToResource"));
        } else if (targetType == SAMPLED_DATA) {
            return logger.exit(Optional.of("castToSampledData"));
        } else if (targetType == SIGNATURE) {
            return logger.exit(Optional.of("castToSignature"));
        } else if (targetType == SIMPLE_QUANTITY) {
            return logger.exit(Optional.of("castToSimpleQuantity"));
        } else if (targetType == STRING) {
            return logger.exit(Optional.of("castToXhtmlString"));
        } else if (targetType == STRING_TYPE) {
            return logger.exit(Optional.of("castToString"));
        } else if (targetType == SUBSTANCE_AMOUNT) {
            return logger.exit(Optional.of("castToSubstanceAmount"));
        } else if (targetType == TIME_TYPE) {
            return logger.exit(Optional.of("castToTime"));
        } else if (targetType == TIMING) {
            return logger.exit(Optional.of("castToTiming"));
        } else if (targetType == TRIGGER_DEFINITION) {
            return logger.exit(Optional.of("castToTriggerDefinition"));
        } else if (targetType == TYPE) {
            return logger.exit(Optional.of("castToType"));
        } else if (targetType == UNSIGNED_INT_TYPE) {
            return logger.exit(Optional.of("castToUnsignedInt"));
        } else if (targetType == URI_TYPE) {
            return logger.exit(Optional.of("castToUri"));
        } else if (targetType == URL_TYPE) {
            return logger.exit(Optional.of("castToUrl"));
        } else if (targetType == USAGE_CONTEXT) {
            return logger.exit(Optional.of("castToUsageContext"));
        } else if (targetType == XHTML_NODE) {
            return logger.exit(Optional.of("castToXhtml"));
        } else {
            return logger.exit(Optional.empty());
        }
    }

    @Override
    protected boolean determineWiredConstructors(TypeName frameworkType, ArrayList<Constructor> result) {
        logger.entry(frameworkType);
        var wiredTypeFound = false;

        if (frameworkType instanceof final ParameterizedTypeName frameworkParameterizedTypeName) {
            // special case for implementations of org.hl7.fhir.r4.model.Enumeration<T extends Enum<?>>
            if (frameworkParameterizedTypeName.rawType().equals(ClassName.get(Enumeration.class))) {
                final var wrappedEnumerationType =
                        frameworkParameterizedTypeName.typeArguments().getFirst();
                final var factoryType =
                        ParameterizedTypeName.get(ClassName.get(EnumFactory.class), wrappedEnumerationType);

                // @Deprecated
                // public Enumeration()
                result.add(new Constructor(ImmutableList.of(), true));

                // public Enumeration(EnumFactory<T> theEnumFactory)
                result.add(new Constructor(
                        ImmutableList.of(new ConstructorParameter("theEnumFactory", factoryType)), false));

                // public Enumeration(EnumFactory<T> theEnumFactory, String theValue)
                result.add(new Constructor(
                        ImmutableList.of(
                                new ConstructorParameter("theEnumFactory", factoryType),
                                new ConstructorParameter("theValue", ClassName.get(String.class))),
                        false));

                // public Enumeration(EnumFactory<T> theEnumFactory, T theValue)
                result.add(new Constructor(
                        ImmutableList.of(
                                new ConstructorParameter("theEnumFactory", factoryType),
                                new ConstructorParameter("theValue", wrappedEnumerationType)),
                        false));

                // public Enumeration(EnumFactory<T> theEnumFactory, T theValue, Element source)
                result.add(new Constructor(
                        ImmutableList.of(
                                new ConstructorParameter("theEnumFactory", factoryType),
                                new ConstructorParameter("theValue", wrappedEnumerationType),
                                new ConstructorParameter("source", ClassName.get(Element.class))),
                        false));

                wiredTypeFound = true;
            }
        }

        return logger.exit(wiredTypeFound);
    }
}
