/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.implementations;

import ca.uhn.fhir.context.FhirContext;
import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableList;
import com.palantir.javapoet.ClassName;
import com.palantir.javapoet.ParameterizedTypeName;
import com.palantir.javapoet.TypeName;
import de.fhlintstone.accessors.implementations.ITypeSpecification.TypeReference;
import de.fhlintstone.process.IContextProvider;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Optional;
import java.util.regex.Pattern;
import lombok.extern.slf4j.XSlf4j;

/**
 * Base class for all {@link IFrameworkTypeLocator} implementations.
 */
@XSlf4j
abstract class FrameworkTypeLocatorBase implements IFrameworkTypeLocator {

    /**
     * The common prefix for HL7 StructureDefinition URLs
     */
    public static final String HL7_STRUCTURE_DEFINITION_PREFIX = "http://hl7.org/fhir/StructureDefinition/";

    /**
     * The common prefix for HL7 ValueSet URLs
     */
    public static final String HL7_VALUE_SET_PREFIX = "http://hl7.org/fhir/ValueSet/";

    private final IContextProvider contextProvider;

    /**
     * Default constructor
     *
     * @param contextProvider the {@link IContextProvider} to use
     */
    FrameworkTypeLocatorBase(IContextProvider contextProvider) {
        super();
        this.contextProvider = contextProvider;
    }

    @Override
    public Optional<TypeName> determineType(URI resourceURI) {
        logger.entry(resourceURI);
        final var uriString = resourceURI.toString();
        Optional<TypeName> result = getWiredImplementation(uriString);
        if (result.isEmpty() && uriString.startsWith(HL7_STRUCTURE_DEFINITION_PREFIX)) {
            final var name = uriString.substring(HL7_STRUCTURE_DEFINITION_PREFIX.length());

            final var elementDefinition = this.contextProvider.getElementDefinition(name);
            if (elementDefinition.isPresent()) {
                result = Optional.of(ClassName.get(elementDefinition.get().getImplementingClass()));
            }

            if (result.isEmpty()) {
                final var resourceDefinition = this.contextProvider.getResourceDefinition(name);
                if (resourceDefinition.isPresent()) {
                    result = Optional.of(ClassName.get(resourceDefinition.get().getImplementingClass()));
                }
            }
        }

        return logger.exit(result);
    }

    /**
     * Attempts to determine the class using a manual wiring. This is necessary
     * because the HAPI {@link FhirContext} will not return abstract or base types
     * like BackbonElement, Element or Resource.
     *
     * @param resourceURI the URI of the resource to lookup
     * @return the framework type, if mapped
     */
    protected abstract Optional<TypeName> getWiredImplementation(String resourceURI);

    @Override
    public boolean isFrameworkType(TypeName type) {
        logger.entry(type);
        switch (type) {
            case final ClassName className -> {
                return logger.exit(isFrameworkClass(className));
            }
            case final ParameterizedTypeName parameterizedType -> {
                return logger.exit(isFrameworkClass(parameterizedType.rawType())
                        && parameterizedType.typeArguments().stream().allMatch(this::isFrameworkType));
            }
            default ->
                throw logger.throwing(new IllegalArgumentException(
                        "Unsupported parameter type " + type.getClass().getCanonicalName()));
        }
    }

    /**
     * Determines whether the class is a framework class.
     *
     * @param className the class to check
     * @return <code>true</code> if the class is a framework class
     * @see #isFrameworkType(TypeName)
     */
    protected abstract boolean isFrameworkClass(ClassName className);

    @Override
    public URI makeAbsoluteStructureDefinitionReference(String reference) throws URISyntaxException {
        logger.entry(reference);
        final var canonicalWithoutVersion = removeVersion(reference);
        var typeCodeURI = new URI(canonicalWithoutVersion);
        if (!typeCodeURI.isAbsolute()) {
            typeCodeURI = new URI(HL7_STRUCTURE_DEFINITION_PREFIX + canonicalWithoutVersion);
        }
        return logger.exit(typeCodeURI);
    }

    @Override
    public URI makeAbsoluteValueSetReference(String reference) throws URISyntaxException {
        logger.entry(reference);
        final var canonicalWithoutVersion = removeVersion(reference);
        var typeCodeURI = new URI(canonicalWithoutVersion);
        if (!typeCodeURI.isAbsolute()) {
            typeCodeURI = new URI(HL7_VALUE_SET_PREFIX + canonicalWithoutVersion);
        }
        return logger.exit(typeCodeURI);
    }

    private String removeVersion(String reference) {
        logger.entry(reference);
        // TODO #192 process thg version information instead of dropping it
        final var pattern = Pattern.compile("^(.*)\\|[0-9\\.]+$");
        final var matcher = pattern.matcher(reference);
        if (matcher.find()) {
            logger.warn("Dropping version specifier from canonical URL {}", reference);
            return logger.exit(matcher.group(1));
        }
        return logger.exit(reference);
    }

    @Override
    public ImmutableCollection<IMappedType> determineFrameworkTypes(ITypeSpecification typeSpecification) {
        return determineFrameworkTypes(ImmutableList.of(typeSpecification));
    }

    @Override
    public ImmutableCollection<IMappedType> determineFrameworkTypes(Collection<ITypeSpecification> typeSpecifications) {
        logger.entry(typeSpecifications);
        final var mappedTypes = new ArrayList<IMappedType>();
        for (final var typeSpecification : typeSpecifications) {
            final TypeReference typeCode = typeSpecification.getTypeCode();

            var typeFound = false;
            // determine if there is a framework type for one of the profiles specified, if any
            for (final var profile : typeSpecification.getProfiles()) {
                final var frameworkType = determineType(profile.canonical());
                if (frameworkType.isPresent()) {
                    mappedTypes.add(MappedType.builder()
                            .withTypeCode(typeCode.code())
                            .withTypeCodeURI(typeCode.canonical())
                            .withProfile(Optional.of(profile.code()))
                            .withProfileURI(Optional.of(profile.canonical()))
                            .withType(frameworkType.get())
                            .build());
                    typeFound = true;
                }
            }
            if (!typeFound) {
                // no framework type found for the profiles (or no profiles specified), use the type code for lookup
                // next
                final var frameworkType = determineType(typeCode.canonical());
                if (frameworkType.isPresent()) {
                    mappedTypes.add(MappedType.builder()
                            .withTypeCode(typeCode.code())
                            .withTypeCodeURI(typeCode.canonical())
                            .withType(frameworkType.get())
                            .build());
                } else {
                    logger.warn("Type {} cannot be mapped to a framework type", typeSpecification);
                }
            }
        }
        return logger.exit(ImmutableList.copyOf(mappedTypes));
    }

    @Override
    public ImmutableList<Constructor> determineConstructors(TypeName frameworkType) throws IllegalArgumentException {
        logger.entry(frameworkType);
        final var result = new ArrayList<Constructor>();
        // try to find a (release-specific) hard-wired implementation
        if (!determineWiredConstructors(frameworkType, result)) {
            determineConstructorsUsingReflection(frameworkType, result);
        }
        return logger.exit(ImmutableList.copyOf(result));
    }

    /**
     * Tries to determine the constructors of a framework type. This method is intended to provide hard-wired,
     * release-specific implementations in the derived classes.
     *
     * @param frameworkType the type to examine
     * @param result        the target list to add the constructors to
     * @return <code>true</code> if a wired implementation was available, <code>false</code> if a generic,
     *         reflection-based result should be used instead
     */
    protected abstract boolean determineWiredConstructors(TypeName frameworkType, ArrayList<Constructor> result);

    /**
     * Tries to determine the constructors of a framework type using a a generic,
     *         reflection-based approach.
     *
     * @param frameworkType the type to examine
     * @param result        the target list to add the constructors to
     */
    private void determineConstructorsUsingReflection(TypeName frameworkType, ArrayList<Constructor> result) {
        logger.entry(frameworkType);
        final var className = getRootClassName(frameworkType);
        logger.debug("Using reflection to resolve class {}", className);
        try {
            final var classDescriptor = Class.forName(className);
            for (final var constructorDescriptor : classDescriptor.getConstructors()) {
                final var parameters = Arrays.stream(constructorDescriptor.getParameters())
                        .map(p -> new ConstructorParameter(p.getName(), TypeName.get(p.getType())))
                        .toList();
                final var deprecated = constructorDescriptor.getAnnotation(Deprecated.class) != null;
                final var constructor = new Constructor(ImmutableList.copyOf(parameters), deprecated);
                result.add(constructor);
            }
        } catch (ClassNotFoundException | SecurityException e) {
            throw logger.throwing(
                    new IllegalArgumentException(String.format("Unable to access class %s", className), e));
        }
        logger.exit();
    }

    /**
     * Determines the "root" class name (for ClassNames, the class itself; for ParameterizedClassNames, the name of the raw class);
     * @param typeName the type to examine
     * @return the "main" class name
     */
    protected String getRootClassName(TypeName typeName) {
        String className;
        switch (typeName) {
            case final ClassName cn -> {
                className = cn.reflectionName();
            }
            case final ParameterizedTypeName ptn -> {
                className = ptn.rawType().reflectionName();
            }
            default -> {
                throw logger.throwing(new IllegalArgumentException(
                        String.format("Unable to process type specification %s", typeName)));
            }
        }
        return className;
    }
}
