/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.dependencies;

import ca.uhn.fhir.context.FhirVersionEnum;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import de.fhlintstone.accessors.UnsupportedFHIRVersionException;
import de.fhlintstone.accessors.implementations.IFrameworkTypeLocator;
import de.fhlintstone.fhir.dependencies.DependencyException;
import de.fhlintstone.fhir.dependencies.IDependencyCollector;
import de.fhlintstone.process.IContextProvider;
import javax.inject.Inject;
import javax.inject.Named;
import lombok.AccessLevel;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.extern.slf4j.XSlf4j;
import org.hl7.fhir.instance.model.api.IBaseResource;

/**
 * Release-independent implementation of {@link IDependencyResourceVisitor} that
 * can be injected into clients. It defers to the actual release-dependent
 * implementation that will be create at the latest possible point in time and
 * will be cached.
 */
@Named
@EqualsAndHashCode
@XSlf4j
public final class VisitorSelector implements IDependencyResourceVisitor {

    @Getter(AccessLevel.PRIVATE)
    private final IContextProvider contextProvider;

    @Getter(AccessLevel.PRIVATE)
    private final IFrameworkTypeLocator frameworkTypeLocator;

    /**
     * Constructor for dependency injection.
     *
     * @param contextProvider      the {@link IContextProvider} to use
     * @param frameworkTypeLocator the {@link IFrameworkTypeLocator} to use
     */
    @Inject
    public VisitorSelector(IContextProvider contextProvider, IFrameworkTypeLocator frameworkTypeLocator) {
        super();
        this.contextProvider = contextProvider;
        this.frameworkTypeLocator = frameworkTypeLocator;
    }

    @SuppressWarnings("java:S4738") // Java supplier does not support memoization
    private final Supplier<IDependencyResourceVisitor> visitorSupplier = Suppliers.memoize(() -> {
        logger.entry();
        IDependencyResourceVisitor result;
        final FhirVersionEnum fhirVersion =
                getContextProvider().getFhirVersion().orElseThrow();
        switch (fhirVersion) {
            case R4:
                result = new DependencyResourceVisitorR4(getFrameworkTypeLocator());
                break;
            case R4B:
                result = new DependencyResourceVisitorR4B(getFrameworkTypeLocator());
                break;
            case R5:
                result = new DependencyResourceVisitorR5(getFrameworkTypeLocator());
                break;
            default:
                throw logger.throwing(new UnsupportedFHIRVersionException(fhirVersion));
        }
        return logger.exit(result);
    });

    @Override
    public void visit(IBaseResource resource, IDependencyCollector collector) throws DependencyException {
        logger.entry(resource);
        this.visitorSupplier.get().visit(resource, collector);
        logger.exit();
    }
}
