/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors.dependencies;

import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.google.common.collect.ImmutableSet;
import de.fhlintstone.accessors.implementations.IFrameworkTypeLocator;
import de.fhlintstone.fhir.dependencies.DependencyException;
import de.fhlintstone.fhir.dependencies.IDependency.Origin;
import de.fhlintstone.fhir.dependencies.IDependencyCollector;
import java.net.URISyntaxException;
import java.util.List;
import lombok.extern.slf4j.XSlf4j;
import org.hl7.fhir.instance.model.api.IBaseResource;
import org.hl7.fhir.r4.model.CodeSystem;
import org.hl7.fhir.r4.model.ElementDefinition;
import org.hl7.fhir.r4.model.Enumerations.BindingStrength;
import org.hl7.fhir.r4.model.StructureDefinition;
import org.hl7.fhir.r4.model.ValueSet;
import org.hl7.fhir.r4.model.ValueSet.ConceptSetComponent;

/**
 * Implementation of {@link IDependencyResourceVisitor} for FHIR Release R4.
 */
@XSlf4j
class DependencyResourceVisitorR4 extends DependencyResourceVisitor implements IDependencyResourceVisitor {

    /**
     * Standard constructor.
     * @param frameworkTypeLocator the {@link IFrameworkTypeLocator} to use
     */
    public DependencyResourceVisitorR4(IFrameworkTypeLocator frameworkTypeLocator) {
        super(frameworkTypeLocator);
    }

    @Override
    public void visit(IBaseResource resource, IDependencyCollector collector) throws DependencyException {
        logger.entry(resource);
        switch (resource) {
            case final StructureDefinition structureDefinition ->
                visitStructureDefinition(structureDefinition, collector);
            case final ValueSet valueSet -> visitValueSet(valueSet, collector);
            case final CodeSystem codeSystem -> visitCodeSystem(codeSystem, collector);
            default ->
                logger.warn(
                        "Unknown resource implementation {} is not checked for dependencies",
                        resource.getClass().getCanonicalName());
        }
        logger.exit();
    }

    private void visitStructureDefinition(StructureDefinition structureDefinition, IDependencyCollector collector)
            throws DependencyException {
        logger.entry(structureDefinition);
        // not relevant: Abstract
        // not relevant: AbstractElement
        if (structureDefinition.hasBaseDefinition()) {
            logger.debug("Found dependency: base type {}", structureDefinition.getBaseDefinition());
            collector.collect(structureDefinition.getBaseDefinition(), Origin.BASE_DEFINITION);
        }
        // not relevant: BaseDefinitionElement
        // not relevant: Contact
        // not relevant: Context
        if (structureDefinition.hasContextInvariant()) {
            logger.warn(
                    "Unchecked structure property ContextInvariant = {}, dependency graph may be incomplete",
                    structureDefinition.getContextInvariant());
        }
        // not relevant: Copyright
        // not relevant: CopyrightElement
        // not relevant: Date
        // not relevant: DateElement
        // not relevant: Derivation
        // not relevant: DerivationElement
        // not relevant: Description
        // not relevant: DescriptionElement
        if (structureDefinition.hasDifferential()) {
            final var location = String.format("StructureDefinition[%s].differential", structureDefinition.getId());
            for (final ElementDefinition differentialElement :
                    structureDefinition.getDifferential().getElement()) {
                visitElementDefinition(differentialElement, collector, location);
            }
        }
        // not relevant: Experimental
        // not relevant: ExperimentalElement
        // not relevant: FhirVersion
        // not relevant: FhirVersionElement
        // not relevant: Identifier
        // not relevant: Jurisdiction
        // not relevant: Keyword
        // not relevant: Kind
        // not relevant: KindElement
        // not relevant: Mapping
        // not relevant: Name
        // not relevant: NameElement
        // not relevant: Publisher
        // not relevant: PublisherElement
        // not relevant: Purpose
        // not relevant: PurposeElement
        if (structureDefinition.hasSnapshot()) {
            final var location = String.format("StructureDefinition[%s].snapshot", structureDefinition.getId());
            for (final ElementDefinition snapshotElement :
                    structureDefinition.getSnapshot().getElement()) {
                visitElementDefinition(snapshotElement, collector, location);
            }
        }
        // not relevant: Status
        // not relevant: StatusElement
        // not relevant: Title
        // not relevant: TitleElement
        // not relevant: Type
        // not relevant: TypeElement
        // not relevant: Url
        // not relevant: UrlElement
        // not relevant: UseContext
        // not relevant: Version
        // not relevant: VersionElement
        logger.exit();
    }

    private void visitElementDefinition(
            ElementDefinition elementDefinition, IDependencyCollector collector, String parentLocation)
            throws DependencyException {
        logger.entry(elementDefinition.getName());
        final var location = String.format("%s/ElementDefinition[%s]", parentLocation, elementDefinition.getId());
        // not relevant: Alias
        // not relevant: Base
        if (elementDefinition.hasBinding()) {
            visitElementDefinitionBase(elementDefinition, collector, location);
        }
        if (elementDefinition.hasCode()) {
            logger.warn(
                    "Unchecked {}.Code = {}, dependency graph may be incomplete",
                    location,
                    elementDefinition.getCode());
        }
        // not relevant: Comment
        // not relevant: CommentElement
        // not relevant: Condition
        // not relevant: Constraint
        // not relevant: ContentReference
        // not relevant: ContentReferenceElement
        // not relevant: DefaultValue
        // not relevant: Definition
        // not relevant: DefinitionElement
        // not relevant: Example
        // not relevant: Fixed
        // not relevant: FixedOrPattern
        // not relevant: IsModifier
        // not relevant: IsModifierElement
        // not relevant: IsModifierReason
        // not relevant: IsModifierReasonElement
        // not relevant: IsSummary
        // not relevant: IsSummaryElement
        // not relevant: Label
        // not relevant: LabelElement
        // not relevant: Mapping
        // not relevant: Max
        // not relevant: MaxElement
        // not relevant: MaxLength
        // not relevant: MaxLengthElement
        // not relevant: MaxValue
        // not relevant: MaxValueDateTimeType
        // not relevant: MaxValueDateType
        // not relevant: MaxValueDecimalType
        // not relevant: MaxValueInstantType
        // not relevant: MaxValueIntegerType
        // not relevant: MaxValuePositiveIntType
        // not relevant: MaxValueQuantity
        // not relevant: MaxValueTimeType
        // not relevant: MaxValueUnsignedIntType
        // not relevant: MeaningWhenMissing
        // not relevant: MeaningWhenMissingElement
        // not relevant: Min
        // not relevant: MinElement
        // not relevant: MinValue
        // not relevant: MinValueDateTimeType
        // not relevant: MinValueDateType
        // not relevant: MinValueDecimalType
        // not relevant: MinValueInstantType
        // not relevant: MinValueIntegerType
        // not relevant: MinValuePositiveIntType
        // not relevant: MinValueQuantity
        // not relevant: MinValueTimeType
        // not relevant: MinValueUnsignedIntType
        // not relevant: MustSupport
        // not relevant: MustSupportElement
        // not relevant: OrderMeaning
        // not relevant: OrderMeaningElement
        // not relevant: Path
        // not relevant: PathElement
        // not relevant: Pattern
        // not relevant: Representation
        // not relevant: Requirements
        // not relevant: RequirementsElement
        // not relevant: Short
        // not relevant: ShortElement
        // not relevant: SliceIsConstraining
        // not relevant: SliceIsConstrainingElement
        // not relevant: SliceName
        // not relevant: SliceNameElement
        // not relevant: Slicing
        if (elementDefinition.hasType()) {
            visitElementDefinitionType(elementDefinition, collector, location);
        }
        logger.exit();
    }

    public void visitElementDefinitionBase(
            ElementDefinition elementDefinition, IDependencyCollector collector, final String location)
            throws DependencyException {
        logger.entry(elementDefinition);
        final var binding = elementDefinition.getBinding();
        final var strength = binding.hasStrength() ? binding.getStrength() : BindingStrength.NULL;
        switch (strength) {
            case BindingStrength.NULL -> logger.warn("{}.binding.strength is not set, ignoring binding", location);

            case BindingStrength.EXAMPLE -> logger.debug("Ignoring example binding of {}.binding", location);

            default -> {
                if (binding.hasValueSet()) {
                    final var valueSet = binding.getValueSet();
                    logger.debug("Found dependency: {}.binding ValueSet {}", location, valueSet);
                    collector.collect(valueSet, Origin.BINDING_VALUE_SET);
                } else {
                    logger.warn(
                            "Unchecked {}.binding with strength {} without ValueSet, dependency graph may be incomplete",
                            location,
                            strength);
                }
            }
        }
        logger.exit();
    }

    public void visitElementDefinitionType(
            ElementDefinition elementDefinition, IDependencyCollector collector, final String location)
            throws DependencyException {
        logger.entry(elementDefinition);
        // CAUTION: Read and understand
        // https://hl7.org/fhir/elementdefinition-definitions.html#ElementDefinition.type
        // before attempting to change this! Type references in FHIR can be tricky.
        for (final var typeRefComponent : elementDefinition.getType()) {
            // code[1..1]: References are URLs that are relative to
            // http://hl7.org/fhir/StructureDefinition e.g. "string"
            // is a reference to http://hl7.org/fhir/StructureDefinition/string. Absolute
            // URLs are only allowed in
            // logical models.
            // If the element is a reference to another resource, this element contains
            // "Reference", and the
            // targetProfile element defines what resources can be referenced. The
            // targetProfile may be a
            // reference to the general definition of a resource (e.g.
            // http://hl7.org/fhir/StructureDefinition/Patient).
            if (!typeRefComponent.hasCode()) {
                logger.warn("ElementDefinition {} does not have a type.code set.", elementDefinition.getId());
            } else {
                visitElementDefinitionType(
                        typeRefComponent.getCode(), location, "type.code", Origin.TYPE_CODE, collector);
            }

            // profile[0..*]: Identifies a profile structure or implementation Guide that
            // applies to the datatype
            // this element refers to. If any profiles are specified, then the content must
            // conform to at least one
            // of them. The URL can be a local reference - to a contained
            // StructureDefinition, or a reference to
            // another StructureDefinition or Implementation Guide by a canonical URL. When
            // an implementation guide
            // is specified, the type SHALL conform to at least one profile defined in the
            // implementation guide.
            for (final var profile : typeRefComponent.getProfile()) {
                visitElementDefinitionType(
                        profile.getValue(), location, "type.profile", Origin.TYPE_PROFILE, collector);
            }

            // targetProfile[0..*]: Used when the type is "Reference" or "canonical", and
            // identifies a profile
            // structure or implementation Guide that applies to the target of the reference
            // this element refers
            // to. If any profiles are specified, then the content must conform to at least
            // one of them. The URL
            // can be a local reference - to a contained StructureDefinition, or a reference
            // to another
            // StructureDefinition or Implementation Guide by a canonical URL. When an
            // implementation guide is
            // specified, the target resource SHALL conform to at least one profile defined
            // in the implementation
            // guide.
            for (final var targetProfile : typeRefComponent.getTargetProfile()) {
                visitElementDefinitionType(
                        targetProfile.getValue(),
                        location,
                        "type.targetProfile",
                        Origin.TYPE_TARGET_PROFILE,
                        collector);
            }
        }
        logger.exit();
    }

    private void visitElementDefinitionType(
            final String type,
            final String location,
            final String subAttribute,
            final Origin origin,
            IDependencyCollector collector)
            throws DependencyException {
        logger.entry(type, location, subAttribute, origin);
        if (this.ignoredTypesSupplier.get().contains(type)) {
            logger.debug("Ignored {}.{} because of type {}", location, subAttribute, type);
        } else {
            try {
                final var typeURI = makeAbsoluteTypeReference(type);
                logger.debug("Found dependency: {}.{} = {}", location, subAttribute, typeURI);
                collector.collect(typeURI, origin);
            } catch (final URISyntaxException e) {
                throw logger.throwing(new DependencyException(
                        String.format("Invalid URI %s in %s.%s", type, location, subAttribute), e));
            }
        }
        logger.exit();
    }

    private void visitValueSet(ValueSet valueSet, IDependencyCollector collector) throws DependencyException {
        logger.entry(valueSet);
        if (valueSet.hasCompose()) {
            visitValueSetCompose(valueSet, collector);
        }
        // not relevant: Contact
        // not relevant: Copyright
        // not relevant: CopyrightElement
        // not relevant: Date
        // not relevant: DateElement
        // not relevant: Description
        // not relevant: DescriptionElement
        // not relevant: Expansion
        // not relevant: Experimental
        // not relevant: ExperimentalElement
        // not relevant: Identifier
        // not relevant: Immutable
        // not relevant: ImmutableElement
        // not relevant: Jurisdiction
        // not relevant: Name
        // not relevant: NameElement
        // not relevant: Publisher
        // not relevant: PublisherElement
        // not relevant: Purpose
        // not relevant: PurposeElement
        // not relevant: Status
        // not relevant: StatusElement
        // not relevant: Title
        // not relevant: TitleElement
        // not relevant: Url
        // not relevant: UrlElement
        // not relevant: UseContext
        // not relevant: Version
        // not relevant: VersionElement
        logger.exit();
    }

    public void visitValueSetCompose(ValueSet valueSet, IDependencyCollector collector) throws DependencyException {
        logger.entry(valueSet);
        final var valueSetCompose = valueSet.getCompose();
        if (valueSetCompose.hasInclude()) {
            visitValueSetComponents(collector, valueSetCompose.getInclude(), "include");
        }
        if (valueSetCompose.hasExclude()) {
            visitValueSetComponents(collector, valueSetCompose.getExclude(), "exclude");
        }
        logger.exit();
    }

    private void visitValueSetComponents(
            IDependencyCollector collector, final List<ConceptSetComponent> components, String mode)
            throws DependencyException {
        for (final var component : components) {
            if (component.hasSystem()) {
                logger.debug("Found dependency: ValueSet.compose.{}.system {}", mode, component.getSystem());
                collector.collect(component.getSystem(), Origin.COMPOSE_CODE_SYSTEM);
            }
            if (component.hasValueSet()) {
                for (final var vs : component.getValueSet()) {
                    logger.debug("Found dependency: ValueSet.compose.{}.valueSet{}", mode, vs.getValue());
                    collector.collect(vs.getValue(), Origin.COMPOSE_VALUE_SET);
                }
            }
        }
    }

    private void visitCodeSystem(CodeSystem codeSystem, IDependencyCollector collector) throws DependencyException {
        logger.entry(codeSystem);
        // not relevant: CaseSensitive
        // not relevant: CaseSensitiveElement
        // not relevant: Compositional
        // not relevant: CompositionalElement
        // not relevant: Concept
        // not relevant: ConceptFirstRep
        // not relevant: Contact
        // not relevant: ContactFirstRep
        // not relevant: Content
        // not relevant: ContentElement
        // not relevant: Copyright
        // not relevant: CopyrightElement
        // not relevant: Count
        // not relevant: CountElement
        // not relevant: Date
        // not relevant: DateElement
        // not relevant: Description
        // not relevant: DescriptionElement
        // not relevant: Experimental
        // not relevant: ExperimentalElement
        // not relevant: Filter
        // not relevant: FilterFirstRep
        // not relevant: HierarchyMeaning
        // not relevant: HierarchyMeaningElement
        // not relevant: Identifier
        // not relevant: IdentifierFirstRep
        // not relevant: Jurisdiction
        // not relevant: JurisdictionFirstRep
        // not relevant: Name
        // not relevant: NameElement
        // not relevant: Publisher
        // not relevant: PublisherElement
        // not relevant: Purpose
        // not relevant: PurposeElement
        // not relevant: ResourceType
        // not relevant: Status
        // not relevant: StatusElement
        if (codeSystem.hasSupplements()) {
            final var supplements = codeSystem.getSupplements();
            logger.debug("Found dependency: CodeSystem.valueSet = {}", supplements);
            collector.collect(supplements, Origin.CODE_SYSTEM_SUPPLEMENTS);
        }
        // not relevant: Title
        // not relevant: TitleElement
        // not relevant: Url
        // not relevant: UrlElement
        // not relevant: UseContext
        // not relevant: UseContextFirstRep
        // inverse reference, not used: ValueSet
        // not relevant: Version
        // not relevant: VersionElement
        // not relevant: VersionNeeded
        // not relevant: VersionNeededElement
        logger.exit();
    }

    @SuppressWarnings("java:S4738") // Java supplier does not support memoization
    private final Supplier<ImmutableSet<String>> ignoredTypesSupplier = Suppliers.memoize(() -> ImmutableSet.of(
            // HL4 FHIR primitive types
            // see https://hl7.org/fhir/R4/datatypes.html#primitive
            "boolean",
            "integer",
            "string",
            "decimal",
            "uri",
            "url",
            "canonical",
            "base64Binary",
            "instant",
            "date",
            "dateTime",
            "time",
            "code",
            "oid",
            "id",
            "markdown",
            "unsignedInt",
            "positiveInt",
            "uuid",
            // FHIRPath types
            // see https://build.fhir.org/fhirpath.html#types
            "http://hl7.org/fhirpath/System.Boolean",
            "http://hl7.org/fhirpath/System.String",
            "http://hl7.org/fhirpath/System.Integer",
            "http://hl7.org/fhirpath/System.Long",
            "http://hl7.org/fhirpath/System.Decimal",
            "http://hl7.org/fhirpath/System.DateTime",
            "http://hl7.org/fhirpath/System.Time"));
}
