/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors;

import de.fhlintstone.accessors.model.ICodeSystemAccessor;
import de.fhlintstone.accessors.model.IStructureDefinitionAccessor;
import de.fhlintstone.accessors.model.IValueSetAccessor;
import java.net.URI;
import org.hl7.fhir.instance.model.api.IBaseResource;

/**
 * Factory to provide accessor implementations.
 */
public interface IAccessorProvider {

    /**
     * Provides an {@link IStructureDefinitionAccessor} for a StructureDefinition
     * resource.
     *
     * @param resource the {@link IBaseResource} that contains the
     *                 StructureDefinition
     * @return the {@link IStructureDefinitionAccessor} wrapping the
     *         StructureDefinition
     */
    IStructureDefinitionAccessor provideStructureDefinitionAccessor(IBaseResource resource);

    /**
     * Provides an {@link IStructureDefinitionAccessor} for a StructureDefinition
     * resource, identified by its URI.
     *
     * @param uri  the URI identifying the StructureDefinition
     * @return the {@link IStructureDefinitionAccessor} wrapping the
     *         StructureDefinition
     * @throws UnresolvableURIException if the provided URI could not be resolved
     */
    IStructureDefinitionAccessor provideStructureDefinitionAccessor(URI uri) throws UnresolvableURIException;

    /**
     * Provides an {@link IValueSetAccessor} for a ValueSet resource.
     *
     * @param resource the {@link IBaseResource} that contains the ValueSet
     * @return the {@link IValueSetAccessor} wrapping the ValueSet
     */
    IValueSetAccessor provideValueSetAccessor(IBaseResource resource);

    /**
     * Provides an {@link IValueSetAccessor} for a ValueSet resource, identified by its URI.
     *
     * @param uri  the URI identifying the ValueSet
     * @return the {@link IValueSetAccessor} wrapping the ValueSet
     * @throws UnresolvableURIException if the provided URI could not be resolved
     */
    IValueSetAccessor provideValueSetAccessor(URI uri) throws UnresolvableURIException;

    /**
     * Provides an {@link ICodeSystemAccessor} for a CodeSystem resource.
     *
     * @param resource the {@link IBaseResource} that contains the CodeSystem
     * @return the {@link ICodeSystemAccessor} wrapping the CodeSystem
     */
    ICodeSystemAccessor provideCodeSystemAccessor(IBaseResource resource);

    /**
     * Provides an {@link ICodeSystemAccessor} for a CodeSystem resource, identified by its URI.
     *
     * @param uri  the URI identifying the CodeSystem
     * @return the {@link ICodeSystemAccessor} wrapping the CodeSystem
     * @throws UnresolvableURIException if the provided URI could not be resolved
     */
    ICodeSystemAccessor provideCodeSystemAccessor(URI uri) throws UnresolvableURIException;
}
