/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors;

import ca.uhn.fhir.context.FhirVersionEnum;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import de.fhlintstone.accessors.model.ICodeSystemAccessor;
import de.fhlintstone.accessors.model.IStructureDefinitionAccessor;
import de.fhlintstone.accessors.model.IValueSetAccessor;
import de.fhlintstone.packages.IPackageRegistry;
import de.fhlintstone.process.IContextProvider;
import java.net.URI;
import javax.inject.Inject;
import javax.inject.Named;
import lombok.AccessLevel;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.extern.slf4j.XSlf4j;
import org.hl7.fhir.instance.model.api.IBaseResource;

/**
 * Default implementation of {@link IAccessorProvider}.
 */
@Named
@EqualsAndHashCode
@XSlf4j
public final class AccessorProvider implements IAccessorProvider {

    @Getter(AccessLevel.PRIVATE)
    private final IContextProvider contextProvider;

    @Getter(AccessLevel.PRIVATE)
    private final IPackageRegistry packageRegistry;

    /**
     * Constructor for dependency injection.
     *
     * @param contextProvider the {@link IContextProvider} to use
     * @param packageRegistry the {@link IPackageRegistry} to use
     */
    @Inject
    public AccessorProvider(IContextProvider contextProvider, IPackageRegistry packageRegistry) {
        super();
        this.contextProvider = contextProvider;
        this.packageRegistry = packageRegistry;
    }

    @Override
    public IStructureDefinitionAccessor provideStructureDefinitionAccessor(IBaseResource resource) {
        logger.entry(resource);
        IStructureDefinitionAccessor result;
        final FhirVersionEnum fhirVersion =
                this.contextProvider.getFhirVersion().orElseThrow();
        switch (fhirVersion) {
            case R4:
                if (resource instanceof org.hl7.fhir.r4.model.StructureDefinition) {
                    result = this.cacheSupplier.get().getAccessor(resource, IStructureDefinitionAccessor.class);
                } else {
                    throw new IllegalArgumentException(String.format(
                            "Expected type org.hl7.fhir.r4.model.StructureDefinition, actual type %s",
                            resource.getClass().getCanonicalName()));
                }
                break;
            case R4B:
                if (resource instanceof org.hl7.fhir.r4b.model.StructureDefinition) {
                    result = this.cacheSupplier.get().getAccessor(resource, IStructureDefinitionAccessor.class);
                } else {
                    throw new IllegalArgumentException(String.format(
                            "Expected type org.hl7.fhir.r4b.model.StructureDefinition, actual type %s",
                            resource.getClass().getCanonicalName()));
                }
                break;
            case R5:
                if (resource instanceof org.hl7.fhir.r5.model.StructureDefinition) {
                    result = this.cacheSupplier.get().getAccessor(resource, IStructureDefinitionAccessor.class);
                } else {
                    throw new IllegalArgumentException(String.format(
                            "Expected type org.hl7.fhir.r5.model.StructureDefinition, actual type %s",
                            resource.getClass().getCanonicalName()));
                }
                break;
            default:
                throw logger.throwing(new UnsupportedFHIRVersionException(fhirVersion));
        }
        return logger.exit(result);
    }

    @Override
    public IStructureDefinitionAccessor provideStructureDefinitionAccessor(URI uri) throws UnresolvableURIException {
        logger.entry(uri);
        final var resource = resolveResource(uri);
        final var accessor = provideStructureDefinitionAccessor(resource);
        return logger.exit(accessor);
    }

    @Override
    public IValueSetAccessor provideValueSetAccessor(IBaseResource resource) {
        logger.entry(resource);
        IValueSetAccessor result;
        final FhirVersionEnum fhirVersion =
                this.contextProvider.getFhirVersion().orElseThrow();
        switch (fhirVersion) {
            case R4:
                if (resource instanceof org.hl7.fhir.r4.model.ValueSet) {
                    result = this.cacheSupplier.get().getAccessor(resource, IValueSetAccessor.class);
                } else {
                    throw new IllegalArgumentException(String.format(
                            "Expected type org.hl7.fhir.r4.model.ValueSet, actual type %s",
                            resource.getClass().getCanonicalName()));
                }
                break;
            case R4B:
                if (resource instanceof org.hl7.fhir.r4b.model.ValueSet) {
                    result = this.cacheSupplier.get().getAccessor(resource, IValueSetAccessor.class);
                } else {
                    throw new IllegalArgumentException(String.format(
                            "Expected type org.hl7.fhir.r4b.model.ValueSet, actual type %s",
                            resource.getClass().getCanonicalName()));
                }
                break;
            case R5:
                if (resource instanceof org.hl7.fhir.r5.model.ValueSet) {
                    result = this.cacheSupplier.get().getAccessor(resource, IValueSetAccessor.class);
                } else {
                    throw new IllegalArgumentException(String.format(
                            "Expected type org.hl7.fhir.r5.model.ValueSet, actual type %s",
                            resource.getClass().getCanonicalName()));
                }
                break;
            default:
                throw logger.throwing(new UnsupportedFHIRVersionException(fhirVersion));
        }
        return logger.exit(result);
    }

    @Override
    public IValueSetAccessor provideValueSetAccessor(URI uri) throws UnresolvableURIException {
        logger.entry(uri);
        final var resource = resolveResource(uri);
        final var accessor = provideValueSetAccessor(resource);
        return logger.exit(accessor);
    }

    @Override
    public ICodeSystemAccessor provideCodeSystemAccessor(IBaseResource resource) {
        logger.entry(resource);
        ICodeSystemAccessor result;
        final FhirVersionEnum fhirVersion =
                this.contextProvider.getFhirVersion().orElseThrow();
        switch (fhirVersion) {
            case R4:
                if (resource instanceof org.hl7.fhir.r4.model.CodeSystem) {
                    result = this.cacheSupplier.get().getAccessor(resource, ICodeSystemAccessor.class);
                } else {
                    throw logger.throwing(new IllegalArgumentException(String.format(
                            "Expected org.hl7.fhir.r4.model.CodeSystem, received %s",
                            resource.getClass().getCanonicalName())));
                }
                break;
            case R4B:
                if (resource instanceof org.hl7.fhir.r4b.model.CodeSystem) {
                    result = this.cacheSupplier.get().getAccessor(resource, ICodeSystemAccessor.class);
                } else {
                    throw logger.throwing(new IllegalArgumentException(String.format(
                            "Expected org.hl7.fhir.r4b.model.CodeSystem, received %s",
                            resource.getClass().getCanonicalName())));
                }
                break;
            case R5:
                if (resource instanceof org.hl7.fhir.r5.model.CodeSystem) {
                    result = this.cacheSupplier.get().getAccessor(resource, ICodeSystemAccessor.class);
                } else {
                    throw logger.throwing(new IllegalArgumentException(String.format(
                            "Expected org.hl7.fhir.r5.model.CodeSystem, received %s",
                            resource.getClass().getCanonicalName())));
                }
                break;
            default:
                throw logger.throwing(new UnsupportedFHIRVersionException(fhirVersion));
        }
        return logger.exit(result);
    }

    @Override
    public ICodeSystemAccessor provideCodeSystemAccessor(URI uri) throws UnresolvableURIException {
        logger.entry(uri);
        final var resource = resolveResource(uri);
        final var accessor = provideCodeSystemAccessor(resource);
        return logger.exit(accessor);
    }

    private IBaseResource resolveResource(URI uri) {
        logger.entry(uri);
        final var resource = this.packageRegistry.getUniqueResource(uri);
        if (!resource.isPresent()) {
            throw logger.throwing(
                    new UnresolvableURIException(String.format("Resource URI %s cannot be resolved", uri)));
        }
        return logger.exit(resource.get());
    }

    @SuppressWarnings("java:S4738") // Java supplier does not support memoization
    private final Supplier<IAccessorCache> cacheSupplier = Suppliers.memoize(() -> {
        logger.entry();
        IAccessorCache result;
        final FhirVersionEnum fhirVersion =
                getContextProvider().getFhirVersion().orElseThrow();
        switch (fhirVersion) {
            case R4:
                result = new AccessorCacheR4();
                break;
            case R4B:
                result = new AccessorCacheR4B();
                break;
            case R5:
                result = new AccessorCacheR5();
                break;
            default:
                throw logger.throwing(new UnsupportedFHIRVersionException(fhirVersion));
        }
        return logger.exit(result);
    });
}
