/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.accessors;

import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.UncheckedExecutionException;
import java.util.concurrent.ExecutionException;
import lombok.extern.slf4j.XSlf4j;

/**
 * The base implementation of the {@link IAccessorCache}. This implementation
 * contains the parts that are independent of the FHIR release.
 */
@XSlf4j
public abstract class AccessorCache implements IAccessorCache {

    /** Create a new instance of AccessorCache */
    protected AccessorCache() {}

    /**
     * Creates the {@link CacheLoader} that provides accessors for each model object type supported.
     *
     * @return the {@link CacheLoader} to use
     */
    protected abstract CacheLoader<Object, IAccessor> provideCacheLoader();

    @SuppressWarnings("java:S4738") // Java supplier does not support memoization
    private final Supplier<LoadingCache<Object, IAccessor>> cacheSupplier = Suppliers.memoize(() -> {
        logger.entry();
        logger.debug("Initializing new accessor cache");
        final LoadingCache<Object, IAccessor> result =
                CacheBuilder.newBuilder().weakKeys().weakValues().build(provideCacheLoader());
        return logger.exit(result);
    });

    @Override
    public IAccessor getAccessor(Object modelObject) {
        logger.entry(modelObject);
        IAccessor result;
        try {
            result = this.cacheSupplier.get().get(modelObject);
        } catch (final ExecutionException | UncheckedExecutionException e) {
            throw logger.throwing(UnsupportedTypeException.forObjectType(modelObject));
        }
        return logger.exit(result);
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T extends IAccessor> T getAccessor(Object modelObject, Class<T> accessorType) {
        logger.entry(modelObject);
        IAccessor result;
        try {
            result = this.cacheSupplier.get().get(modelObject);
        } catch (final ExecutionException | UncheckedExecutionException e) {
            throw logger.throwing(UnsupportedTypeException.forObjectType(modelObject));
        }
        if (accessorType.isAssignableFrom(result.getClass())) {
            return logger.exit((T) result);
        } else {
            throw new InvalidTypeException(modelObject, accessorType);
        }
    }
}
