package com.googlecode.d2j.smali;

import com.googlecode.d2j.node.DexClassNode;
import com.googlecode.d2j.node.DexFileNode;
import com.googlecode.d2j.smali.antlr4.SmaliLexer;
import com.googlecode.d2j.smali.antlr4.SmaliParser;
import com.googlecode.d2j.visitors.DexFileVisitor;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import org.antlr.v4.runtime.CharStream;
import org.antlr.v4.runtime.CharStreams;
import org.antlr.v4.runtime.CodePointCharStream;
import org.antlr.v4.runtime.CommonTokenStream;

public final class Smali {

    private Smali() {
        throw new UnsupportedOperationException();
    }

    public static void smaliFile(Path path, DexFileVisitor dcv) throws IOException {
        try (Reader reader = Files.newBufferedReader(path, StandardCharsets.UTF_8)) {
            CodePointCharStream is = CharStreams.fromReader(reader, path.toString());
            smali0(dcv, is);
        }
    }

    public static void smaliFile(String name, String buff, DexFileVisitor dcv) {
        CodePointCharStream is = CharStreams.fromString(buff, name);
        smali0(dcv, is);
    }

    public static void smaliFile(String name, InputStream in, DexFileVisitor dcv) throws IOException {
        try (InputStreamReader reader = new InputStreamReader(in, StandardCharsets.UTF_8)) {
            CodePointCharStream is = CharStreams.fromReader(reader, name);
            smali0(dcv, is);
        }
    }

    public static DexClassNode smaliFile2Node(String name, InputStream in) throws IOException {
        DexFileNode dfn = new DexFileNode();
        smaliFile(name, in, dfn);
        return !dfn.clzs.isEmpty() ? dfn.clzs.get(0) : null;
    }

    public static DexClassNode smaliFile2Node(String name, String buff) {
        DexFileNode dfn = new DexFileNode();
        smaliFile(name, buff, dfn);
        return !dfn.clzs.isEmpty() ? dfn.clzs.get(0) : null;
    }

    private static void smali0(DexFileVisitor dcv, CharStream is) {
        SmaliLexer lexer = new SmaliLexer(is);
        CommonTokenStream ts = new CommonTokenStream(lexer);
        SmaliParser parser = new SmaliParser(ts);

        for (SmaliParser.SFileContext ctx : parser.sFiles().sFile()) {
            AntlrSmaliUtil.acceptFile(ctx, dcv);
        }
    }

    public static void smaliFile(String fileName, char[] data, DexFileVisitor dcv) {
        // System.err.println("parsing " + f.getAbsoluteFile());
        CodePointCharStream is = CharStreams.fromString(new String(data), fileName);
        smali0(dcv, is);
    }

    public static void smali(Path base, final DexFileVisitor dfv) throws IOException {
        if (Files.isDirectory(base)) {
            Files.walkFileTree(base, new SimpleFileVisitor<Path>() {
                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    Path fn = dir.getFileName();
                    if (fn != null && fn.toString().startsWith(".")) {
                        return FileVisitResult.SKIP_SUBTREE;
                    }
                    return super.preVisitDirectory(dir, attrs);
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    smaliFile(file, dfv);
                    return super.visitFile(file, attrs);
                }
            });
        } else if (Files.isRegularFile(base)) {
            smaliFile(base, dfv);
        }
    }

}
