package physx.physics;

/**
 * flags for configuring properties of the scene
 * <p>
 * <b>See also:</b> PxScene
 */
public enum PxSceneFlagEnum {

    /**
     * Enable Active Actors Notification.
     * <p>
     * This flag enables the Active Actor Notification feature for a scene.  This
     * feature defaults to disabled.  When disabled, the function
     * PxScene::getActiveActors() will always return a NULL list.
     * <p>
     * <b>Note:</b> There may be a performance penalty for enabling the Active Actor Notification, hence this flag should
     * only be enabled if the application intends to use the feature.
     * <p>
     * <b>Default:</b> False
     */
    eENABLE_ACTIVE_ACTORS(geteENABLE_ACTIVE_ACTORS()),
    /**
     * Enables a second broad phase check after integration that makes it possible to prevent objects from tunneling through eachother.
     * <p>
     * PxPairFlag::eDETECT_CCD_CONTACT requires this flag to be specified.
     * <p>
     * <b>Note:</b> For this feature to be effective for bodies that can move at a significant velocity, the user should raise the flag PxRigidBodyFlag::eENABLE_CCD for them.
     * <b>Note:</b> This flag is not mutable, and must be set in PxSceneDesc at scene creation.
     * <p>
     * <b>Default:</b> False
     * <p>
     * <b>See also:</b> PxRigidBodyFlag::eENABLE_CCD, PxPairFlag::eDETECT_CCD_CONTACT, eDISABLE_CCD_RESWEEP
     */
    eENABLE_CCD(geteENABLE_CCD()),
    /**
     * Enables a simplified swept integration strategy, which sacrifices some accuracy for improved performance.
     * <p>
     * This simplified swept integration approach makes certain assumptions about the motion of objects that are not made when using a full swept integration. 
     * These assumptions usually hold but there are cases where they could result in incorrect behavior between a set of fast-moving rigid bodies. A key issue is that
     * fast-moving dynamic objects may tunnel through each-other after a rebound. This will not happen if this mode is disabled. However, this approach will be potentially 
     * faster than a full swept integration because it will perform significantly fewer sweeps in non-trivial scenes involving many fast-moving objects. This approach 
     * should successfully resist objects passing through the static environment.
     * <p>
     * PxPairFlag::eDETECT_CCD_CONTACT requires this flag to be specified.
     * <p>
     * <b>Note:</b> This scene flag requires eENABLE_CCD to be enabled as well. If it is not, this scene flag will do nothing.
     * <b>Note:</b> For this feature to be effective for bodies that can move at a significant velocity, the user should raise the flag PxRigidBodyFlag::eENABLE_CCD for them.
     * <b>Note:</b> This flag is not mutable, and must be set in PxSceneDesc at scene creation.
     * <p>
     * <b>Default:</b> False
     * <p>
     * <b>See also:</b> PxRigidBodyFlag::eENABLE_CCD, PxPairFlag::eDETECT_CCD_CONTACT, eENABLE_CCD
     */
    eDISABLE_CCD_RESWEEP(geteDISABLE_CCD_RESWEEP()),
    /**
     * Enable GJK-based distance collision detection system.
     * <p>
     * <b>Note:</b> This flag is not mutable, and must be set in PxSceneDesc at scene creation.
     * <p>
     * <b>Default:</b> true
     */
    eENABLE_PCM(geteENABLE_PCM()),
    /**
     * Disable contact report buffer resize. Once the contact buffer is full, the rest of the contact reports will 
     * not be buffered and sent.
     * <p>
     * <b>Note:</b> This flag is not mutable, and must be set in PxSceneDesc at scene creation.
     * <p>
     * <b>Default:</b> false
     */
    eDISABLE_CONTACT_REPORT_BUFFER_RESIZE(geteDISABLE_CONTACT_REPORT_BUFFER_RESIZE()),
    /**
     * Disable contact cache.
     * <p>
     * Contact caches are used internally to provide faster contact generation. You can disable all contact caches
     * if memory usage for this feature becomes too high.
     * <p>
     * <b>Note:</b> This flag is not mutable, and must be set in PxSceneDesc at scene creation.
     * <p>
     * <b>Default:</b> false
     */
    eDISABLE_CONTACT_CACHE(geteDISABLE_CONTACT_CACHE()),
    /**
     * Require scene-level locking
     * <p>
     * When set to true this requires that threads accessing the PxScene use the
     * multi-threaded lock methods.
     * <p>
     * <b>Note:</b> This flag is not mutable, and must be set in PxSceneDesc at scene creation.
     * <p>
     * <b>See also:</b> PxScene::lockRead
     * <b>See also:</b> PxScene::unlockRead
     * <b>See also:</b> PxScene::lockWrite
     * <b>See also:</b> PxScene::unlockWrite
     * <p>
     * <b>Default:</b> false
     */
    eREQUIRE_RW_LOCK(geteREQUIRE_RW_LOCK()),
    /**
     * Enables additional stabilization pass in solver
     * <p>
     * When set to true, this enables additional stabilization processing to improve that stability of complex interactions between large numbers of bodies.
     * <p>
     * Note that this flag is not mutable and must be set in PxSceneDesc at scene creation. Also, this is an experimental feature which does result in some loss of momentum.
     */
    eENABLE_STABILIZATION(geteENABLE_STABILIZATION()),
    /**
     * Enables average points in contact manifolds
     * <p>
     * When set to true, this enables additional contacts to be generated per manifold to represent the average point in a manifold. This can stabilize stacking when only a small
     * number of solver iterations is used.
     * <p>
     * Note that this flag is not mutable and must be set in PxSceneDesc at scene creation.
     */
    eENABLE_AVERAGE_POINT(geteENABLE_AVERAGE_POINT()),
    /**
     * Do not report kinematics in list of active actors.
     * <p>
     * Since the target pose for kinematics is set by the user, an application can track the activity state directly and use
     * this flag to avoid that kinematics get added to the list of active actors.
     * <p>
     * <b>Note:</b> This flag has only an effect in combination with eENABLE_ACTIVE_ACTORS.
     * <p>
     * <b>See also:</b> eENABLE_ACTIVE_ACTORS
     * <p>
     * <b>Default:</b> false
     */
    eEXCLUDE_KINEMATICS_FROM_ACTIVE_ACTORS(geteEXCLUDE_KINEMATICS_FROM_ACTIVE_ACTORS()),
    eENABLE_GPU_DYNAMICS(geteENABLE_GPU_DYNAMICS()),
    /**
     * Provides improved determinism at the expense of performance. 
     * <p>
     * By default, PhysX provides limited determinism guarantees. Specifically, PhysX guarantees that the exact scene (same actors created in the same order) and simulated using the same 
     * time-stepping scheme should provide the exact same behaviour.
     * <p>
     * However, if additional actors are added to the simulation, this can affect the behaviour of the existing actors in the simulation, even if the set of new actors do not interact with 
     * the existing actors.
     * <p>
     * This flag provides an additional level of determinism that guarantees that the simulation will not change if additional actors are added to the simulation, provided those actors do not interfere
     * with the existing actors in the scene. Determinism is only guaranteed if the actors are inserted in a consistent order each run in a newly-created scene and simulated using a consistent time-stepping
     * scheme.
     * <p>
     * Note that this flag is not mutable and must be set at scene creation.
     * <p>
     * Note that enabling this flag can have a negative impact on performance.
     * <p>
     * Note that this feature is not currently supported on GPU.
     * <p>
     * <b>Default</b> false
     */
    eENABLE_ENHANCED_DETERMINISM(geteENABLE_ENHANCED_DETERMINISM()),
    /**
     * Controls processing friction in all solver iterations
     * <p>
     * By default, PhysX processes friction only in the final 3 position iterations, and all velocity
     * iterations. This flag enables friction processing in all position and velocity iterations.
     * <p>
     * The default behaviour provides a good trade-off between performance and stability and is aimed
     * primarily at game development.
     * <p>
     * When simulating more complex frictional behaviour, such as grasping of complex geometries with
     * a robotic manipulator, better results can be achieved by enabling friction in all solver iterations.
     * <p>
     * <b>Note:</b> This flag only has effect with the default solver. The TGS solver always performs friction per-iteration.
     */
    eENABLE_FRICTION_EVERY_ITERATION(geteENABLE_FRICTION_EVERY_ITERATION()),
    eENABLE_DIRECT_GPU_API(geteENABLE_DIRECT_GPU_API()),
    eMUTABLE_FLAGS(geteMUTABLE_FLAGS());
    public final int value;
    
    PxSceneFlagEnum(int value) {
        this.value = value;
    }

    private static native int _geteENABLE_ACTIVE_ACTORS();
    private static int geteENABLE_ACTIVE_ACTORS() {
        de.fabmax.physxjni.Loader.load();
        return _geteENABLE_ACTIVE_ACTORS();
    }

    private static native int _geteENABLE_CCD();
    private static int geteENABLE_CCD() {
        de.fabmax.physxjni.Loader.load();
        return _geteENABLE_CCD();
    }

    private static native int _geteDISABLE_CCD_RESWEEP();
    private static int geteDISABLE_CCD_RESWEEP() {
        de.fabmax.physxjni.Loader.load();
        return _geteDISABLE_CCD_RESWEEP();
    }

    private static native int _geteENABLE_PCM();
    private static int geteENABLE_PCM() {
        de.fabmax.physxjni.Loader.load();
        return _geteENABLE_PCM();
    }

    private static native int _geteDISABLE_CONTACT_REPORT_BUFFER_RESIZE();
    private static int geteDISABLE_CONTACT_REPORT_BUFFER_RESIZE() {
        de.fabmax.physxjni.Loader.load();
        return _geteDISABLE_CONTACT_REPORT_BUFFER_RESIZE();
    }

    private static native int _geteDISABLE_CONTACT_CACHE();
    private static int geteDISABLE_CONTACT_CACHE() {
        de.fabmax.physxjni.Loader.load();
        return _geteDISABLE_CONTACT_CACHE();
    }

    private static native int _geteREQUIRE_RW_LOCK();
    private static int geteREQUIRE_RW_LOCK() {
        de.fabmax.physxjni.Loader.load();
        return _geteREQUIRE_RW_LOCK();
    }

    private static native int _geteENABLE_STABILIZATION();
    private static int geteENABLE_STABILIZATION() {
        de.fabmax.physxjni.Loader.load();
        return _geteENABLE_STABILIZATION();
    }

    private static native int _geteENABLE_AVERAGE_POINT();
    private static int geteENABLE_AVERAGE_POINT() {
        de.fabmax.physxjni.Loader.load();
        return _geteENABLE_AVERAGE_POINT();
    }

    private static native int _geteEXCLUDE_KINEMATICS_FROM_ACTIVE_ACTORS();
    private static int geteEXCLUDE_KINEMATICS_FROM_ACTIVE_ACTORS() {
        de.fabmax.physxjni.Loader.load();
        return _geteEXCLUDE_KINEMATICS_FROM_ACTIVE_ACTORS();
    }

    private static native int _geteENABLE_GPU_DYNAMICS();
    private static int geteENABLE_GPU_DYNAMICS() {
        de.fabmax.physxjni.Loader.load();
        return _geteENABLE_GPU_DYNAMICS();
    }

    private static native int _geteENABLE_ENHANCED_DETERMINISM();
    private static int geteENABLE_ENHANCED_DETERMINISM() {
        de.fabmax.physxjni.Loader.load();
        return _geteENABLE_ENHANCED_DETERMINISM();
    }

    private static native int _geteENABLE_FRICTION_EVERY_ITERATION();
    private static int geteENABLE_FRICTION_EVERY_ITERATION() {
        de.fabmax.physxjni.Loader.load();
        return _geteENABLE_FRICTION_EVERY_ITERATION();
    }

    private static native int _geteENABLE_DIRECT_GPU_API();
    private static int geteENABLE_DIRECT_GPU_API() {
        de.fabmax.physxjni.Loader.load();
        return _geteENABLE_DIRECT_GPU_API();
    }

    private static native int _geteMUTABLE_FLAGS();
    private static int geteMUTABLE_FLAGS() {
        de.fabmax.physxjni.Loader.load();
        return _geteMUTABLE_FLAGS();
    }

    public static PxSceneFlagEnum forValue(int value) {
        for (int i = 0; i < values().length; i++) {
            if (values()[i].value == value) {
                return values()[i];
            }
        }
        throw new IllegalArgumentException("Unknown value for enum PxSceneFlagEnum: " + value);
    }

}
