package physx.physics;

import physx.NativeObject;
import physx.PlatformChecks;
import physx.common.PxBounds3;
import physx.common.PxCpuDispatcher;
import physx.common.PxCudaContextManager;
import physx.common.PxGpuDynamicsMemoryConfig;
import physx.common.PxTolerancesScale;
import physx.common.PxVec3;

/**
 * Descriptor class for scenes. See #PxScene.
 * <p>
 * This struct must be initialized with the same PxTolerancesScale values used to initialize PxPhysics.
 * @see PxScene
 * @see PxPhysics#createScene
 * @see physx.common.PxTolerancesScale
 */
public class PxSceneDesc extends NativeObject {

    protected PxSceneDesc() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxSceneDesc wrapPointer(long address) {
        return address != 0L ? new PxSceneDesc(address) : null;
    }
    
    public static PxSceneDesc arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxSceneDesc(long address) {
        super(address);
    }

    // Placed Constructors

    /**
     * @param address Pre-allocated memory, where the object is created.
     * @param scale   WebIDL type: {@link PxTolerancesScale} [Const, Ref]
     * @return Stack allocated object of PxSceneDesc
     */
    public static PxSceneDesc createAt(long address, PxTolerancesScale scale) {
        __placement_new_PxSceneDesc(address, scale.getAddress());
        PxSceneDesc createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    /**
     * @param <T>       Allocator class, e.g. LWJGL's MemoryStack.
     * @param allocator Object to use for allocation, e.g. an instance of LWJGL's MemoryStack.
     * @param allocate  Method to call on allocator to obtain the target address, e.g. MemoryStack::nmalloc.
     * @param scale     WebIDL type: {@link PxTolerancesScale} [Const, Ref]
     * @return Stack allocated object of PxSceneDesc
     */
    public static <T> PxSceneDesc createAt(T allocator, Allocator<T> allocate, PxTolerancesScale scale) {
        long address = allocate.on(allocator, ALIGNOF, SIZEOF); 
        __placement_new_PxSceneDesc(address, scale.getAddress());
        PxSceneDesc createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    private static native void __placement_new_PxSceneDesc(long address, long scale);

    // Constructors

    /**
     * constructor sets to default.
     * @param scale scale values for the tolerances in the scene, these must be the same values passed into
     * PxCreatePhysics(). The affected tolerances are bounceThresholdVelocity and frictionOffsetThreshold.
     * @see physx.common.PxTolerancesScale
     */
    public PxSceneDesc(PxTolerancesScale scale) {
        address = _PxSceneDesc(scale.getAddress());
    }
    private static native long _PxSceneDesc(long scale);

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * Gravity vector.
     * <p>
     * <b>Range:</b> force vector<br>
     * <b>Default:</b> Zero
     * @see PxScene#setGravity
     * @see PxScene#getGravity
     * <p>
     * When setting gravity, you should probably also set bounce threshold.
     */
    public PxVec3 getGravity() {
        checkNotNull();
        return PxVec3.wrapPointer(_getGravity(address));
    }
    private static native long _getGravity(long address);

    /**
     * Gravity vector.
     * <p>
     * <b>Range:</b> force vector<br>
     * <b>Default:</b> Zero
     * @see PxScene#setGravity
     * @see PxScene#getGravity
     * <p>
     * When setting gravity, you should probably also set bounce threshold.
     */
    public void setGravity(PxVec3 value) {
        checkNotNull();
        _setGravity(address, value.getAddress());
    }
    private static native void _setGravity(long address, long value);

    /**
     * Possible notification callback.
     * <p>
     * <b>Default:</b> NULL
     * @see PxSimulationEventCallback
     * @see PxScene#setSimulationEventCallback
     * @see PxScene#getSimulationEventCallback
     */
    public PxSimulationEventCallback getSimulationEventCallback() {
        checkNotNull();
        return PxSimulationEventCallback.wrapPointer(_getSimulationEventCallback(address));
    }
    private static native long _getSimulationEventCallback(long address);

    /**
     * Possible notification callback.
     * <p>
     * <b>Default:</b> NULL
     * @see PxSimulationEventCallback
     * @see PxScene#setSimulationEventCallback
     * @see PxScene#getSimulationEventCallback
     */
    public void setSimulationEventCallback(PxSimulationEventCallback value) {
        checkNotNull();
        _setSimulationEventCallback(address, value.getAddress());
    }
    private static native void _setSimulationEventCallback(long address, long value);

    /**
     * Shared global filter data which will get passed into the filter shader.
     * <p>
     * <b>Note:</b> The provided data will get copied to internal buffers and this copy will be used for filtering calls.
     * <p>
     * <b>Default:</b> NULL
     * @see PxSimulationFilterShader
     * @see PxScene#setFilterShaderData
     * @see PxScene#getFilterShaderData
     */
    public NativeObject getFilterShaderData() {
        checkNotNull();
        return NativeObject.wrapPointer(_getFilterShaderData(address));
    }
    private static native long _getFilterShaderData(long address);

    /**
     * Shared global filter data which will get passed into the filter shader.
     * <p>
     * <b>Note:</b> The provided data will get copied to internal buffers and this copy will be used for filtering calls.
     * <p>
     * <b>Default:</b> NULL
     * @see PxSimulationFilterShader
     * @see PxScene#setFilterShaderData
     * @see PxScene#getFilterShaderData
     */
    public void setFilterShaderData(NativeObject value) {
        checkNotNull();
        _setFilterShaderData(address, value.getAddress());
    }
    private static native void _setFilterShaderData(long address, long value);

    /**
     * Size (in bytes) of the shared global filter data #filterShaderData.
     * <p>
     * <b>Default:</b> 0
     * @see PxSimulationFilterShader
     * @see PxScene#getFilterShaderDataSize
     */
    public int getFilterShaderDataSize() {
        checkNotNull();
        return _getFilterShaderDataSize(address);
    }
    private static native int _getFilterShaderDataSize(long address);

    /**
     * Size (in bytes) of the shared global filter data #filterShaderData.
     * <p>
     * <b>Default:</b> 0
     * @see PxSimulationFilterShader
     * @see PxScene#getFilterShaderDataSize
     */
    public void setFilterShaderDataSize(int value) {
        checkNotNull();
        _setFilterShaderDataSize(address, value);
    }
    private static native void _setFilterShaderDataSize(long address, int value);

    /**
     * The custom filter shader to use for collision filtering.
     * <p>
     * <b>Note:</b> This parameter is compulsory. If you don't want to define your own filter shader you can
     * use the default shader #PxDefaultSimulationFilterShader which can be found in the PhysX extensions 
     * library.
     * @see PxSimulationFilterShader
     * @see PxScene#getFilterShader
     */
    public PxSimulationFilterShader getFilterShader() {
        checkNotNull();
        return PxSimulationFilterShader.wrapPointer(_getFilterShader(address));
    }
    private static native long _getFilterShader(long address);

    /**
     * The custom filter shader to use for collision filtering.
     * <p>
     * <b>Note:</b> This parameter is compulsory. If you don't want to define your own filter shader you can
     * use the default shader #PxDefaultSimulationFilterShader which can be found in the PhysX extensions 
     * library.
     * @see PxSimulationFilterShader
     * @see PxScene#getFilterShader
     */
    public void setFilterShader(PxSimulationFilterShader value) {
        checkNotNull();
        _setFilterShader(address, value.getAddress());
    }
    private static native void _setFilterShader(long address, long value);

    /**
     * Filtering mode for kinematic-kinematic pairs in the broadphase.
     * <p>
     * <b>Default:</b> PxPairFilteringMode::eDEFAULT
     * @see PxScene#getKinematicKinematicFilteringMode
     */
    public PxPairFilteringModeEnum getKineKineFilteringMode() {
        checkNotNull();
        return PxPairFilteringModeEnum.forValue(_getKineKineFilteringMode(address));
    }
    private static native int _getKineKineFilteringMode(long address);

    /**
     * Filtering mode for kinematic-kinematic pairs in the broadphase.
     * <p>
     * <b>Default:</b> PxPairFilteringMode::eDEFAULT
     * @see PxScene#getKinematicKinematicFilteringMode
     */
    public void setKineKineFilteringMode(PxPairFilteringModeEnum value) {
        checkNotNull();
        _setKineKineFilteringMode(address, value.value);
    }
    private static native void _setKineKineFilteringMode(long address, int value);

    /**
     * Filtering mode for static-kinematic pairs in the broadphase.
     * <p>
     * <b>Default:</b> PxPairFilteringMode::eDEFAULT
     * @see PxScene#getStaticKinematicFilteringMode
     */
    public PxPairFilteringModeEnum getStaticKineFilteringMode() {
        checkNotNull();
        return PxPairFilteringModeEnum.forValue(_getStaticKineFilteringMode(address));
    }
    private static native int _getStaticKineFilteringMode(long address);

    /**
     * Filtering mode for static-kinematic pairs in the broadphase.
     * <p>
     * <b>Default:</b> PxPairFilteringMode::eDEFAULT
     * @see PxScene#getStaticKinematicFilteringMode
     */
    public void setStaticKineFilteringMode(PxPairFilteringModeEnum value) {
        checkNotNull();
        _setStaticKineFilteringMode(address, value.value);
    }
    private static native void _setStaticKineFilteringMode(long address, int value);

    /**
     * Selects the broad-phase algorithm to use.
     * <p>
     * <b>Default:</b> PxBroadPhaseType::ePABP
     * @see PxScene#getBroadPhaseType
     */
    public PxBroadPhaseTypeEnum getBroadPhaseType() {
        checkNotNull();
        return PxBroadPhaseTypeEnum.forValue(_getBroadPhaseType(address));
    }
    private static native int _getBroadPhaseType(long address);

    /**
     * Selects the broad-phase algorithm to use.
     * <p>
     * <b>Default:</b> PxBroadPhaseType::ePABP
     * @see PxScene#getBroadPhaseType
     */
    public void setBroadPhaseType(PxBroadPhaseTypeEnum value) {
        checkNotNull();
        _setBroadPhaseType(address, value.value);
    }
    private static native void _setBroadPhaseType(long address, int value);

    /**
     * Expected scene limits.
     * @see PxSceneLimits
     * @see PxScene#getLimits
     */
    public PxSceneLimits getLimits() {
        checkNotNull();
        return PxSceneLimits.wrapPointer(_getLimits(address));
    }
    private static native long _getLimits(long address);

    /**
     * Expected scene limits.
     * @see PxSceneLimits
     * @see PxScene#getLimits
     */
    public void setLimits(PxSceneLimits value) {
        checkNotNull();
        _setLimits(address, value.getAddress());
    }
    private static native void _setLimits(long address, long value);

    /**
     * Selects the friction algorithm to use for simulation.
     * <p>
     * <b>Note:</b> frictionType cannot be modified after the first call to any of PxScene::simulate, PxScene::solve and PxScene::collide
     * <p>
     * <b>Default:</b> PxFrictionType::ePATCH
     * @see PxScene#getFrictionType
     */
    public PxFrictionTypeEnum getFrictionType() {
        checkNotNull();
        return PxFrictionTypeEnum.forValue(_getFrictionType(address));
    }
    private static native int _getFrictionType(long address);

    /**
     * Selects the friction algorithm to use for simulation.
     * <p>
     * <b>Note:</b> frictionType cannot be modified after the first call to any of PxScene::simulate, PxScene::solve and PxScene::collide
     * <p>
     * <b>Default:</b> PxFrictionType::ePATCH
     * @see PxScene#getFrictionType
     */
    public void setFrictionType(PxFrictionTypeEnum value) {
        checkNotNull();
        _setFrictionType(address, value.value);
    }
    private static native void _setFrictionType(long address, int value);

    /**
     * Selects the solver algorithm to use.
     * <p>
     * <b>Default:</b> PxSolverType::ePGS
     * @see PxScene#getSolverType
     */
    public PxSolverTypeEnum getSolverType() {
        checkNotNull();
        return PxSolverTypeEnum.forValue(_getSolverType(address));
    }
    private static native int _getSolverType(long address);

    /**
     * Selects the solver algorithm to use.
     * <p>
     * <b>Default:</b> PxSolverType::ePGS
     * @see PxScene#getSolverType
     */
    public void setSolverType(PxSolverTypeEnum value) {
        checkNotNull();
        _setSolverType(address, value.value);
    }
    private static native void _setSolverType(long address, int value);

    /**
     * A contact with a relative velocity below this will not bounce. A typical value for simulation.
     * stability is about 0.2 * gravity.
     * <p>
     * <b>Range:</b> (0, PX_MAX_F32)<br>
     * <b>Default:</b> 0.2 * PxTolerancesScale::speed
     * @see PxMaterial
     * @see PxScene#setBounceThresholdVelocity
     * @see PxScene#getBounceThresholdVelocity
     */
    public float getBounceThresholdVelocity() {
        checkNotNull();
        return _getBounceThresholdVelocity(address);
    }
    private static native float _getBounceThresholdVelocity(long address);

    /**
     * A contact with a relative velocity below this will not bounce. A typical value for simulation.
     * stability is about 0.2 * gravity.
     * <p>
     * <b>Range:</b> (0, PX_MAX_F32)<br>
     * <b>Default:</b> 0.2 * PxTolerancesScale::speed
     * @see PxMaterial
     * @see PxScene#setBounceThresholdVelocity
     * @see PxScene#getBounceThresholdVelocity
     */
    public void setBounceThresholdVelocity(float value) {
        checkNotNull();
        _setBounceThresholdVelocity(address, value);
    }
    private static native void _setBounceThresholdVelocity(long address, float value);

    /**
     * A threshold of contact separation distance used to decide if a contact point will experience friction forces.
     * <p>
     * <b>Note:</b> If the separation distance of a contact point is greater than the threshold then the contact point will not experience friction forces. 
     * <p>
     * <b>Note:</b> If the aggregated contact offset of a pair of shapes is large it might be desirable to neglect friction
     * for contact points whose separation distance is sufficiently large that the shape surfaces are clearly separated. 
     * <p>
     * <b>Note:</b> This parameter can be used to tune the separation distance of contact points at which friction starts to have an effect.  
     * <p>
     * <b>Range:</b> [0, PX_MAX_F32)<br>
     * <b>Default:</b> 0.04 * PxTolerancesScale::length
     * @see PxScene#setFrictionOffsetThreshold
     * @see PxScene#getFrictionOffsetThreshold
     */
    public float getFrictionOffsetThreshold() {
        checkNotNull();
        return _getFrictionOffsetThreshold(address);
    }
    private static native float _getFrictionOffsetThreshold(long address);

    /**
     * A threshold of contact separation distance used to decide if a contact point will experience friction forces.
     * <p>
     * <b>Note:</b> If the separation distance of a contact point is greater than the threshold then the contact point will not experience friction forces. 
     * <p>
     * <b>Note:</b> If the aggregated contact offset of a pair of shapes is large it might be desirable to neglect friction
     * for contact points whose separation distance is sufficiently large that the shape surfaces are clearly separated. 
     * <p>
     * <b>Note:</b> This parameter can be used to tune the separation distance of contact points at which friction starts to have an effect.  
     * <p>
     * <b>Range:</b> [0, PX_MAX_F32)<br>
     * <b>Default:</b> 0.04 * PxTolerancesScale::length
     * @see PxScene#setFrictionOffsetThreshold
     * @see PxScene#getFrictionOffsetThreshold
     */
    public void setFrictionOffsetThreshold(float value) {
        checkNotNull();
        _setFrictionOffsetThreshold(address, value);
    }
    private static native void _setFrictionOffsetThreshold(long address, float value);

    /**
     * Friction correlation distance used to decide whether contacts are close enough to be merged into a single friction anchor point or not.
     * <p>
     * <b>Note:</b> If the correlation distance is larger than the distance between contact points generated between a pair of shapes, some of the contacts may not experience frictional forces.
     * <p>
     * <b>Note:</b> This parameter can be used to tune the correlation distance used in the solver. Contact points can be merged into a single friction anchor if the distance between the contacts is smaller than correlation distance.
     * <p>
     * <b>Range:</b> [0, PX_MAX_F32)<br>
     * <b>Default:</b> 0.025f * PxTolerancesScale::length
     * @see PxScene#setFrictionCorrelationDistance
     * @see PxScene#getFrictionCorrelationDistance
     */
    public float getFrictionCorrelationDistance() {
        checkNotNull();
        return _getFrictionCorrelationDistance(address);
    }
    private static native float _getFrictionCorrelationDistance(long address);

    /**
     * Friction correlation distance used to decide whether contacts are close enough to be merged into a single friction anchor point or not.
     * <p>
     * <b>Note:</b> If the correlation distance is larger than the distance between contact points generated between a pair of shapes, some of the contacts may not experience frictional forces.
     * <p>
     * <b>Note:</b> This parameter can be used to tune the correlation distance used in the solver. Contact points can be merged into a single friction anchor if the distance between the contacts is smaller than correlation distance.
     * <p>
     * <b>Range:</b> [0, PX_MAX_F32)<br>
     * <b>Default:</b> 0.025f * PxTolerancesScale::length
     * @see PxScene#setFrictionCorrelationDistance
     * @see PxScene#getFrictionCorrelationDistance
     */
    public void setFrictionCorrelationDistance(float value) {
        checkNotNull();
        _setFrictionCorrelationDistance(address, value);
    }
    private static native void _setFrictionCorrelationDistance(long address, float value);

    /**
     * Flags used to select scene options.
     * <p>
     * <b>Default:</b> PxSceneFlag::eENABLE_PCM
     * @see PxSceneFlags
     * @see PxScene#getFlags
     * @see PxScene#setFlag
     */
    public PxSceneFlags getFlags() {
        checkNotNull();
        return PxSceneFlags.wrapPointer(_getFlags(address));
    }
    private static native long _getFlags(long address);

    /**
     * Flags used to select scene options.
     * <p>
     * <b>Default:</b> PxSceneFlag::eENABLE_PCM
     * @see PxSceneFlags
     * @see PxScene#getFlags
     * @see PxScene#setFlag
     */
    public void setFlags(PxSceneFlags value) {
        checkNotNull();
        _setFlags(address, value.getAddress());
    }
    private static native void _setFlags(long address, long value);

    /**
     * The CPU task dispatcher for the scene.
     */
    public PxCpuDispatcher getCpuDispatcher() {
        checkNotNull();
        return PxCpuDispatcher.wrapPointer(_getCpuDispatcher(address));
    }
    private static native long _getCpuDispatcher(long address);

    /**
     * The CPU task dispatcher for the scene.
     */
    public void setCpuDispatcher(PxCpuDispatcher value) {
        checkNotNull();
        _setCpuDispatcher(address, value.getAddress());
    }
    private static native void _setCpuDispatcher(long address, long value);

    /**
     * The CUDA context manager for the scene.
     * <p>
     * <b>Platform specific:</b> Applies to PC GPU only.
     */
    public PxCudaContextManager getCudaContextManager() {
        checkNotNull();
        PlatformChecks.requirePlatform(3, "physx.physics.PxSceneDesc");
        return PxCudaContextManager.wrapPointer(_getCudaContextManager(address));
    }
    private static native long _getCudaContextManager(long address);

    /**
     * The CUDA context manager for the scene.
     * <p>
     * <b>Platform specific:</b> Applies to PC GPU only.
     */
    public void setCudaContextManager(PxCudaContextManager value) {
        checkNotNull();
        PlatformChecks.requirePlatform(3, "physx.physics.PxSceneDesc");
        _setCudaContextManager(address, value.getAddress());
    }
    private static native void _setCudaContextManager(long address, long value);

    /**
     * Will be copied to PxScene::userData.
     * <p>
     * <b>Default:</b> NULL
     */
    public NativeObject getUserData() {
        checkNotNull();
        return NativeObject.wrapPointer(_getUserData(address));
    }
    private static native long _getUserData(long address);

    /**
     * Will be copied to PxScene::userData.
     * <p>
     * <b>Default:</b> NULL
     */
    public void setUserData(NativeObject value) {
        checkNotNull();
        _setUserData(address, value.getAddress());
    }
    private static native void _setUserData(long address, long value);

    /**
     * Defines the number of actors required to spawn a separate rigid body solver island task chain.
     * <p>
     * This parameter defines the minimum number of actors required to spawn a separate rigid body solver task chain. Setting a low value 
     * will potentially cause more task chains to be generated. This may result in the overhead of spawning tasks can become a limiting performance factor. 
     * Setting a high value will potentially cause fewer islands to be generated. This may reduce thread scaling (fewer task chains spawned) and may 
     * detrimentally affect performance if some bodies in the scene have large solver iteration counts because all constraints in a given island are solved by the 
     * maximum number of solver iterations requested by any body in the island.
     * <p>
     * Note that a rigid body solver task chain is spawned as soon as either a sufficient number of rigid bodies or articulations are batched together.
     * <p>
     * <b>Default:</b> 128
     * @see PxScene#setSolverBatchSize
     * @see PxScene#getSolverBatchSize
     */
    public int getSolverBatchSize() {
        checkNotNull();
        return _getSolverBatchSize(address);
    }
    private static native int _getSolverBatchSize(long address);

    /**
     * Defines the number of actors required to spawn a separate rigid body solver island task chain.
     * <p>
     * This parameter defines the minimum number of actors required to spawn a separate rigid body solver task chain. Setting a low value 
     * will potentially cause more task chains to be generated. This may result in the overhead of spawning tasks can become a limiting performance factor. 
     * Setting a high value will potentially cause fewer islands to be generated. This may reduce thread scaling (fewer task chains spawned) and may 
     * detrimentally affect performance if some bodies in the scene have large solver iteration counts because all constraints in a given island are solved by the 
     * maximum number of solver iterations requested by any body in the island.
     * <p>
     * Note that a rigid body solver task chain is spawned as soon as either a sufficient number of rigid bodies or articulations are batched together.
     * <p>
     * <b>Default:</b> 128
     * @see PxScene#setSolverBatchSize
     * @see PxScene#getSolverBatchSize
     */
    public void setSolverBatchSize(int value) {
        checkNotNull();
        _setSolverBatchSize(address, value);
    }
    private static native void _setSolverBatchSize(long address, int value);

    /**
     * Defines the number of articulations required to spawn a separate rigid body solver island task chain.
     * <p>
     * This parameter defines the minimum number of articulations required to spawn a separate rigid body solver task chain. Setting a low value
     * will potentially cause more task chains to be generated. This may result in the overhead of spawning tasks can become a limiting performance factor.
     * Setting a high value will potentially cause fewer islands to be generated. This may reduce thread scaling (fewer task chains spawned) and may
     * detrimentally affect performance if some bodies in the scene have large solver iteration counts because all constraints in a given island are solved by the
     * maximum number of solver iterations requested by any body in the island.
     * <p>
     * Note that a rigid body solver task chain is spawned as soon as either a sufficient number of rigid bodies or articulations are batched together. 
     * <p>
     * <b>Default:</b> 16
     * @see PxScene#setSolverArticulationBatchSize
     * @see PxScene#getSolverArticulationBatchSize
     */
    public int getSolverArticulationBatchSize() {
        checkNotNull();
        return _getSolverArticulationBatchSize(address);
    }
    private static native int _getSolverArticulationBatchSize(long address);

    /**
     * Defines the number of articulations required to spawn a separate rigid body solver island task chain.
     * <p>
     * This parameter defines the minimum number of articulations required to spawn a separate rigid body solver task chain. Setting a low value
     * will potentially cause more task chains to be generated. This may result in the overhead of spawning tasks can become a limiting performance factor.
     * Setting a high value will potentially cause fewer islands to be generated. This may reduce thread scaling (fewer task chains spawned) and may
     * detrimentally affect performance if some bodies in the scene have large solver iteration counts because all constraints in a given island are solved by the
     * maximum number of solver iterations requested by any body in the island.
     * <p>
     * Note that a rigid body solver task chain is spawned as soon as either a sufficient number of rigid bodies or articulations are batched together. 
     * <p>
     * <b>Default:</b> 16
     * @see PxScene#setSolverArticulationBatchSize
     * @see PxScene#getSolverArticulationBatchSize
     */
    public void setSolverArticulationBatchSize(int value) {
        checkNotNull();
        _setSolverArticulationBatchSize(address, value);
    }
    private static native void _setSolverArticulationBatchSize(long address, int value);

    /**
     * Setting to define the number of 16K blocks that will be initially reserved to store contact, friction, and contact cache data.
     * This is the number of 16K memory blocks that will be automatically allocated from the user allocator when the scene is instantiated. Further 16k
     * memory blocks may be allocated during the simulation up to maxNbContactDataBlocks.
     * <p>
     * <b>Note:</b> This value cannot be larger than maxNbContactDataBlocks because that defines the maximum number of 16k blocks that can be allocated by the SDK.
     * <p>
     * <b>Default:</b> 0
     * <p>
     * <b>Range:</b> [0, PX_MAX_U32]<br>
     */
    public int getNbContactDataBlocks() {
        checkNotNull();
        return _getNbContactDataBlocks(address);
    }
    private static native int _getNbContactDataBlocks(long address);

    /**
     * Setting to define the number of 16K blocks that will be initially reserved to store contact, friction, and contact cache data.
     * This is the number of 16K memory blocks that will be automatically allocated from the user allocator when the scene is instantiated. Further 16k
     * memory blocks may be allocated during the simulation up to maxNbContactDataBlocks.
     * <p>
     * <b>Note:</b> This value cannot be larger than maxNbContactDataBlocks because that defines the maximum number of 16k blocks that can be allocated by the SDK.
     * <p>
     * <b>Default:</b> 0
     * <p>
     * <b>Range:</b> [0, PX_MAX_U32]<br>
     */
    public void setNbContactDataBlocks(int value) {
        checkNotNull();
        _setNbContactDataBlocks(address, value);
    }
    private static native void _setNbContactDataBlocks(long address, int value);

    /**
     * Setting to define the maximum number of 16K blocks that can be allocated to store contact, friction, and contact cache data.
     * As the complexity of a scene increases, the SDK may require to allocate new 16k blocks in addition to the blocks it has already 
     * allocated. This variable controls the maximum number of blocks that the SDK can allocate.
     * <p>
     * In the case that the scene is sufficiently complex that all the permitted 16K blocks are used, contacts will be dropped and 
     * a warning passed to the error stream.
     * <p>
     * If a warning is reported to the error stream to indicate the number of 16K blocks is insufficient for the scene complexity 
     * then the choices are either (i) re-tune the number of 16K data blocks until a number is found that is sufficient for the scene complexity,
     * (ii) to simplify the scene or (iii) to opt to not increase the memory requirements of physx and accept some dropped contacts.
     * <p>
     * <b>Default:</b> 65536
     * <p>
     * <b>Range:</b> [0, PX_MAX_U32]<br>
     * @see PxScene#setNbContactDataBlocks
     */
    public int getMaxNbContactDataBlocks() {
        checkNotNull();
        return _getMaxNbContactDataBlocks(address);
    }
    private static native int _getMaxNbContactDataBlocks(long address);

    /**
     * Setting to define the maximum number of 16K blocks that can be allocated to store contact, friction, and contact cache data.
     * As the complexity of a scene increases, the SDK may require to allocate new 16k blocks in addition to the blocks it has already 
     * allocated. This variable controls the maximum number of blocks that the SDK can allocate.
     * <p>
     * In the case that the scene is sufficiently complex that all the permitted 16K blocks are used, contacts will be dropped and 
     * a warning passed to the error stream.
     * <p>
     * If a warning is reported to the error stream to indicate the number of 16K blocks is insufficient for the scene complexity 
     * then the choices are either (i) re-tune the number of 16K data blocks until a number is found that is sufficient for the scene complexity,
     * (ii) to simplify the scene or (iii) to opt to not increase the memory requirements of physx and accept some dropped contacts.
     * <p>
     * <b>Default:</b> 65536
     * <p>
     * <b>Range:</b> [0, PX_MAX_U32]<br>
     * @see PxScene#setNbContactDataBlocks
     */
    public void setMaxNbContactDataBlocks(int value) {
        checkNotNull();
        _setMaxNbContactDataBlocks(address, value);
    }
    private static native void _setMaxNbContactDataBlocks(long address, int value);

    /**
     * The maximum bias coefficient used in the constraint solver
     * <p>
     * When geometric errors are found in the constraint solver, either as a result of shapes penetrating
     * or joints becoming separated or violating limits, a bias is introduced in the solver position iterations
     * to correct these errors. This bias is proportional to 1/dt, meaning that the bias becomes increasingly 
     * strong as the time-step passed to PxScene::simulate(...) becomes smaller. This coefficient allows the
     * application to restrict how large the bias coefficient is, to reduce how violent error corrections are.
     * This can improve simulation quality in cases where either variable time-steps or extremely small time-steps
     * are used.
     * <p>
     * <b>Default:</b> PX_MAX_F32
     * <p>
     * <b> Range</b> [0, PX_MAX_F32] <br>
     * @see PxScene#setMaxBiasCoefficient
     * @see PxScene#getMaxBiasCoefficient
     */
    public float getMaxBiasCoefficient() {
        checkNotNull();
        return _getMaxBiasCoefficient(address);
    }
    private static native float _getMaxBiasCoefficient(long address);

    /**
     * The maximum bias coefficient used in the constraint solver
     * <p>
     * When geometric errors are found in the constraint solver, either as a result of shapes penetrating
     * or joints becoming separated or violating limits, a bias is introduced in the solver position iterations
     * to correct these errors. This bias is proportional to 1/dt, meaning that the bias becomes increasingly 
     * strong as the time-step passed to PxScene::simulate(...) becomes smaller. This coefficient allows the
     * application to restrict how large the bias coefficient is, to reduce how violent error corrections are.
     * This can improve simulation quality in cases where either variable time-steps or extremely small time-steps
     * are used.
     * <p>
     * <b>Default:</b> PX_MAX_F32
     * <p>
     * <b> Range</b> [0, PX_MAX_F32] <br>
     * @see PxScene#setMaxBiasCoefficient
     * @see PxScene#getMaxBiasCoefficient
     */
    public void setMaxBiasCoefficient(float value) {
        checkNotNull();
        _setMaxBiasCoefficient(address, value);
    }
    private static native void _setMaxBiasCoefficient(long address, float value);

    /**
     * Size of the contact report stream (in bytes).
     * <p>
     * The contact report stream buffer is used during the simulation to store all the contact reports. 
     * If the size is not sufficient, the buffer will grow by a factor of two.
     * It is possible to disable the buffer growth by setting the flag PxSceneFlag::eDISABLE_CONTACT_REPORT_BUFFER_RESIZE.
     * In that case the buffer will not grow but contact reports not stored in the buffer will not get sent in the contact report callbacks.
     * <p>
     * <b>Default:</b> 8192
     * <p>
     * <b>Range:</b> (0, PX_MAX_U32]<br>
     * @see PxScene#getContactReportStreamBufferSize
     */
    public int getContactReportStreamBufferSize() {
        checkNotNull();
        return _getContactReportStreamBufferSize(address);
    }
    private static native int _getContactReportStreamBufferSize(long address);

    /**
     * Size of the contact report stream (in bytes).
     * <p>
     * The contact report stream buffer is used during the simulation to store all the contact reports. 
     * If the size is not sufficient, the buffer will grow by a factor of two.
     * It is possible to disable the buffer growth by setting the flag PxSceneFlag::eDISABLE_CONTACT_REPORT_BUFFER_RESIZE.
     * In that case the buffer will not grow but contact reports not stored in the buffer will not get sent in the contact report callbacks.
     * <p>
     * <b>Default:</b> 8192
     * <p>
     * <b>Range:</b> (0, PX_MAX_U32]<br>
     * @see PxScene#getContactReportStreamBufferSize
     */
    public void setContactReportStreamBufferSize(int value) {
        checkNotNull();
        _setContactReportStreamBufferSize(address, value);
    }
    private static native void _setContactReportStreamBufferSize(long address, int value);

    /**
     * Maximum number of CCD passes 
     * <p>
     * The CCD performs multiple passes, where each pass every object advances to its time of first impact. This value defines how many passes the CCD system should perform.
     * <p>
     * <b>Note:</b> The CCD system is a multi-pass best-effort conservative advancement approach. After the defined number of passes has been completed, any remaining time is dropped.
     * <b>Note:</b> This defines the maximum number of passes the CCD can perform. It may perform fewer if additional passes are not necessary.
     * <p>
     * <b>Default:</b> 1
     * <b>Range:</b> [1, PX_MAX_U32]<br>
     * @see PxScene#setCCDMaxPasses
     * @see PxScene#getCCDMaxPasses
     */
    public int getCcdMaxPasses() {
        checkNotNull();
        return _getCcdMaxPasses(address);
    }
    private static native int _getCcdMaxPasses(long address);

    /**
     * Maximum number of CCD passes 
     * <p>
     * The CCD performs multiple passes, where each pass every object advances to its time of first impact. This value defines how many passes the CCD system should perform.
     * <p>
     * <b>Note:</b> The CCD system is a multi-pass best-effort conservative advancement approach. After the defined number of passes has been completed, any remaining time is dropped.
     * <b>Note:</b> This defines the maximum number of passes the CCD can perform. It may perform fewer if additional passes are not necessary.
     * <p>
     * <b>Default:</b> 1
     * <b>Range:</b> [1, PX_MAX_U32]<br>
     * @see PxScene#setCCDMaxPasses
     * @see PxScene#getCCDMaxPasses
     */
    public void setCcdMaxPasses(int value) {
        checkNotNull();
        _setCcdMaxPasses(address, value);
    }
    private static native void _setCcdMaxPasses(long address, int value);

    /**
     * CCD threshold
     * <p>
     * CCD performs sweeps against shapes if and only if the relative motion of the shapes is fast-enough that a collision would be missed
     * by the discrete contact generation. However, in some circumstances, e.g. when the environment is constructed from large convex shapes, this 
     * approach may produce undesired simulation artefacts. This parameter defines the minimum relative motion that would be required to force CCD between shapes.
     * The smaller of this value and the sum of the thresholds calculated for the shapes involved will be used.
     * <p>
     * <b>Note:</b> It is not advisable to set this to a very small value as this may lead to CCD "jamming" and detrimentally effect performance. This value should be at least larger than the translation caused by a single frame's gravitational effect
     * <p>
     * <b>Default:</b> PX_MAX_F32
     * <b>Range:</b> [Eps, PX_MAX_F32]<br>
     * @see PxScene#setCCDThreshold
     * @see PxScene#getCCDThreshold
     */
    public float getCcdThreshold() {
        checkNotNull();
        return _getCcdThreshold(address);
    }
    private static native float _getCcdThreshold(long address);

    /**
     * CCD threshold
     * <p>
     * CCD performs sweeps against shapes if and only if the relative motion of the shapes is fast-enough that a collision would be missed
     * by the discrete contact generation. However, in some circumstances, e.g. when the environment is constructed from large convex shapes, this 
     * approach may produce undesired simulation artefacts. This parameter defines the minimum relative motion that would be required to force CCD between shapes.
     * The smaller of this value and the sum of the thresholds calculated for the shapes involved will be used.
     * <p>
     * <b>Note:</b> It is not advisable to set this to a very small value as this may lead to CCD "jamming" and detrimentally effect performance. This value should be at least larger than the translation caused by a single frame's gravitational effect
     * <p>
     * <b>Default:</b> PX_MAX_F32
     * <b>Range:</b> [Eps, PX_MAX_F32]<br>
     * @see PxScene#setCCDThreshold
     * @see PxScene#getCCDThreshold
     */
    public void setCcdThreshold(float value) {
        checkNotNull();
        _setCcdThreshold(address, value);
    }
    private static native void _setCcdThreshold(long address, float value);

    /**
     * A threshold for speculative CCD. Used to control whether bias, restitution or a combination of the two are used to resolve the contacts.
     * <p>
     * <b>Note:</b> This only has any effect on contacting pairs where one of the bodies has PxRigidBodyFlag::eENABLE_SPECULATIVE_CCD raised.
     * <p>
     * <b>Range:</b> [0, PX_MAX_F32)<br>
     * <b>Default:</b> 0.04 * PxTolerancesScale::length
     * @see PxScene#setCCDMaxSeparation
     * @see PxScene#getCCDMaxSeparation
     */
    public float getCcdMaxSeparation() {
        checkNotNull();
        return _getCcdMaxSeparation(address);
    }
    private static native float _getCcdMaxSeparation(long address);

    /**
     * A threshold for speculative CCD. Used to control whether bias, restitution or a combination of the two are used to resolve the contacts.
     * <p>
     * <b>Note:</b> This only has any effect on contacting pairs where one of the bodies has PxRigidBodyFlag::eENABLE_SPECULATIVE_CCD raised.
     * <p>
     * <b>Range:</b> [0, PX_MAX_F32)<br>
     * <b>Default:</b> 0.04 * PxTolerancesScale::length
     * @see PxScene#setCCDMaxSeparation
     * @see PxScene#getCCDMaxSeparation
     */
    public void setCcdMaxSeparation(float value) {
        checkNotNull();
        _setCcdMaxSeparation(address, value);
    }
    private static native void _setCcdMaxSeparation(long address, float value);

    /**
     * The wake counter reset value
     * <p>
     * Calling wakeUp() on objects which support sleeping will set their wake counter value to the specified reset value.
     * <p>
     * <b>Range:</b> (0, PX_MAX_F32)<br>
     * <b>Default:</b> 0.4 (which corresponds to 20 frames for a time step of 0.02)
     * @see PxScene#getWakeCounterResetValue
     */
    public float getWakeCounterResetValue() {
        checkNotNull();
        return _getWakeCounterResetValue(address);
    }
    private static native float _getWakeCounterResetValue(long address);

    /**
     * The wake counter reset value
     * <p>
     * Calling wakeUp() on objects which support sleeping will set their wake counter value to the specified reset value.
     * <p>
     * <b>Range:</b> (0, PX_MAX_F32)<br>
     * <b>Default:</b> 0.4 (which corresponds to 20 frames for a time step of 0.02)
     * @see PxScene#getWakeCounterResetValue
     */
    public void setWakeCounterResetValue(float value) {
        checkNotNull();
        _setWakeCounterResetValue(address, value);
    }
    private static native void _setWakeCounterResetValue(long address, float value);

    /**
     * The bounds used to sanity check user-set positions of actors and articulation links
     * <p>
     * These bounds are used to check the position values of rigid actors inserted into the scene, and positions set for rigid actors
     * already within the scene.
     * <p>
     * <b>Range:</b> any valid PxBounds3 <br> 
     * <b>Default:</b> (-PX_MAX_BOUNDS_EXTENTS, PX_MAX_BOUNDS_EXTENTS) on each axis
     */
    public PxBounds3 getSanityBounds() {
        checkNotNull();
        return PxBounds3.wrapPointer(_getSanityBounds(address));
    }
    private static native long _getSanityBounds(long address);

    /**
     * The bounds used to sanity check user-set positions of actors and articulation links
     * <p>
     * These bounds are used to check the position values of rigid actors inserted into the scene, and positions set for rigid actors
     * already within the scene.
     * <p>
     * <b>Range:</b> any valid PxBounds3 <br> 
     * <b>Default:</b> (-PX_MAX_BOUNDS_EXTENTS, PX_MAX_BOUNDS_EXTENTS) on each axis
     */
    public void setSanityBounds(PxBounds3 value) {
        checkNotNull();
        _setSanityBounds(address, value.getAddress());
    }
    private static native void _setSanityBounds(long address, long value);

    /**
     * The pre-allocations performed in the GPU dynamics pipeline.
     */
    public PxGpuDynamicsMemoryConfig getGpuDynamicsConfig() {
        checkNotNull();
        PlatformChecks.requirePlatform(3, "physx.physics.PxSceneDesc");
        return PxGpuDynamicsMemoryConfig.wrapPointer(_getGpuDynamicsConfig(address));
    }
    private static native long _getGpuDynamicsConfig(long address);

    /**
     * The pre-allocations performed in the GPU dynamics pipeline.
     */
    public void setGpuDynamicsConfig(PxGpuDynamicsMemoryConfig value) {
        checkNotNull();
        PlatformChecks.requirePlatform(3, "physx.physics.PxSceneDesc");
        _setGpuDynamicsConfig(address, value.getAddress());
    }
    private static native void _setGpuDynamicsConfig(long address, long value);

    /**
     * Limitation for the partitions in the GPU dynamics pipeline.
     * This variable must be power of 2.
     * A value greater than 32 is currently not supported.
     * <b>Range:</b> (1, 32)<br>
     */
    public int getGpuMaxNumPartitions() {
        checkNotNull();
        return _getGpuMaxNumPartitions(address);
    }
    private static native int _getGpuMaxNumPartitions(long address);

    /**
     * Limitation for the partitions in the GPU dynamics pipeline.
     * This variable must be power of 2.
     * A value greater than 32 is currently not supported.
     * <b>Range:</b> (1, 32)<br>
     */
    public void setGpuMaxNumPartitions(int value) {
        checkNotNull();
        _setGpuMaxNumPartitions(address, value);
    }
    private static native void _setGpuMaxNumPartitions(long address, int value);

    /**
     * Limitation for the number of static rigid body partitions in the GPU dynamics pipeline.
     * <b>Range:</b> (1, 255)<br>
     * <b>Default:</b> 16
     */
    public int getGpuMaxNumStaticPartitions() {
        checkNotNull();
        return _getGpuMaxNumStaticPartitions(address);
    }
    private static native int _getGpuMaxNumStaticPartitions(long address);

    /**
     * Limitation for the number of static rigid body partitions in the GPU dynamics pipeline.
     * <b>Range:</b> (1, 255)<br>
     * <b>Default:</b> 16
     */
    public void setGpuMaxNumStaticPartitions(int value) {
        checkNotNull();
        _setGpuMaxNumStaticPartitions(address, value);
    }
    private static native void _setGpuMaxNumStaticPartitions(long address, int value);

    /**
     * Defines which compute version the GPU dynamics should target. DO NOT MODIFY
     */
    public int getGpuComputeVersion() {
        checkNotNull();
        return _getGpuComputeVersion(address);
    }
    private static native int _getGpuComputeVersion(long address);

    /**
     * Defines which compute version the GPU dynamics should target. DO NOT MODIFY
     */
    public void setGpuComputeVersion(int value) {
        checkNotNull();
        _setGpuComputeVersion(address, value);
    }
    private static native void _setGpuComputeVersion(long address, int value);

    /**
     * Defines the size of a contact pool slab.
     * Contact pairs and associated data are allocated using a pool allocator. Increasing the slab size can trade
     * off some performance spikes when a large number of new contacts are found for an increase in overall memory 
     * usage.
     * <p>
     * <b>Range:</b>(1, PX_MAX_U32)<br>
     * <b>Default:</b> 256
     */
    public int getContactPairSlabSize() {
        checkNotNull();
        return _getContactPairSlabSize(address);
    }
    private static native int _getContactPairSlabSize(long address);

    /**
     * Defines the size of a contact pool slab.
     * Contact pairs and associated data are allocated using a pool allocator. Increasing the slab size can trade
     * off some performance spikes when a large number of new contacts are found for an increase in overall memory 
     * usage.
     * <p>
     * <b>Range:</b>(1, PX_MAX_U32)<br>
     * <b>Default:</b> 256
     */
    public void setContactPairSlabSize(int value) {
        checkNotNull();
        _setContactPairSlabSize(address, value);
    }
    private static native void _setContactPairSlabSize(long address, int value);

    /**
     * @return WebIDL type: {@link PxPruningStructureTypeEnum} [enum]
     */
    public PxPruningStructureTypeEnum getStaticStructure() {
        checkNotNull();
        return PxPruningStructureTypeEnum.forValue(_getStaticStructure(address));
    }
    private static native int _getStaticStructure(long address);

    /**
     * @param value WebIDL type: {@link PxPruningStructureTypeEnum} [enum]
     */
    public void setStaticStructure(PxPruningStructureTypeEnum value) {
        checkNotNull();
        _setStaticStructure(address, value.value);
    }
    private static native void _setStaticStructure(long address, int value);

    /**
     * @return WebIDL type: {@link PxPruningStructureTypeEnum} [enum]
     */
    public PxPruningStructureTypeEnum getDynamicStructure() {
        checkNotNull();
        return PxPruningStructureTypeEnum.forValue(_getDynamicStructure(address));
    }
    private static native int _getDynamicStructure(long address);

    /**
     * @param value WebIDL type: {@link PxPruningStructureTypeEnum} [enum]
     */
    public void setDynamicStructure(PxPruningStructureTypeEnum value) {
        checkNotNull();
        _setDynamicStructure(address, value.value);
    }
    private static native void _setDynamicStructure(long address, int value);

    /**
     * @return WebIDL type: unsigned long
     */
    public int getDynamicTreeRebuildRateHint() {
        checkNotNull();
        return _getDynamicTreeRebuildRateHint(address);
    }
    private static native int _getDynamicTreeRebuildRateHint(long address);

    /**
     * @param value WebIDL type: unsigned long
     */
    public void setDynamicTreeRebuildRateHint(int value) {
        checkNotNull();
        _setDynamicTreeRebuildRateHint(address, value);
    }
    private static native void _setDynamicTreeRebuildRateHint(long address, int value);

    /**
     * @return WebIDL type: {@link PxDynamicTreeSecondaryPrunerEnum} [enum]
     */
    public PxDynamicTreeSecondaryPrunerEnum getDynamicTreeSecondaryPruner() {
        checkNotNull();
        return PxDynamicTreeSecondaryPrunerEnum.forValue(_getDynamicTreeSecondaryPruner(address));
    }
    private static native int _getDynamicTreeSecondaryPruner(long address);

    /**
     * @param value WebIDL type: {@link PxDynamicTreeSecondaryPrunerEnum} [enum]
     */
    public void setDynamicTreeSecondaryPruner(PxDynamicTreeSecondaryPrunerEnum value) {
        checkNotNull();
        _setDynamicTreeSecondaryPruner(address, value.value);
    }
    private static native void _setDynamicTreeSecondaryPruner(long address, int value);

    /**
     * @return WebIDL type: {@link PxBVHBuildStrategyEnum} [enum]
     */
    public PxBVHBuildStrategyEnum getStaticBVHBuildStrategy() {
        checkNotNull();
        return PxBVHBuildStrategyEnum.forValue(_getStaticBVHBuildStrategy(address));
    }
    private static native int _getStaticBVHBuildStrategy(long address);

    /**
     * @param value WebIDL type: {@link PxBVHBuildStrategyEnum} [enum]
     */
    public void setStaticBVHBuildStrategy(PxBVHBuildStrategyEnum value) {
        checkNotNull();
        _setStaticBVHBuildStrategy(address, value.value);
    }
    private static native void _setStaticBVHBuildStrategy(long address, int value);

    /**
     * @return WebIDL type: {@link PxBVHBuildStrategyEnum} [enum]
     */
    public PxBVHBuildStrategyEnum getDynamicBVHBuildStrategy() {
        checkNotNull();
        return PxBVHBuildStrategyEnum.forValue(_getDynamicBVHBuildStrategy(address));
    }
    private static native int _getDynamicBVHBuildStrategy(long address);

    /**
     * @param value WebIDL type: {@link PxBVHBuildStrategyEnum} [enum]
     */
    public void setDynamicBVHBuildStrategy(PxBVHBuildStrategyEnum value) {
        checkNotNull();
        _setDynamicBVHBuildStrategy(address, value.value);
    }
    private static native void _setDynamicBVHBuildStrategy(long address, int value);

    /**
     * @return WebIDL type: unsigned long
     */
    public int getStaticNbObjectsPerNode() {
        checkNotNull();
        return _getStaticNbObjectsPerNode(address);
    }
    private static native int _getStaticNbObjectsPerNode(long address);

    /**
     * @param value WebIDL type: unsigned long
     */
    public void setStaticNbObjectsPerNode(int value) {
        checkNotNull();
        _setStaticNbObjectsPerNode(address, value);
    }
    private static native void _setStaticNbObjectsPerNode(long address, int value);

    /**
     * @return WebIDL type: unsigned long
     */
    public int getDynamicNbObjectsPerNode() {
        checkNotNull();
        return _getDynamicNbObjectsPerNode(address);
    }
    private static native int _getDynamicNbObjectsPerNode(long address);

    /**
     * @param value WebIDL type: unsigned long
     */
    public void setDynamicNbObjectsPerNode(int value) {
        checkNotNull();
        _setDynamicNbObjectsPerNode(address, value);
    }
    private static native void _setDynamicNbObjectsPerNode(long address, int value);

    /**
     * @return WebIDL type: {@link PxSceneQueryUpdateModeEnum} [enum]
     */
    public PxSceneQueryUpdateModeEnum getSceneQueryUpdateMode() {
        checkNotNull();
        return PxSceneQueryUpdateModeEnum.forValue(_getSceneQueryUpdateMode(address));
    }
    private static native int _getSceneQueryUpdateMode(long address);

    /**
     * @param value WebIDL type: {@link PxSceneQueryUpdateModeEnum} [enum]
     */
    public void setSceneQueryUpdateMode(PxSceneQueryUpdateModeEnum value) {
        checkNotNull();
        _setSceneQueryUpdateMode(address, value.value);
    }
    private static native void _setSceneQueryUpdateMode(long address, int value);

    // Functions

    /**
     * (re)sets the structure to the default.
     * @param scale scale values for the tolerances in the scene, these must be the same values passed into
     * PxCreatePhysics(). The affected tolerances are bounceThresholdVelocity and frictionOffsetThreshold.
     * @see physx.common.PxTolerancesScale
     */
    public void setToDefault(PxTolerancesScale scale) {
        checkNotNull();
        _setToDefault(address, scale.getAddress());
    }
    private static native void _setToDefault(long address, long scale);

    /**
     * Returns true if the descriptor is valid.
     * @return true if the current settings are valid.
     */
    public boolean isValid() {
        checkNotNull();
        return _isValid(address);
    }
    private static native boolean _isValid(long address);

}
