package physx.physics;

import physx.NativeObject;
import physx.PlatformChecks;
import physx.common.PxCudaContextManager;
import physx.common.PxFoundation;
import physx.common.PxInputData;
import physx.common.PxInsertionCallback;
import physx.common.PxTolerancesScale;
import physx.common.PxTransform;
import physx.geometry.PxConvexMesh;
import physx.geometry.PxGeometry;
import physx.geometry.PxTriangleMesh;
import physx.particles.PxPBDMaterial;
import physx.particles.PxPBDParticleSystem;
import physx.particles.PxParticleBuffer;
import physx.particles.PxParticleClothBuffer;

/**
 * Abstract singleton factory class used for instancing objects in the Physics SDK.
 * <p>
 * In addition you can use PxPhysics to set global parameters which will effect all scenes and create
 * objects that can be shared across multiple scenes.
 * <p>
 * You can get an instance of this class by calling PxCreatePhysics().
 * @see PxScene
 */
public class PxPhysics extends NativeObject {

    protected PxPhysics() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxPhysics wrapPointer(long address) {
        return address != 0L ? new PxPhysics(address) : null;
    }
    
    public static PxPhysics arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxPhysics(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Destroys the instance it is called on.
     * <p>
     * Use this release method to destroy an instance of this class. Be sure
     * to not keep a reference to this object after calling release.
     * Avoid release calls while a scene is simulating (in between simulate() and fetchResults() calls).
     * <p>
     * Note that this must be called once for each prior call to PxCreatePhysics, as
     * there is a reference counter. Also note that you mustn't destroy the PxFoundation instance (holding the allocator, error callback etc.)
     * until after the reference count reaches 0 and the SDK is actually removed.
     * <p>
     * Releasing an SDK will also release any objects created through it (scenes, triangle meshes, convex meshes, heightfields, shapes etc.),
     * provided the user hasn't already done so.
     * <p>
     * <b>Note:</b> Releasing the PxPhysics instance is a prerequisite to releasing the PxFoundation instance.
     * @see physx.common.PxFoundation
     */
    public void release() {
        checkNotNull();
        _release(address);
    }
    private static native void _release(long address);

    /**
     * Retrieves the Foundation instance.
     * @return A reference to the Foundation object.
     */
    public PxFoundation getFoundation() {
        checkNotNull();
        return PxFoundation.wrapPointer(_getFoundation(address));
    }
    private static native long _getFoundation(long address);

    /**
     * Creates an aggregate with the specified maximum size and filtering hint.
     * <p>
     * The previous API used "bool enableSelfCollision" which should now silently evaluates
     * to a PxAggregateType::eGENERIC aggregate with its self-collision bit.
     * <p>
     * Use PxAggregateType::eSTATIC or PxAggregateType::eKINEMATIC for aggregates that will
     * only contain static or kinematic actors. This provides faster filtering when used in
     * combination with PxPairFilteringMode.
     * @return The new aggregate.
     * @see PxAggregate
     */
    public PxAggregate createAggregate(int maxActor, int maxShape, boolean enableSelfCollision) {
        checkNotNull();
        return PxAggregate.wrapPointer(_createAggregate(address, maxActor, maxShape, enableSelfCollision));
    }
    private static native long _createAggregate(long address, int maxActor, int maxShape, boolean enableSelfCollision);

    /**
     * Returns the simulation tolerance parameters.
     * @return The current simulation tolerance parameters.
     */
    public PxTolerancesScale getTolerancesScale() {
        checkNotNull();
        return PxTolerancesScale.wrapPointer(_getTolerancesScale(address));
    }
    private static native long _getTolerancesScale(long address);

    /**
     * Creates a scene.
     * <p>
     * <b>Note:</b> Every scene uses a Thread Local Storage slot. This imposes a platform specific limit on the
     * number of scenes that can be created.
     * @return The new scene object.
     * @see PxScene
     * @see PxScene#release
     * @see PxSceneDesc
     */
    public PxScene createScene(PxSceneDesc sceneDesc) {
        checkNotNull();
        return PxScene.wrapPointer(_createScene(address, sceneDesc.getAddress()));
    }
    private static native long _createScene(long address, long sceneDesc);

    /**
     * Creates a static rigid actor with the specified pose and all other fields initialized
     * to their default values.
     * @see PxRigidStatic
     */
    public PxRigidStatic createRigidStatic(PxTransform pose) {
        checkNotNull();
        return PxRigidStatic.wrapPointer(_createRigidStatic(address, pose.getAddress()));
    }
    private static native long _createRigidStatic(long address, long pose);

    /**
     * Creates a dynamic rigid actor with the specified pose and all other fields initialized
     * to their default values.
     * @see PxRigidDynamic
     */
    public PxRigidDynamic createRigidDynamic(PxTransform pose) {
        checkNotNull();
        return PxRigidDynamic.wrapPointer(_createRigidDynamic(address, pose.getAddress()));
    }
    private static native long _createRigidDynamic(long address, long pose);

    /**
     * Creates a shape which may be attached to multiple actors
     * <p>
     * The shape will be created with a reference count of 1.
     * @return The shape
     * <p>
     * <b>Note:</b> Shared shapes are not mutable when they are attached to an actor
     * @see PxShape
     */
    public PxShape createShape(PxGeometry geometry, PxMaterial material) {
        checkNotNull();
        return PxShape.wrapPointer(_createShape(address, geometry.getAddress(), material.getAddress()));
    }
    private static native long _createShape(long address, long geometry, long material);

    /**
     * Creates a shape which may be attached to multiple actors
     * <p>
     * The shape will be created with a reference count of 1.
     * @return The shape
     * <p>
     * <b>Note:</b> Shared shapes are not mutable when they are attached to an actor
     * @see PxShape
     */
    public PxShape createShape(PxGeometry geometry, PxMaterial material, boolean isExclusive) {
        checkNotNull();
        return PxShape.wrapPointer(_createShape(address, geometry.getAddress(), material.getAddress(), isExclusive));
    }
    private static native long _createShape(long address, long geometry, long material, boolean isExclusive);

    /**
     * Creates a shape which may be attached to multiple actors
     * <p>
     * The shape will be created with a reference count of 1.
     * @return The shape
     * <p>
     * <b>Note:</b> Shared shapes are not mutable when they are attached to an actor
     * @see PxShape
     */
    public PxShape createShape(PxGeometry geometry, PxMaterial material, boolean isExclusive, PxShapeFlags shapeFlags) {
        checkNotNull();
        return PxShape.wrapPointer(_createShape(address, geometry.getAddress(), material.getAddress(), isExclusive, shapeFlags.getAddress()));
    }
    private static native long _createShape(long address, long geometry, long material, boolean isExclusive, long shapeFlags);

    /**
     * Creates a triangle mesh object.
     * <p>
     * This can then be instanced into #PxShape objects.
     * @return The new triangle mesh.
     * @see physx.geometry.PxTriangleMesh
     */
    public PxTriangleMesh createTriangleMesh(PxInputData stream) {
        checkNotNull();
        return PxTriangleMesh.wrapPointer(_createTriangleMesh(address, stream.getAddress()));
    }
    private static native long _createTriangleMesh(long address, long stream);

    /**
     * Creates a convex mesh object.
     * <p>
     * This can then be instanced into #PxShape objects.
     * @return The new convex mesh.
     * @see physx.geometry.PxConvexMesh
     * @see #createTriangleMesh
     * @see physx.geometry.PxConvexMeshGeometry
     * @see PxShape
     */
    public PxConvexMesh createConvexMesh(PxInputData stream) {
        checkNotNull();
        return PxConvexMesh.wrapPointer(_createConvexMesh(address, stream.getAddress()));
    }
    private static native long _createConvexMesh(long address, long stream);

    /**
     * Return the number of shapes that currently exist.
     * @return Number of shapes.
     */
    public int getNbShapes() {
        checkNotNull();
        return _getNbShapes(address);
    }
    private static native int _getNbShapes(long address);

    /**
     * Creates a reduced-coordinate articulation with all fields initialized to their default values.
     * @return the new articulation
     * @see PxArticulationReducedCoordinate
     */
    public PxArticulationReducedCoordinate createArticulationReducedCoordinate() {
        checkNotNull();
        return PxArticulationReducedCoordinate.wrapPointer(_createArticulationReducedCoordinate(address));
    }
    private static native long _createArticulationReducedCoordinate(long address);

    /**
     * Creates a new rigid body material with certain default properties.
     * @return The new rigid body material.
     * @see PxMaterial
     */
    public PxMaterial createMaterial(float staticFriction, float dynamicFriction, float restitution) {
        checkNotNull();
        return PxMaterial.wrapPointer(_createMaterial(address, staticFriction, dynamicFriction, restitution));
    }
    private static native long _createMaterial(long address, float staticFriction, float dynamicFriction, float restitution);

    /**
     * Gets PxPhysics object insertion interface.
     * <p>
     * The insertion interface is needed for PxCreateTriangleMesh, PxCooking::createTriangleMesh etc., this allows runtime mesh creation.
     * <p>
     *      PxCooking::createTriangleMesh PxCooking::createHeightfield PxCooking::createTetrahedronMesh PxCooking::createBVH
     */
    public PxInsertionCallback getPhysicsInsertionCallback() {
        checkNotNull();
        return PxInsertionCallback.wrapPointer(_getPhysicsInsertionCallback(address));
    }
    private static native long _getPhysicsInsertionCallback(long address);

    /**
     * Creates a particle system with a position-based dynamics (PBD) solver.
     * <p>
     * A PBD particle system can be used to simulate particle systems with fluid and granular particles. It also allows simulating cloth using
     * mass-spring constraints and rigid bodies by shape matching the bodies with particles.
     * <p>
     * In order to accelerate neighborhood finding for particle-particle interactions (e.g.: for fluid density constraints) a regular grid is used.
     * This grid is built every time step but may provide inaccurate neighborhood information during the solver iterations. The neighborhood scale
     * parameter can be used to configure the grid such that it provides a more conservative neighborhood at the cost of run-time performance. 
     * <p>
     * The neighborhood scale parameter should typically not be much larger than 1.
     * <p>
     * The maxNeighborhood defines how many particles fit into the neighborhood, at the cost of memory.
     * <p>
     * Both maxNeighborhood and neighborhoodScale should be set as low as possible for performance, but high enough to not cause any behavioral degredation.
     * @param cudaContextManager The PxCudaContextManager this instance is tied to.
     * @return the new particle system
     * @see physx.particles.PxPBDParticleSystem
     */
    public PxPBDParticleSystem createPBDParticleSystem(PxCudaContextManager cudaContextManager) {
        checkNotNull();
        PlatformChecks.requirePlatform(3, "physx.physics.PxPhysics");
        return PxPBDParticleSystem.wrapPointer(_createPBDParticleSystem(address, cudaContextManager.getAddress()));
    }
    private static native long _createPBDParticleSystem(long address, long cudaContextManager);

    /**
     * Creates a particle system with a position-based dynamics (PBD) solver.
     * <p>
     * A PBD particle system can be used to simulate particle systems with fluid and granular particles. It also allows simulating cloth using
     * mass-spring constraints and rigid bodies by shape matching the bodies with particles.
     * <p>
     * In order to accelerate neighborhood finding for particle-particle interactions (e.g.: for fluid density constraints) a regular grid is used.
     * This grid is built every time step but may provide inaccurate neighborhood information during the solver iterations. The neighborhood scale
     * parameter can be used to configure the grid such that it provides a more conservative neighborhood at the cost of run-time performance. 
     * <p>
     * The neighborhood scale parameter should typically not be much larger than 1.
     * <p>
     * The maxNeighborhood defines how many particles fit into the neighborhood, at the cost of memory.
     * <p>
     * Both maxNeighborhood and neighborhoodScale should be set as low as possible for performance, but high enough to not cause any behavioral degredation.
     * @param cudaContextManager The PxCudaContextManager this instance is tied to.
     * @param maxNeighborhood The maximum number of particles considered in neighborhood-based particle interaction calculations.
     * @return the new particle system
     * @see physx.particles.PxPBDParticleSystem
     */
    public PxPBDParticleSystem createPBDParticleSystem(PxCudaContextManager cudaContextManager, int maxNeighborhood) {
        checkNotNull();
        PlatformChecks.requirePlatform(3, "physx.physics.PxPhysics");
        return PxPBDParticleSystem.wrapPointer(_createPBDParticleSystem(address, cudaContextManager.getAddress(), maxNeighborhood));
    }
    private static native long _createPBDParticleSystem(long address, long cudaContextManager, int maxNeighborhood);

    /**
     * Create particle buffer to simulate fluid/granular material.
     * @param maxParticles The maximum number of particles in this buffer.
     * @param maxVolumes The maximum number of volumes in this buffer. See PxParticleVolume.
     * @param cudaContextManager The PxCudaContextManager this buffer is tied to.
     * @return PxParticleBuffer instance
     * @see physx.particles.PxParticleBuffer
     */
    public PxParticleBuffer createParticleBuffer(int maxParticles, int maxVolumes, PxCudaContextManager cudaContextManager) {
        checkNotNull();
        PlatformChecks.requirePlatform(3, "physx.physics.PxPhysics");
        return PxParticleBuffer.wrapPointer(_createParticleBuffer(address, maxParticles, maxVolumes, cudaContextManager.getAddress()));
    }
    private static native long _createParticleBuffer(long address, int maxParticles, int maxVolumes, long cudaContextManager);

    /**
     * Create a particle buffer to simulate particle cloth.
     * @param maxParticles The maximum number of particles in this buffer.
     * @param maxNumVolumes The maximum number of volumes in this buffer. See #PxParticleVolume.
     * @param maxNumCloths The maximum number of cloths in this buffer. See #PxParticleCloth.
     * @param maxNumTriangles The maximum number of triangles for aerodynamics.
     * @param maxNumSprings The maximum number of springs to connect particles. See #PxParticleSpring.
     * @param cudaContextManager The PxCudaContextManager this buffer is tied to.
     * @return PxParticleClothBuffer instance
     * @see physx.particles.PxParticleClothBuffer
     */
    public PxParticleClothBuffer createParticleClothBuffer(int maxParticles, int maxNumVolumes, int maxNumCloths, int maxNumTriangles, int maxNumSprings, PxCudaContextManager cudaContextManager) {
        checkNotNull();
        PlatformChecks.requirePlatform(3, "physx.physics.PxPhysics");
        return PxParticleClothBuffer.wrapPointer(_createParticleClothBuffer(address, maxParticles, maxNumVolumes, maxNumCloths, maxNumTriangles, maxNumSprings, cudaContextManager.getAddress()));
    }
    private static native long _createParticleClothBuffer(long address, int maxParticles, int maxNumVolumes, int maxNumCloths, int maxNumTriangles, int maxNumSprings, long cudaContextManager);

    /**
     * Creates a new PBD material with certain default properties.
     * @return The new PBD material.
     * @see physx.particles.PxPBDMaterial
     */
    public PxPBDMaterial createPBDMaterial(float friction, float damping, float adhesion, float viscosity, float vorticityConfinement, float surfaceTension, float cohesion, float lift, float drag) {
        checkNotNull();
        PlatformChecks.requirePlatform(3, "physx.physics.PxPhysics");
        return PxPBDMaterial.wrapPointer(_createPBDMaterial(address, friction, damping, adhesion, viscosity, vorticityConfinement, surfaceTension, cohesion, lift, drag));
    }
    private static native long _createPBDMaterial(long address, float friction, float damping, float adhesion, float viscosity, float vorticityConfinement, float surfaceTension, float cohesion, float lift, float drag);

    /**
     * Creates a new PBD material with certain default properties.
     * @return The new PBD material.
     * @see physx.particles.PxPBDMaterial
     */
    public PxPBDMaterial createPBDMaterial(float friction, float damping, float adhesion, float viscosity, float vorticityConfinement, float surfaceTension, float cohesion, float lift, float drag, float cflCoefficient) {
        checkNotNull();
        PlatformChecks.requirePlatform(3, "physx.physics.PxPhysics");
        return PxPBDMaterial.wrapPointer(_createPBDMaterial(address, friction, damping, adhesion, viscosity, vorticityConfinement, surfaceTension, cohesion, lift, drag, cflCoefficient));
    }
    private static native long _createPBDMaterial(long address, float friction, float damping, float adhesion, float viscosity, float vorticityConfinement, float surfaceTension, float cohesion, float lift, float drag, float cflCoefficient);

    /**
     * Creates a new PBD material with certain default properties.
     * @return The new PBD material.
     * @see physx.particles.PxPBDMaterial
     */
    public PxPBDMaterial createPBDMaterial(float friction, float damping, float adhesion, float viscosity, float vorticityConfinement, float surfaceTension, float cohesion, float lift, float drag, float cflCoefficient, float gravityScale) {
        checkNotNull();
        PlatformChecks.requirePlatform(3, "physx.physics.PxPhysics");
        return PxPBDMaterial.wrapPointer(_createPBDMaterial(address, friction, damping, adhesion, viscosity, vorticityConfinement, surfaceTension, cohesion, lift, drag, cflCoefficient, gravityScale));
    }
    private static native long _createPBDMaterial(long address, float friction, float damping, float adhesion, float viscosity, float vorticityConfinement, float surfaceTension, float cohesion, float lift, float drag, float cflCoefficient, float gravityScale);

}
