package physx.physics;

import physx.common.PxVec3;

/**
 * Scene query hit information for raycasts and sweeps returning hit position and normal information.
 * <p>
 * ::PxHitFlag flags can be passed to scene query functions, as an optimization, to cause the SDK to
 * only generate specific members of this structure.
 */
public class PxLocationHit extends PxQueryHit {

    protected PxLocationHit() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxLocationHit wrapPointer(long address) {
        return address != 0L ? new PxLocationHit(address) : null;
    }
    
    public static PxLocationHit arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxLocationHit(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * Hit flags specifying which members contain valid values.
     */
    public PxHitFlags getFlags() {
        checkNotNull();
        return PxHitFlags.wrapPointer(_getFlags(address));
    }
    private static native long _getFlags(long address);

    /**
     * Hit flags specifying which members contain valid values.
     */
    public void setFlags(PxHitFlags value) {
        checkNotNull();
        _setFlags(address, value.getAddress());
    }
    private static native void _setFlags(long address, long value);

    /**
     * World-space hit position (flag: #PxHitFlag::ePOSITION)
     */
    public PxVec3 getPosition() {
        checkNotNull();
        return PxVec3.wrapPointer(_getPosition(address));
    }
    private static native long _getPosition(long address);

    /**
     * World-space hit position (flag: #PxHitFlag::ePOSITION)
     */
    public void setPosition(PxVec3 value) {
        checkNotNull();
        _setPosition(address, value.getAddress());
    }
    private static native void _setPosition(long address, long value);

    /**
     * World-space hit normal (flag: #PxHitFlag::eNORMAL)
     */
    public PxVec3 getNormal() {
        checkNotNull();
        return PxVec3.wrapPointer(_getNormal(address));
    }
    private static native long _getNormal(long address);

    /**
     * World-space hit normal (flag: #PxHitFlag::eNORMAL)
     */
    public void setNormal(PxVec3 value) {
        checkNotNull();
        _setNormal(address, value.getAddress());
    }
    private static native void _setNormal(long address, long value);

    /**
     * Distance to hit.
     * <b>Note:</b> If the eMTD flag is used, distance will be a negative value if shapes are overlapping indicating the penetration depth.
     * <b>Note:</b> Otherwise, this value will be &gt;= 0 
     */
    public float getDistance() {
        checkNotNull();
        return _getDistance(address);
    }
    private static native float _getDistance(long address);

    /**
     * Distance to hit.
     * <b>Note:</b> If the eMTD flag is used, distance will be a negative value if shapes are overlapping indicating the penetration depth.
     * <b>Note:</b> Otherwise, this value will be &gt;= 0 
     */
    public void setDistance(float value) {
        checkNotNull();
        _setDistance(address, value);
    }
    private static native void _setDistance(long address, float value);

}
