package physx.physics;

import physx.NativeObject;
import physx.common.PxBase;
import physx.common.PxVec3;

/**
 * This class connects a custom constraint to the SDK
 * <p>
 * This class connects a custom constraint to the SDK, and functions are called by the SDK
 * to query the custom implementation for specific information to pass on to the application
 * or inform the constraint when the application makes calls into the SDK which will update
 * the custom constraint's internal implementation
 */
public class PxConstraintConnector extends NativeObject {

    protected PxConstraintConnector() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxConstraintConnector wrapPointer(long address) {
        return address != 0L ? new PxConstraintConnector(address) : null;
    }
    
    public static PxConstraintConnector arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxConstraintConnector(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     *  Pre-simulation data preparation
     * when the constraint is marked dirty, this function is called at the start of the simulation
     * step for the SDK to copy the constraint data block.
     */
    public void prepareData() {
        checkNotNull();
        _prepareData(address);
    }
    private static native void _prepareData(long address);

    /**
     * this function is called by the SDK to update OmniPVD's view of it
     */
    public void updateOmniPvdProperties() {
        checkNotNull();
        _updateOmniPvdProperties(address);
    }
    private static native void _updateOmniPvdProperties(long address);

    /**
     * Constraint release callback
     * <p>
     * When the SDK deletes a PxConstraint object this function is called by the SDK. In general
     * custom constraints should not be deleted directly by applications: rather, the constraint
     * should respond to a release() request by calling PxConstraint::release(), then wait for
     * this call to release its own resources.
     * <p>
     * This function is also called when a PxConstraint object is deleted on cleanup due to 
     * destruction of the PxPhysics object.
     */
    public void onConstraintRelease() {
        checkNotNull();
        _onConstraintRelease(address);
    }
    private static native void _onConstraintRelease(long address);

    /**
     * Center-of-mass shift callback
     * <p>
     * This function is called by the SDK when the CoM of one of the actors is moved. Since the
     * API specifies constraint positions relative to actors, and the constraint shader functions
     * are supplied with coordinates relative to bodies, some synchronization is usually required
     * when the application moves an object's center of mass.
     */
    public void onComShift(int actor) {
        checkNotNull();
        _onComShift(address, actor);
    }
    private static native void _onComShift(long address, int actor);

    /**
     * Origin shift callback
     * <p>
     * This function is called by the SDK when the scene origin gets shifted and allows to adjust
     * custom data which contains world space transforms.
     * <p>
     * <b>Note:</b> If the adjustments affect constraint shader data, it is necessary to call PxConstraint::markDirty()
     * to make sure that the data gets synced at the beginning of the next simulation step.
     * @param shift Translation vector the origin is shifted by.
     * @see PxScene#shiftOrigin
     */
    public void onOriginShift(PxVec3 shift) {
        checkNotNull();
        _onOriginShift(address, shift.getAddress());
    }
    private static native void _onOriginShift(long address, long shift);

    /**
     * Obtain a reference to a PxBase interface if the constraint has one.
     * <p>
     * If the constraint does not implement the PxBase interface, it should return NULL. 
     */
    public PxBase getSerializable() {
        checkNotNull();
        return PxBase.wrapPointer(_getSerializable(address));
    }
    private static native long _getSerializable(long address);

    /**
     * Obtain the shader function pointer used to prep rows for this constraint
     */
    public PxConstraintSolverPrep getPrep() {
        checkNotNull();
        return PxConstraintSolverPrep.wrapPointer(_getPrep(address));
    }
    private static native long _getPrep(long address);

    /**
     * Obtain the pointer to the constraint's constant data
     */
    public void getConstantBlock() {
        checkNotNull();
        _getConstantBlock(address);
    }
    private static native void _getConstantBlock(long address);

    /**
     * Let the connector know it has been connected to a constraint.
     */
    public void connectToConstraint(PxConstraint constraint) {
        checkNotNull();
        _connectToConstraint(address, constraint.getAddress());
    }
    private static native void _connectToConstraint(long address, long constraint);

}
