package physx.physics;

import physx.common.PxBase;
import physx.common.PxBounds3;
import physx.common.PxTransform;
import physx.common.PxVec3;

/**
 * A tree structure of bodies connected by joints that is treated as a unit by the dynamics solver. Parametrized in reduced (joint) coordinates.
 */
public class PxArticulationReducedCoordinate extends PxBase {

    protected PxArticulationReducedCoordinate() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxArticulationReducedCoordinate wrapPointer(long address) {
        return address != 0L ? new PxArticulationReducedCoordinate(address) : null;
    }
    
    public static PxArticulationReducedCoordinate arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxArticulationReducedCoordinate(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Returns the scene which this articulation belongs to.
     * @return Owner Scene. NULL if not part of a scene.
     * @see PxScene
     */
    public PxScene getScene() {
        checkNotNull();
        return PxScene.wrapPointer(_getScene(address));
    }
    private static native long _getScene(long address);

    /**
     * Sets the solver iteration counts for the articulation.
     * <p>
     * The solver iteration count determines how accurately contacts, drives, and limits are resolved.
     * Setting a higher position iteration count may therefore help in scenarios where the articulation
     * is subject to many constraints; for example, a manipulator articulation with drives and joint limits
     * that is grasping objects, or several such articulations interacting through contacts. Other situations
     * where higher position iterations may improve simulation fidelity are: large mass ratios within the
     * articulation or between the articulation and an object in contact with it; or strong drives in the
     * articulation being used to manipulate a light object.
     * <p>
     * If intersecting bodies are being depenetrated too violently, increase the number of velocity
     * iterations. More velocity iterations will drive the relative exit velocity of the intersecting
     * objects closer to the correct value given the restitution.
     * @param minPositionIters Number of position iterations the solver should perform for this articulation. <b>Range:</b> [1,255]. <b>Default:</b> 4.
     * <p>
     * <b>Note:</b> This call may not be made during simulation.
     */
    public void setSolverIterationCounts(int minPositionIters) {
        checkNotNull();
        _setSolverIterationCounts(address, minPositionIters);
    }
    private static native void _setSolverIterationCounts(long address, int minPositionIters);

    /**
     * Sets the solver iteration counts for the articulation.
     * <p>
     * The solver iteration count determines how accurately contacts, drives, and limits are resolved.
     * Setting a higher position iteration count may therefore help in scenarios where the articulation
     * is subject to many constraints; for example, a manipulator articulation with drives and joint limits
     * that is grasping objects, or several such articulations interacting through contacts. Other situations
     * where higher position iterations may improve simulation fidelity are: large mass ratios within the
     * articulation or between the articulation and an object in contact with it; or strong drives in the
     * articulation being used to manipulate a light object.
     * <p>
     * If intersecting bodies are being depenetrated too violently, increase the number of velocity
     * iterations. More velocity iterations will drive the relative exit velocity of the intersecting
     * objects closer to the correct value given the restitution.
     * @param minPositionIters Number of position iterations the solver should perform for this articulation. <b>Range:</b> [1,255]. <b>Default:</b> 4.
     * @param minVelocityIters Number of velocity iterations the solver should perform for this articulation. <b>Range:</b> [0,255]. <b>Default:</b> 1
     * <p>
     * <b>Note:</b> This call may not be made during simulation.
     */
    public void setSolverIterationCounts(int minPositionIters, int minVelocityIters) {
        checkNotNull();
        _setSolverIterationCounts(address, minPositionIters, minVelocityIters);
    }
    private static native void _setSolverIterationCounts(long address, int minPositionIters, int minVelocityIters);

    /**
     * Returns true if this articulation is sleeping.
     * <p>
     * When an actor does not move for a period of time, it is no longer simulated in order to reduce computational cost. This state
     * is called sleeping. However, because the object automatically wakes up when it is either touched by an awake object,
     * or a sleep-affecting property is changed by the user, the entire sleep mechanism should be transparent to the user.
     * <p>
     * An articulation can only go to sleep if all links are ready for sleeping. An articulation is guaranteed to be awake
     * if at least one of the following holds:
     * <p>
     * \li The wake counter of any link in the articulation is positive (see #setWakeCounter()).
     * \li The mass-normalized energy of any link in the articulation is above a threshold (see #setSleepThreshold()).
     * \li A non-zero force or torque has been applied to any joint or link.
     * <p>
     * If an articulation is sleeping, the following state is guaranteed:
     * <p>
     * \li The wake counter is zero.
     * \li The linear and angular velocity of all links is zero.
     * \li There is no force update pending.
     * <p>
     * When an articulation gets inserted into a scene, it will be considered asleep if all the points above hold, else it will
     * be treated as awake.
     * <p>
     * If an articulation is asleep after the call to #PxScene::fetchResults() returns, it is guaranteed that the poses of the
     * links were not changed. You can use this information to avoid updating the transforms of associated objects.
     * @return True if the articulation is sleeping.
     * <p>
     * <b>Note:</b> This call may only be made on articulations that are in a scene, and may not be made during simulation,
     * except in a split simulation in-between #PxScene::fetchCollision and #PxScene::advance.
     * @see #wakeUp
     * @see #putToSleep
     * @see #getSleepThreshold
     * @see #setSleepThreshold
     */
    public boolean isSleeping() {
        checkNotNull();
        return _isSleeping(address);
    }
    private static native boolean _isSleeping(long address);

    /**
     * Sets the mass-normalized energy threshold below which the articulation may go to sleep.
     * <p>
     * The articulation will sleep if the energy of each link is below this threshold.
     * @param threshold Energy below which the articulation may go to sleep. <b>Range:</b> [0, PX_MAX_F32)
     * <p>
     * <b>Note:</b> This call may not be made during simulation.
     * <p>
     * <b>Default:</b> 5e-5f * PxTolerancesScale::speed * PxTolerancesScale::speed;
     * @see #isSleeping
     * @see #getSleepThreshold
     * @see #wakeUp
     * @see #putToSleep
     */
    public void setSleepThreshold(float threshold) {
        checkNotNull();
        _setSleepThreshold(address, threshold);
    }
    private static native void _setSleepThreshold(long address, float threshold);

    /**
     * Returns the mass-normalized energy below which the articulation may go to sleep.
     * @return The energy threshold for sleeping.
     * @see #isSleeping
     * @see #wakeUp
     * @see #putToSleep
     * @see #setSleepThreshold
     */
    public float getSleepThreshold() {
        checkNotNull();
        return _getSleepThreshold(address);
    }
    private static native float _getSleepThreshold(long address);

    /**
     * Sets the mass-normalized kinetic energy threshold below which the articulation may participate in stabilization.
     * <p>
     * Articulations whose kinetic energy divided by their mass is above this threshold will not participate in stabilization.
     * <p>
     * This value has no effect if PxSceneFlag::eENABLE_STABILIZATION was not enabled on the PxSceneDesc.
     * <p>
     * <b>Default:</b> 5e-6f * PxTolerancesScale::speed * PxTolerancesScale::speed
     * @param threshold Energy below which the articulation may participate in stabilization. <b>Range:</b> [0,inf)
     * <p>
     * <b>Note:</b> This call may not be made during simulation.
     * @see #getStabilizationThreshold
     */
    public void setStabilizationThreshold(float threshold) {
        checkNotNull();
        _setStabilizationThreshold(address, threshold);
    }
    private static native void _setStabilizationThreshold(long address, float threshold);

    /**
     * Returns the mass-normalized kinetic energy below which the articulation may participate in stabilization.
     * <p>
     * Articulations whose kinetic energy divided by their mass is above this threshold will not participate in stabilization.
     * @return The energy threshold for participating in stabilization.
     * @see #setStabilizationThreshold
     */
    public float getStabilizationThreshold() {
        checkNotNull();
        return _getStabilizationThreshold(address);
    }
    private static native float _getStabilizationThreshold(long address);

    /**
     * Sets the wake counter for the articulation in seconds.
     * <p>
     * - The wake counter value specifies a time threshold used to determine whether an articulation may be put to sleep.
     * - The articulation will be put to sleep if all links have experienced a mass-normalised energy less than a threshold for at least
     * a threshold time, as specified by the wake counter.
     * - Passing in a positive value will wake up the articulation automatically.
     * <p>
     * <b>Default:</b> 0.4s (which corresponds to 20 frames for a time step of 0.02s)
     * @param wakeCounterValue Wake counter value in seconds. <b>Range:</b> [0, PX_MAX_F32)
     * <p>
     * <b>Note:</b> This call may not be made during simulation, except in a split simulation in-between #PxScene::fetchCollision and #PxScene::advance.
     * @see #isSleeping
     * @see #getWakeCounter
     */
    public void setWakeCounter(float wakeCounterValue) {
        checkNotNull();
        _setWakeCounter(address, wakeCounterValue);
    }
    private static native void _setWakeCounter(long address, float wakeCounterValue);

    /**
     * Returns the wake counter of the articulation in seconds.
     * @return The wake counter of the articulation in seconds.
     * <p>
     * <b>Note:</b> This call may not be made during simulation, except in a split simulation in-between #PxScene::fetchCollision and #PxScene::advance.
     * @see #isSleeping
     * @see #setWakeCounter
     */
    public float getWakeCounter() {
        checkNotNull();
        return _getWakeCounter(address);
    }
    private static native float _getWakeCounter(long address);

    /**
     * Wakes up the articulation if it is sleeping.
     * <p>
     * - The articulation will be woken up and might cause other touching objects to wake up as well during the next simulation step.
     * - This will set the wake counter of the articulation to the value specified in #PxSceneDesc::wakeCounterResetValue.
     * <p>
     * <b>Note:</b> This call may only be made on articulations that are in a scene, and may not be made during simulation,
     * except in a split simulation in-between #PxScene::fetchCollision and #PxScene::advance.
     * @see #isSleeping
     * @see #putToSleep
     */
    public void wakeUp() {
        checkNotNull();
        _wakeUp(address);
    }
    private static native void _wakeUp(long address);

    /**
     * Forces the articulation to sleep.
     * <p>
     * - The articulation will stay asleep during the next simulation step if not touched by another non-sleeping actor.
     * - This will set any applied force, the velocity, and the wake counter of all bodies in the articulation to zero.
     * <p>
     * <b>Note:</b> This call may not be made during simulation, and may only be made on articulations that are in a scene.
     * @see #isSleeping
     * @see #wakeUp
     */
    public void putToSleep() {
        checkNotNull();
        _putToSleep(address);
    }
    private static native void _putToSleep(long address);

    /**
     * @deprecated  The COM velocity limits will be removed in a future version without replacement.
     * <p>
     * Sets the limit on the magnitude of the linear velocity of the articulation's center of mass.
     * <p>
     * - The limit acts on the linear velocity of the entire articulation. The velocity is calculated from the total momentum
     * and the spatial inertia of the articulation.
     * - The limit only applies to floating-base articulations.
     * - A benefit of the COM velocity limit is that it is evenly applied to the whole articulation, which results in fewer visual
     * artifacts compared to link rigid-body damping or joint-velocity limits. However, these per-link or per-degree-of-freedom
     * limits may still help avoid numerical issues.
     * <p>
     * <b>Note:</b> This call may not be made during simulation.
     */
    @Deprecated
    public void setMaxCOMLinearVelocity(float maxLinerVelocity) {
        checkNotNull();
        _setMaxCOMLinearVelocity(address, maxLinerVelocity);
    }
    private static native void _setMaxCOMLinearVelocity(long address, float maxLinerVelocity);

    /**
     * @deprecated  The COM velocity limits will be removed in a future version without replacement.
     * Gets the limit on the magnitude of the linear velocity of the articulation's center of mass.
     * @return The maximal linear velocity magnitude.
     * @see #setMaxCOMLinearVelocity
     */
    @Deprecated
    public float getMaxCOMLinearVelocity() {
        checkNotNull();
        return _getMaxCOMLinearVelocity(address);
    }
    private static native float _getMaxCOMLinearVelocity(long address);

    /**
     * @deprecated  The COM velocity limits will be removed in a future version without replacement.
     * Sets the limit on the magnitude of the angular velocity at the articulation's center of mass.
     * <p>
     * - The limit acts on the angular velocity of the entire articulation. The velocity is calculated from the total momentum
     * and the spatial inertia of the articulation.
     * - The limit only applies to floating-base articulations.
     * - A benefit of the COM velocity limit is that it is evenly applied to the whole articulation, which results in fewer visual
     * artifacts compared to link rigid-body damping or joint-velocity limits. However, these per-link or per-degree-of-freedom
     * limits may still help avoid numerical issues.
     * <p>
     * <b>Note:</b> This call may not be made during simulation.
     * @param  maxAngularVelocity The maximal angular velocity magnitude. <b>Range:</b> [0, PX_MAX_F32); <b>Default:</b> 1e+6
     */
    @Deprecated
    public void setMaxCOMAngularVelocity(float maxAngularVelocity) {
        checkNotNull();
        _setMaxCOMAngularVelocity(address, maxAngularVelocity);
    }
    private static native void _setMaxCOMAngularVelocity(long address, float maxAngularVelocity);

    /**
     * @deprecated  The COM velocity limits will be removed in a future version without replacement.
     * Gets the limit on the magnitude of the angular velocity at the articulation's center of mass.
     * @return The maximal angular velocity magnitude.
     * @see #setMaxCOMAngularVelocity
     */
    @Deprecated
    public float getMaxCOMAngularVelocity() {
        checkNotNull();
        return _getMaxCOMAngularVelocity(address);
    }
    private static native float _getMaxCOMAngularVelocity(long address);

    /**
     * Adds a link to the articulation with default attribute values.
     * @param parent The parent link in the articulation. Must be NULL if (and only if) this is the root link.
     * @param pose The initial pose of the new link. Must be a valid transform.
     * @return The new link, or NULL if the link cannot be created.
     * <p>
     * <b>Note:</b> Creating a link is not allowed while the articulation is in a scene. In order to add a link,
     * remove and then re-add the articulation to the scene.
     * <p>
     * <b>Note:</b>  When the articulation is added to a scene, the root link adopts the specified pose. The pose of the 
     * root link is propagated through the ensemble of links from parent to child after accounting for each child's 
     * inbound joint frames and the joint positions set by  PxArticulationJointReducedCoordinate::setJointPosition().
     * As a consequence, the pose of each non-root link is automatically overwritten when adding the articulation to the scene.
     * @see PxArticulationLink
     */
    public PxArticulationLink createLink(PxArticulationLink parent, PxTransform pose) {
        checkNotNull();
        return PxArticulationLink.wrapPointer(_createLink(address, (parent != null ? parent.getAddress() : 0L), pose.getAddress()));
    }
    private static native long _createLink(long address, long parent, long pose);

    /**
     * Returns the number of links in the articulation.
     * @return The number of links.
     */
    public int getNbLinks() {
        checkNotNull();
        return _getNbLinks(address);
    }
    private static native int _getNbLinks(long address);

    /**
     * Returns the number of shapes in the articulation.
     * @return The number of shapes.
     */
    public int getNbShapes() {
        checkNotNull();
        return _getNbShapes(address);
    }
    private static native int _getNbShapes(long address);

    /**
     * Sets a name string for the articulation that can be retrieved with getName().
     * <p>
     * This is for debugging and is not used by the SDK. The string is not copied by the SDK,
     * only the pointer is stored.
     * @param name A pointer to a char buffer used to specify the name of the articulation.
     * @see #getName
     */
    public void setName(String name) {
        checkNotNull();
        _setName(address, name);
    }
    private static native void _setName(long address, String name);

    /**
     * Returns the name string set with setName().
     * @return Name string associated with the articulation.
     * @see #setName
     */
    public String getName() {
        checkNotNull();
        return _getName(address);
    }
    private static native String _getName(long address);

    /**
     * Returns the axis-aligned bounding box enclosing the articulation.
     * @return The articulation's bounding box.
     * <p>
     * <b>Note:</b> It is not allowed to use this method while the simulation is running, except in a split simulation
     * during #PxScene::collide() and up to #PxScene::advance(), and in PxContactModifyCallback or in contact report callbacks.
     * @see physx.common.PxBounds3
     */
    public PxBounds3 getWorldBounds() {
        checkNotNull();
        return PxBounds3.wrapPointer(_getWorldBounds(address));
    }
    private static native long _getWorldBounds(long address);

    /**
     * Returns the axis-aligned bounding box enclosing the articulation.
     * @param inflation Scale factor for computed world bounds. Box extents are multiplied by this value.
     * @return The articulation's bounding box.
     * <p>
     * <b>Note:</b> It is not allowed to use this method while the simulation is running, except in a split simulation
     * during #PxScene::collide() and up to #PxScene::advance(), and in PxContactModifyCallback or in contact report callbacks.
     * @see physx.common.PxBounds3
     */
    public PxBounds3 getWorldBounds(float inflation) {
        checkNotNull();
        return PxBounds3.wrapPointer(_getWorldBounds(address, inflation));
    }
    private static native long _getWorldBounds(long address, float inflation);

    /**
     * Returns the aggregate associated with the articulation.
     * @return The aggregate associated with the articulation or NULL if the articulation does not belong to an aggregate.
     * @see PxAggregate
     */
    public PxAggregate getAggregate() {
        checkNotNull();
        return PxAggregate.wrapPointer(_getAggregate(address));
    }
    private static native long _getAggregate(long address);

    /**
     * Sets flags on the articulation.
     * @param flags The articulation flags.
     * <p>
     * <b>Note:</b> This call may not be made during simulation.
     */
    public void setArticulationFlags(PxArticulationFlags flags) {
        checkNotNull();
        _setArticulationFlags(address, flags.getAddress());
    }
    private static native void _setArticulationFlags(long address, long flags);

    /**
     * Raises or clears a flag on the articulation.
     * @param flag The articulation flag.
     * @param value The value to set the flag to.
     * <p>
     * <b>Note:</b> This call may not be made during simulation.
     */
    public void setArticulationFlag(PxArticulationFlagEnum flag, boolean value) {
        checkNotNull();
        _setArticulationFlag(address, flag.value, value);
    }
    private static native void _setArticulationFlag(long address, int flag, boolean value);

    /**
     * Returns the articulation's flags.
     * @return The articulation's flags.
     */
    public PxArticulationFlags getArticulationFlags() {
        checkNotNull();
        return PxArticulationFlags.wrapPointer(_getArticulationFlags(address));
    }
    private static native long _getArticulationFlags(long address);

    /**
     * Returns the total number of joint degrees-of-freedom (DOFs) of the articulation.
     * <p>
     * - The six DOFs of the base of a floating-base articulation are not included in this count.
     * - Example: Both a fixed-base and a floating-base double-pendulum with two revolute joints will have getDofs() == 2.
     * - The return value is only valid for articulations that are in a scene.
     * @return The number of joint DOFs, or 0xFFFFFFFF if the articulation is not in a scene.
     */
    public int getDofs() {
        checkNotNull();
        return _getDofs(address);
    }
    private static native int _getDofs(long address);

    /**
     * Creates an articulation cache that can be used to read and write internal articulation data.
     * <p>
     * - When the structure of the articulation changes (e.g. adding a link) after the cache was created,
     * the cache needs to be released and recreated.
     * - Free the memory allocated for the cache by calling the release() method on the cache.
     * - Caches can only be created by articulations that are in a scene.
     * @return The cache, or NULL if the articulation is not in a scene.
     * @see #copyInternalStateToCache
     */
    public PxArticulationCache createCache() {
        checkNotNull();
        return PxArticulationCache.wrapPointer(_createCache(address));
    }
    private static native long _createCache(long address);

    /**
     * Returns the size of the articulation cache in bytes.
     * <p>
     * - The size does not include: the user-allocated memory for the coefficient matrix or lambda values;
     * the scratch-related memory/members; and the cache version. See comment in #PxArticulationCache.
     * - The return value is only valid for articulations that are in a scene.
     * @return The byte size of the cache, or 0xFFFFFFFF if the articulation is not in a scene.
     * @see PxArticulationCache
     */
    public int getCacheDataSize() {
        checkNotNull();
        return _getCacheDataSize(address);
    }
    private static native int _getCacheDataSize(long address);

    /**
     * Zeroes all data in the articulation cache, except user-provided and scratch memory, and cache version.
     * <p>
     * <b>Note:</b> This call may only be made on articulations that are in a scene.
     * @see PxArticulationCache
     */
    public void zeroCache(PxArticulationCache cache) {
        checkNotNull();
        _zeroCache(address, cache.getAddress());
    }
    private static native void _zeroCache(long address, long cache);

    /**
     * Applies the data in the cache to the articulation.
     * <p>
     * This call wakes the articulation if it is sleeping, and the autowake parameter is true (default) or:
     * - a nonzero joint velocity is applied or
     * - a nonzero joint force is applied or
     * - a nonzero root velocity is applied
     * @param cache The articulation data.
     * @param flags Indicate which data in the cache to apply to the articulation.
     * if the counter value is below the reset value.
     * <p>
     * <b>Note:</b> This call may only be made on articulations that are in a scene, and may not be made during simulation.
     */
    public void applyCache(PxArticulationCache cache, PxArticulationCacheFlags flags) {
        checkNotNull();
        _applyCache(address, cache.getAddress(), flags.getAddress());
    }
    private static native void _applyCache(long address, long cache, long flags);

    /**
     * Applies the data in the cache to the articulation.
     * <p>
     * This call wakes the articulation if it is sleeping, and the autowake parameter is true (default) or:
     * - a nonzero joint velocity is applied or
     * - a nonzero joint force is applied or
     * - a nonzero root velocity is applied
     * @param cache The articulation data.
     * @param flags Indicate which data in the cache to apply to the articulation.
     * @param autowake If true, the call wakes up the articulation and increases the wake counter to #PxSceneDesc::wakeCounterResetValue
     * if the counter value is below the reset value.
     * <p>
     * <b>Note:</b> This call may only be made on articulations that are in a scene, and may not be made during simulation.
     */
    public void applyCache(PxArticulationCache cache, PxArticulationCacheFlags flags, boolean autowake) {
        checkNotNull();
        _applyCache(address, cache.getAddress(), flags.getAddress(), autowake);
    }
    private static native void _applyCache(long address, long cache, long flags, boolean autowake);

    /**
     * Copies internal data of the articulation to the cache.
     * @param cache The articulation data.
     * @param flags Indicate which data to copy from the articulation to the cache.
     * <p>
     * <b>Note:</b> This call may only be made on articulations that are in a scene, and may not be made during simulation.
     * @see #applyCache
     */
    public void copyInternalStateToCache(PxArticulationCache cache, PxArticulationCacheFlags flags) {
        checkNotNull();
        _copyInternalStateToCache(address, cache.getAddress(), flags.getAddress());
    }
    private static native void _copyInternalStateToCache(long address, long cache, long flags);

    /**
     * Prepares common articulation data based on articulation pose for inverse dynamics calculations.
     * <p>
     * Usage:
     * -# Set articulation pose (joint positions and base transform) via articulation cache and applyCache().
     * -# Call commonInit.
     * -# Call inverse dynamics computation method.
     * <p>
     * <b>Note:</b> This call may only be made on articulations that are in a scene, and may not be made during simulation.
     * <p>
     * computeJointForce, computeDenseJacobian, computeMassMatrix
     */
    public void commonInit() {
        checkNotNull();
        _commonInit(address);
    }
    private static native void _commonInit(long address);

    /**
     * @deprecated  Please use computeGravityCompensation instead. It provides a more complete gravity compensation force for floating-base articulations.
     * <p>
     * Computes the joint DOF forces required to counteract gravitational forces for the given articulation pose.
     * <p>
     * - Inputs: Articulation pose (joint positions + base transform).
     * - Outputs: Joint forces to counteract gravity (in cache).
     * <p>
     * - The joint forces returned are determined purely by gravity for the articulation in the current joint and base pose, and joints at rest;
     * i.e. external forces, joint velocities, and joint accelerations are set to zero. Joint drives are also not considered in the computation.
     * - commonInit() must be called before the computation, and after setting the articulation pose via applyCache().
     * @param cache Out: PxArticulationCache::jointForce.
     * <p>
     * <b>Note:</b> This call may only be made on articulations that are in a scene, and may not be made during simulation.
     * @see #commonInit
     */
    @Deprecated
    public void computeGeneralizedGravityForce(PxArticulationCache cache) {
        checkNotNull();
        _computeGeneralizedGravityForce(address, cache.getAddress());
    }
    private static native void _computeGeneralizedGravityForce(long address, long cache);

    /**
     * @deprecated  Please use computeCoriolisCompensation instead. It provides a more complete Coriolis and centrifugal compensation force for floating-base articulations.
     * <p>
     * Computes the joint DOF forces required to counteract Coriolis and centrifugal forces for the given articulation state.
     * <p>
     * - Inputs: Articulation state (joint positions and velocities (in cache), and base transform and spatial velocity).
     * - Outputs: Joint forces to counteract Coriolis and centrifugal forces (in cache).
     * <p>
     * - The joint forces returned are determined purely by the articulation's state; i.e. external forces, gravity, and joint accelerations are set to zero.
     * Joint drives and potential damping terms, such as link angular or linear damping, or joint friction, are also not considered in the computation.
     * - Prior to the computation, update/set the base spatial velocity with PxArticulationCache::rootLinkData and applyCache().
     * - commonInit() must be called before the computation, and after setting the articulation pose via applyCache().
     * <p>
     * \param[in,out] cache In: PxArticulationCache::jointVelocity; Out: PxArticulationCache::jointForce.
     * <p>
     * <b>Note:</b> This call may only be made on articulations that are in a scene, and may not be made during simulation.
     * @see #commonInit
     */
    @Deprecated
    public void computeCoriolisAndCentrifugalForce(PxArticulationCache cache) {
        checkNotNull();
        _computeCoriolisAndCentrifugalForce(address, cache.getAddress());
    }
    private static native void _computeCoriolisAndCentrifugalForce(long address, long cache);

    /**
     * Computes the joint DOF forces required to counteract external spatial forces applied to articulation links.
     * <p>
     * - Inputs: External forces on links (in cache), articulation pose (joint positions + base transform).
     * - Outputs: Joint forces to counteract the external forces (in cache).
     * <p>
     * - Only the external spatial forces provided in the cache and the articulation pose are considered in the computation.
     * - The external spatial forces are with respect to the links' centers of mass, and not the actor's origin.
     * - commonInit() must be called before the computation, and after setting the articulation pose via applyCache().
     * <p>
     * \param[in,out] cache In: PxArticulationCache::externalForces; Out: PxArticulationCache::jointForce.
     * <p>
     * <b>Note:</b> This call may only be made on articulations that are in a scene, and may not be made during simulation.
     * @see #commonInit
     */
    public void computeGeneralizedExternalForce(PxArticulationCache cache) {
        checkNotNull();
        _computeGeneralizedExternalForce(address, cache.getAddress());
    }
    private static native void _computeGeneralizedExternalForce(long address, long cache);

    /**
     * Computes the joint accelerations for the given articulation state and joint forces.
     * <p>
     * - Inputs: Joint forces (in cache) and articulation state (joint positions and velocities (in cache), and base transform and spatial velocity).
     * - Outputs: Joint accelerations (in cache).
     * <p>
     * - The computation includes Coriolis terms and gravity. However, joint drives, external forces, and potential damping (link damping, friction) terms
     * are not considered in the computation.
     * - Prior to the computation, update/set the base spatial velocity with PxArticulationCache::rootLinkData and applyCache().
     * - commonInit() must be called before the computation, and after setting the articulation pose via applyCache().
     * <p>
     * \param[in,out] cache In: PxArticulationCache::jointForce and PxArticulationCache::jointVelocity; Out: PxArticulationCache::jointAcceleration.
     * <p>
     * <b>Note:</b> This call may only be made on articulations that are in a scene, and may not be made during simulation.
     * @see #commonInit
     */
    public void computeJointAcceleration(PxArticulationCache cache) {
        checkNotNull();
        _computeJointAcceleration(address, cache.getAddress());
    }
    private static native void _computeJointAcceleration(long address, long cache);

    /**
     * Computes the joint forces for the given articulation state and joint accelerations, not considering gravity.
     * <p>
     * - Inputs: Joint accelerations (in cache) and articulation state (joint positions and velocities (in cache), and base transform and spatial velocity).
     * - Outputs: Joint forces (in cache).
     * <p>
     * - The computation includes Coriolis terms. However, joint drives and potential damping terms are not considered in the computation
     * (for example, linear link damping or joint friction).
     * - Prior to the computation, update/set the base spatial velocity with PxArticulationCache::rootLinkData and applyCache().
     * - commonInit() must be called before the computation, and after setting the articulation pose via applyCache().
     * <p>
     * \param[in,out] cache In: PxArticulationCache::jointAcceleration and PxArticulationCache::jointVelocity; Out: PxArticulationCache::jointForce.
     * <p>
     * <b>Note:</b> This call may only be made on articulations that are in a scene, and may not be made during simulation.
     * @see #commonInit
     */
    public void computeJointForce(PxArticulationCache cache) {
        checkNotNull();
        _computeJointForce(address, cache.getAddress());
    }
    private static native void _computeJointForce(long address, long cache);

    /**
     * @deprecated  The API related to loop joints will be removed in a future version once a replacement is made available.
     * <p>
     * Computes the coefficient matrix for contact forces.
     * <p>
     * - The matrix dimension is getCoefficientMatrixSize() = getDofs() * getNbLoopJoints(), and the DOF (column) indexing follows the internal DOF order, see PxArticulationCache::jointVelocity.
     * - Each column in the matrix is the joint forces effected by a contact based on impulse strength 1.
     * - The user must allocate memory for PxArticulationCache::coefficientMatrix where the required size of the PxReal array is equal to getCoefficientMatrixSize().
     * - commonInit() must be called before the computation, and after setting the articulation pose via applyCache().
     * @param cache Out: PxArticulationCache::coefficientMatrix.
     * <p>
     * <b>Note:</b> This call may only be made on articulations that are in a scene, and may not be made during simulation.
     * @see #getCoefficientMatrixSize
     */
    @Deprecated
    public void computeCoefficientMatrix(PxArticulationCache cache) {
        checkNotNull();
        _computeCoefficientMatrix(address, cache.getAddress());
    }
    private static native void _computeCoefficientMatrix(long address, long cache);

    /**
     * @deprecated  Please use computeMassMatrix instead. It provides a more complete mass matrix for floating-base articulations.
     * <p>
     * Compute the joint-space inertia matrix that maps joint accelerations to joint forces: forces = M * accelerations.
     * <p>
     * - Inputs: Articulation pose (joint positions and base transform).
     * - Outputs: Mass matrix (in cache).
     * <p>
     * commonInit() must be called before the computation, and after setting the articulation pose via applyCache().
     * @param cache Out: PxArticulationCache::massMatrix.
     * <p>
     * <b>Note:</b> This call may only be made on articulations that are in a scene, and may not be made during simulation.
     * @see #commonInit
     */
    @Deprecated
    public void computeGeneralizedMassMatrix(PxArticulationCache cache) {
        checkNotNull();
        _computeGeneralizedMassMatrix(address, cache.getAddress());
    }
    private static native void _computeGeneralizedMassMatrix(long address, long cache);

    /**
     * @deprecated  The API related to loop joints will be removed in a future version once a replacement is made available.
     * <p>
     * Adds a loop joint to the articulation system for inverse dynamics.
     * @param joint The joint to add.
     * <p>
     * <b>Note:</b> This call may not be made during simulation.
     * @see physx.extensions.PxD6Joint
     */
    @Deprecated
    public void addLoopJoint(PxConstraint joint) {
        checkNotNull();
        _addLoopJoint(address, joint.getAddress());
    }
    private static native void _addLoopJoint(long address, long joint);

    /**
     * @deprecated  The API related to loop joints will be removed in a future version once a replacement is made available.
     * <p>
     * Removes a loop joint from the articulation for inverse dynamics.
     * <p>
     * <b>Note:</b> This call may not be made during simulation.
     * @param joint The joint to remove.
     */
    @Deprecated
    public void removeLoopJoint(PxConstraint joint) {
        checkNotNull();
        _removeLoopJoint(address, joint.getAddress());
    }
    private static native void _removeLoopJoint(long address, long joint);

    /**
     * @deprecated  The API related to loop joints will be removed in a future version once a replacement is made available.
     * <p>
     * Returns the number of loop joints in the articulation for inverse dynamics.
     * @return The number of loop joints.
     */
    @Deprecated
    public int getNbLoopJoints() {
        checkNotNull();
        return _getNbLoopJoints(address);
    }
    private static native int _getNbLoopJoints(long address);

    /**
     * @deprecated  The API related to loop joints will be removed in a future version once a replacement is made available.
     * <p>
     * Returns the required size of the coefficient matrix in the articulation.
     * @return Size of the coefficient matrix (equal to getDofs() * getNbLoopJoints()).
     * <p>
     * <b>Note:</b> This call may only be made on articulations that are in a scene.
     * @see #computeCoefficientMatrix
     */
    @Deprecated
    public int getCoefficientMatrixSize() {
        checkNotNull();
        return _getCoefficientMatrixSize(address);
    }
    private static native int _getCoefficientMatrixSize(long address);

    /**
     * Sets the root link transform in the world frame.
     * <p>
     * - Use updateKinematic() after all state updates to the articulation via non-cache API such as this method,
     * in order to update link states for the next simulation frame or querying.
     * @param pose The new root link transform.
     * each link will be reset to #PxSceneDesc::wakeCounterResetValue.
     * <p>
     * <b>Note:</b> This call may not be made during simulation.
     * <p>
     * <b>Note:</b> PxArticulationCache::rootLinkData similarly allows the root link pose to be updated and potentially offers better performance 
     * if the root link pose is to be updated along with other state variables. 
     * @see #applyCache
     */
    public void setRootGlobalPose(PxTransform pose) {
        checkNotNull();
        _setRootGlobalPose(address, pose.getAddress());
    }
    private static native void _setRootGlobalPose(long address, long pose);

    /**
     * Sets the root link transform in the world frame.
     * <p>
     * - Use updateKinematic() after all state updates to the articulation via non-cache API such as this method,
     * in order to update link states for the next simulation frame or querying.
     * @param pose The new root link transform.
     * @param autowake If true and the articulation is in a scene, the articulation will be woken up and the wake counter of 
     * each link will be reset to #PxSceneDesc::wakeCounterResetValue.
     * <p>
     * <b>Note:</b> This call may not be made during simulation.
     * <p>
     * <b>Note:</b> PxArticulationCache::rootLinkData similarly allows the root link pose to be updated and potentially offers better performance 
     * if the root link pose is to be updated along with other state variables. 
     * @see #applyCache
     */
    public void setRootGlobalPose(PxTransform pose, boolean autowake) {
        checkNotNull();
        _setRootGlobalPose(address, pose.getAddress(), autowake);
    }
    private static native void _setRootGlobalPose(long address, long pose, boolean autowake);

    /**
     * Returns the root link transform (world to actor frame).
     * @return The root link transform.
     * <p>
     * <b>Note:</b> This call is not allowed while the simulation is running except in a split simulation during #PxScene::collide() and up to #PxScene::advance(),
     * and in PxContactModifyCallback or in contact report callbacks.
     * <p>
     * <b>Note:</b> PxArticulationCache::rootLinkData similarly allows the root link pose to be queried and potentially offers better performance if the root
     * link pose is to be queried along with other state variables. 
     * @see #copyInternalStateToCache
     */
    public PxTransform getRootGlobalPose() {
        checkNotNull();
        return PxTransform.wrapPointer(_getRootGlobalPose(address));
    }
    private static native long _getRootGlobalPose(long address);

    /**
     * Sets the root link linear center-of-mass velocity.
     * <p>
     * - The linear velocity is with respect to the link's center of mass and not the actor frame origin.
     * - The articulation is woken up if the input velocity is nonzero (ignoring autowake) and the articulation is in a scene.
     * - Use updateKinematic() after all state updates to the articulation via non-cache API such as this method,
     * in order to update link states for the next simulation frame or querying.
     * @param linearVelocity The new root link center-of-mass linear velocity.
     * to #PxSceneDesc::wakeCounterResetValue if the counter value is below the reset value.
     * <p>
     * <b>Note:</b> This call may not be made during simulation, except in a split simulation in-between #PxScene::fetchCollision and #PxScene::advance.
     * <p>
     * <b>Note:</b> PxArticulationCache::rootLinkData similarly allows the root link linear velocity to be updated and potentially offers better performance 
     * if the root link linear velocity is to be updated along with other state variables. 
     * @see #applyCache
     */
    public void setRootLinearVelocity(PxVec3 linearVelocity) {
        checkNotNull();
        _setRootLinearVelocity(address, linearVelocity.getAddress());
    }
    private static native void _setRootLinearVelocity(long address, long linearVelocity);

    /**
     * Sets the root link linear center-of-mass velocity.
     * <p>
     * - The linear velocity is with respect to the link's center of mass and not the actor frame origin.
     * - The articulation is woken up if the input velocity is nonzero (ignoring autowake) and the articulation is in a scene.
     * - Use updateKinematic() after all state updates to the articulation via non-cache API such as this method,
     * in order to update link states for the next simulation frame or querying.
     * @param linearVelocity The new root link center-of-mass linear velocity.
     * @param autowake If true and the articulation is in a scene, the call wakes up the articulation and increases the wake counter
     * to #PxSceneDesc::wakeCounterResetValue if the counter value is below the reset value.
     * <p>
     * <b>Note:</b> This call may not be made during simulation, except in a split simulation in-between #PxScene::fetchCollision and #PxScene::advance.
     * <p>
     * <b>Note:</b> PxArticulationCache::rootLinkData similarly allows the root link linear velocity to be updated and potentially offers better performance 
     * if the root link linear velocity is to be updated along with other state variables. 
     * @see #applyCache
     */
    public void setRootLinearVelocity(PxVec3 linearVelocity, boolean autowake) {
        checkNotNull();
        _setRootLinearVelocity(address, linearVelocity.getAddress(), autowake);
    }
    private static native void _setRootLinearVelocity(long address, long linearVelocity, boolean autowake);

    /**
     * Gets the root link center-of-mass linear velocity.
     * <p>
     * - The linear velocity is with respect to the link's center of mass and not the actor frame origin.
     * @return The root link center-of-mass linear velocity.
     * <p>
     * <b>Note:</b> This call is not allowed while the simulation is running except in a split simulation during #PxScene::collide() and up to #PxScene::advance(),
     * and in PxContactModifyCallback or in contact report callbacks.
     * <p>
     * <b>Note:</b> PxArticulationCache::rootLinkData similarly allows the root link linear velocity to be queried and potentially offers better performance 
     * if the root link linear velocity is to be queried along with other state variables. 
     * @see #applyCache
     */
    public PxVec3 getRootLinearVelocity() {
        checkNotNull();
        return PxVec3.wrapPointer(_getRootLinearVelocity(address));
    }
    private static native long _getRootLinearVelocity(long address);

    /**
     * Sets the root link angular velocity.
     * <p>
     * - The articulation is woken up if the input velocity is nonzero (ignoring autowake) and the articulation is in a scene.
     * - Use updateKinematic() after all state updates to the articulation via non-cache API such as this method,
     * in order to update link states for the next simulation frame or querying.
     * @param angularVelocity The new root link angular velocity.
     * to #PxSceneDesc::wakeCounterResetValue if the counter value is below the reset value.
     * <p>
     * <b>Note:</b> This call may not be made during simulation, except in a split simulation in-between #PxScene::fetchCollision and #PxScene::advance.
     * <p>
     * <b>Note:</b> PxArticulationCache::rootLinkData similarly allows the root link angular velocity to be updated and potentially offers better performance 
     * if the root link angular velocity is to be updated along with other state variables. 
     * @see #applyCache
     */
    public void setRootAngularVelocity(PxVec3 angularVelocity) {
        checkNotNull();
        _setRootAngularVelocity(address, angularVelocity.getAddress());
    }
    private static native void _setRootAngularVelocity(long address, long angularVelocity);

    /**
     * Sets the root link angular velocity.
     * <p>
     * - The articulation is woken up if the input velocity is nonzero (ignoring autowake) and the articulation is in a scene.
     * - Use updateKinematic() after all state updates to the articulation via non-cache API such as this method,
     * in order to update link states for the next simulation frame or querying.
     * @param angularVelocity The new root link angular velocity.
     * @param autowake If true and the articulation is in a scene, the call wakes up the articulation and increases the wake counter
     * to #PxSceneDesc::wakeCounterResetValue if the counter value is below the reset value.
     * <p>
     * <b>Note:</b> This call may not be made during simulation, except in a split simulation in-between #PxScene::fetchCollision and #PxScene::advance.
     * <p>
     * <b>Note:</b> PxArticulationCache::rootLinkData similarly allows the root link angular velocity to be updated and potentially offers better performance 
     * if the root link angular velocity is to be updated along with other state variables. 
     * @see #applyCache
     */
    public void setRootAngularVelocity(PxVec3 angularVelocity, boolean autowake) {
        checkNotNull();
        _setRootAngularVelocity(address, angularVelocity.getAddress(), autowake);
    }
    private static native void _setRootAngularVelocity(long address, long angularVelocity, boolean autowake);

    /**
     * Gets the root link angular velocity.
     * @return The root link angular velocity.
     * <p>
     * <b>Note:</b> This call is not allowed while the simulation is running except in a split simulation during #PxScene::collide() and up to #PxScene::advance(),
     * and in PxContactModifyCallback or in contact report callbacks.
     * <p>
     * <b>Note:</b> PxArticulationCache::rootLinkData similarly allows the root link angular velocity to be queried and potentially offers better performance 
     * if the root link angular velocity is to be queried along with other state variables. 
     * @see #applyCache
     */
    public PxVec3 getRootAngularVelocity() {
        checkNotNull();
        return PxVec3.wrapPointer(_getRootAngularVelocity(address));
    }
    private static native long _getRootAngularVelocity(long address);

    /**
     * Returns the (classical) link acceleration in world space for the given low-level link index.
     * <p>
     * - The returned acceleration is not a spatial, but a classical, i.e. body-fixed acceleration (https://en.wikipedia.org/wiki/Spatial_acceleration).
     * - The (linear) acceleration is with respect to the link's center of mass and not the actor frame origin.
     * @param linkId The low-level link index, see PxArticulationLink::getLinkIndex.
     * @return The link's center-of-mass classical acceleration, or 0 if the call is made before the articulation participated in a first simulation step.
     * <p>
     * <b>Note:</b> This call may only be made on articulations that are in a scene. It is not allowed to use this method while the simulation
     * is running.  The exceptions to this rule are a split simulation during #PxScene::collide() and up to #PxScene::advance(); 
     * in PxContactModifyCallback; and in contact report callbacks.
     */
    public PxSpatialVelocity getLinkAcceleration(int linkId) {
        checkNotNull();
        return PxSpatialVelocity.wrapPointer(_getLinkAcceleration(address, linkId));
    }
    private static native long _getLinkAcceleration(long address, int linkId);

    /**
     * Returns the GPU articulation index.
     * @return The GPU index, or 0xFFFFFFFF if the articulation is not in a scene or PxSceneFlag::eENABLE_DIRECT_GPU_API is not set.
     * @deprecated  use getGpuIndex() instead.
     */
    @Deprecated
    public int getGpuArticulationIndex() {
        checkNotNull();
        return _getGpuArticulationIndex(address);
    }
    private static native int _getGpuArticulationIndex(long address);

    /**
     * Creates a spatial tendon to attach to the articulation with default attribute values.
     * @return The new spatial tendon.
     * <p>
     * <b>Note:</b> Creating a spatial tendon is not allowed while the articulation is in a scene. In order to
     * add the tendon, remove and then re-add the articulation to the scene.
     * <p>
     * <b>Note:</b> The spatial tendon is released with PxArticulationReducedCoordinate::release()
     * @see PxArticulationSpatialTendon
     */
    public PxArticulationSpatialTendon createSpatialTendon() {
        checkNotNull();
        return PxArticulationSpatialTendon.wrapPointer(_createSpatialTendon(address));
    }
    private static native long _createSpatialTendon(long address);

    /**
     * Creates a fixed tendon to attach to the articulation with default attribute values.
     * @return The new fixed tendon.
     * <p>
     * <b>Note:</b> Creating a fixed tendon is not allowed while the articulation is in a scene. In order to
     * add the tendon, remove and then re-add the articulation to the scene.
     * <p>
     * <b>Note:</b> The fixed tendon is released with PxArticulationReducedCoordinate::release()
     * @see PxArticulationFixedTendon
     */
    public PxArticulationFixedTendon createFixedTendon() {
        checkNotNull();
        return PxArticulationFixedTendon.wrapPointer(_createFixedTendon(address));
    }
    private static native long _createFixedTendon(long address);

    /**
     * Returns the number of spatial tendons in the articulation.
     * @return The number of tendons.
     */
    public int getNbSpatialTendons() {
        checkNotNull();
        return _getNbSpatialTendons(address);
    }
    private static native int _getNbSpatialTendons(long address);

    /**
     * Returns the number of fixed tendons in the articulation.
     * @return The number of tendons.
     */
    public int getNbFixedTendons() {
        checkNotNull();
        return _getNbFixedTendons(address);
    }
    private static native int _getNbFixedTendons(long address);

    /**
     * Update link velocities and/or positions in the articulation.
     * <p>
     *    An alternative that potentially offers better performance is to use the PxArticulationCache API.
     * <p>
     * If the application updates the root state (position and velocity) or joint state via any combination of
     * the non-cache API calls
     * <p>
     * - setRootGlobalPose(), setRootLinearVelocity(), setRootAngularVelocity()
     * - PxArticulationJointReducedCoordinate::setJointPosition(), PxArticulationJointReducedCoordinate::setJointVelocity()
     * <p>
     * the application needs to call this method after the state setting in order to update the link states for
     * the next simulation frame or querying.
     * <p>
     * Use
     * - PxArticulationKinematicFlag::ePOSITION after any changes to the articulation root or joint positions using non-cache API calls. Updates links' positions and velocities.
     * - PxArticulationKinematicFlag::eVELOCITY after velocity-only changes to the articulation root or joints using non-cache API calls. Updates links' velocities only.
     * <p>
     * <b>Note:</b> This call may only be made on articulations that are in a scene, and may not be made during simulation.
     * @see #applyCache
     */
    public void updateKinematic(PxArticulationKinematicFlags flags) {
        checkNotNull();
        _updateKinematic(address, flags.getAddress());
    }
    private static native void _updateKinematic(long address, long flags);

}
