package physx.geometry;

import physx.NativeObject;
import physx.common.PxBounds3;
import physx.common.PxRefCounted;
import physx.common.PxVec3;
import physx.support.PxU32ConstPtr;

/**
 * A tetramedron mesh, also called a 'tetrahedron soup'.
 * <p>
 * It is represented as an indexed tetrahedron list. There are no restrictions on the
 * tetrahedron data.
 * <p>
 * To avoid duplicating data when you have several instances of a particular
 * mesh positioned differently, you do not use this class to represent a
 * mesh object directly. Instead, you create an instance of this mesh via
 * the PxTetrahedronMeshGeometry and PxShape classes.
 * <p>
 * <h3>Creation</h3>
 * <p>
 * To create an instance of this class call PxPhysics::createTetrahedronMesh(),
 * and release() to delete it. This is only possible
 * once you have released all of its PxShape instances.
 * <p>
 * <h3>Visualizations:</h3>
 * \li #PxVisualizationParameter::eCOLLISION_AABBS
 * \li #PxVisualizationParameter::eCOLLISION_SHAPES
 * \li #PxVisualizationParameter::eCOLLISION_AXES
 * \li #PxVisualizationParameter::eCOLLISION_FNORMALS
 * \li #PxVisualizationParameter::eCOLLISION_EDGES
 * @see PxTetrahedronMeshDesc
 * @see PxTetrahedronMeshGeometry
 * @see physx.physics.PxShape
 */
public class PxTetrahedronMesh extends PxRefCounted {

    protected PxTetrahedronMesh() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxTetrahedronMesh wrapPointer(long address) {
        return address != 0L ? new PxTetrahedronMesh(address) : null;
    }
    
    public static PxTetrahedronMesh arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxTetrahedronMesh(long address) {
        super(address);
    }

    // Functions

    /**
     * Returns the number of vertices.
     * @return number of vertices
     * @see #getVertices
     */
    public int getNbVertices() {
        checkNotNull();
        return _getNbVertices(address);
    }
    private static native int _getNbVertices(long address);

    /**
     * Returns the vertices
     * @return array of vertices
     * @see #getNbVertices
     */
    public PxVec3 getVertices() {
        checkNotNull();
        return PxVec3.wrapPointer(_getVertices(address));
    }
    private static native long _getVertices(long address);

    /**
     * Returns the number of tetrahedrons.
     * @return number of tetrahedrons
     * @see #getTetrahedrons
     */
    public int getNbTetrahedrons() {
        checkNotNull();
        return _getNbTetrahedrons(address);
    }
    private static native int _getNbTetrahedrons(long address);

    /**
     * Returns the tetrahedron indices.
     * <p>
     * The indices can be 16 or 32bit depending on the number of tetrahedrons in the mesh.
     * Call getTetrahedronMeshFlags() to know if the indices are 16 or 32 bits.
     * <p>
     * The number of indices is the number of tetrahedrons * 4.
     * @return array of tetrahedrons
     * @see #getTetrahedronMeshFlags
     * @see #getTetrahedraRemap
     */
    public NativeObject getTetrahedrons() {
        checkNotNull();
        return NativeObject.wrapPointer(_getTetrahedrons(address));
    }
    private static native long _getTetrahedrons(long address);

    /**
     * Reads the PxTetrahedronMesh flags.
     * <p>
     * See the list of flags #PxTetrahedronMeshFlags
     * @return The values of the PxTetrahedronMesh flags.
     */
    public PxTetrahedronMeshFlags getTetrahedronMeshFlags() {
        checkNotNull();
        return PxTetrahedronMeshFlags.wrapPointer(_getTetrahedronMeshFlags(address));
    }
    private static native long _getTetrahedronMeshFlags(long address);

    /**
     * Returns the tetrahedra remapping table.
     * <p>
     * The tetrahedra are internally sorted according to various criteria. Hence the internal tetrahedron order
     * does not always match the original (user-defined) order. The remapping table helps finding the old
     * indices knowing the new ones:
     * <p>
     *  remapTable[ internalTetrahedronIndex ] = originalTetrahedronIndex
     * @return the remapping table (or NULL if 'PxCookingParams::suppressTriangleMeshRemapTable' has been used)
     * @see #getTetrahedrons
     */
    public PxU32ConstPtr getTetrahedraRemap() {
        checkNotNull();
        return PxU32ConstPtr.wrapPointer(_getTetrahedraRemap(address));
    }
    private static native long _getTetrahedraRemap(long address);

    /**
     * Returns the local-space (vertex space) AABB from the tetrahedron mesh.
     * @return local-space bounds
     */
    public PxBounds3 getLocalBounds() {
        checkNotNull();
        return PxBounds3.wrapPointer(_getLocalBounds(address));
    }
    private static native long _getLocalBounds(long address);

}
