package physx.geometry;

import physx.NativeObject;
import physx.common.PxBoundedData;
import physx.support.PxArray_PxU32;
import physx.support.PxArray_PxVec3;
import physx.support.PxI32ConstPtr;
import physx.support.PxU32ConstPtr;

/**
 * Provides functionality to create a tetrahedral mesh from a triangle mesh.
 */
public class PxTetMaker extends NativeObject {

    protected PxTetMaker() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxTetMaker wrapPointer(long address) {
        return address != 0L ? new PxTetMaker(address) : null;
    }
    
    public static PxTetMaker arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxTetMaker(long address) {
        super(address);
    }

    // Functions

    /**
     * Create conforming tetrahedron mesh using TetMaker
     * @param triangleMesh The description of the triangle mesh including vertices and indices
     * @param outVertices The vertices to store the conforming tetrahedral mesh
     * @param outTetIndices The indices to store the conforming tetrahedral mesh
     * @param validate If set to true the input triangle mesh will get analyzed to find possible deficiencies
     * @param volumeThreshold Tetrahedra with a volume smaller than the specified threshold will be removed from the mesh
     * @return True if success
     */
    public static boolean createConformingTetrahedronMesh(PxSimpleTriangleMesh triangleMesh, PxArray_PxVec3 outVertices, PxArray_PxU32 outTetIndices, boolean validate, float volumeThreshold) {
        return _createConformingTetrahedronMesh(triangleMesh.getAddress(), outVertices.getAddress(), outTetIndices.getAddress(), validate, volumeThreshold);
    }
    private static native boolean _createConformingTetrahedronMesh(long triangleMesh, long outVertices, long outTetIndices, boolean validate, float volumeThreshold);

    /**
     * Create voxel-based tetrahedron mesh using TetMaker
     * @param tetMesh The description of the tetrahedral mesh including vertices and indices
     * @param numVoxelsAlongLongestBoundingBoxAxis The number of voxels along the longest bounding box axis
     * @param outVertices The vertices to store the voxel-based tetrahedral mesh
     * @param outTetIndices The indices to store the voxel-based tetrahedral mesh
     * @return True if success
     */
    public static boolean createVoxelTetrahedronMesh(PxTetrahedronMeshDesc tetMesh, int numVoxelsAlongLongestBoundingBoxAxis, PxArray_PxVec3 outVertices, PxArray_PxU32 outTetIndices) {
        return _createVoxelTetrahedronMesh(tetMesh.getAddress(), numVoxelsAlongLongestBoundingBoxAxis, outVertices.getAddress(), outTetIndices.getAddress());
    }
    private static native boolean _createVoxelTetrahedronMesh(long tetMesh, int numVoxelsAlongLongestBoundingBoxAxis, long outVertices, long outTetIndices);

    /**
     * Create voxel-based tetrahedron mesh using TetMaker
     * @param tetMesh The description of the tetrahedral mesh including vertices and indices
     * @param voxelEdgeLength The edge length of a voxel.Can be adjusted slightly such that a multiple of it matches the input points' bounding box size
     * @param outVertices The vertices to store the voxel-based tetrahedral mesh
     * @param outTetIndices The indices to store the voxel-based tetrahedral mesh 
     * @return True if success
     */
    public static boolean createVoxelTetrahedronMeshFromEdgeLength(PxTetrahedronMeshDesc tetMesh, float voxelEdgeLength, PxArray_PxVec3 outVertices, PxArray_PxU32 outTetIndices) {
        return _createVoxelTetrahedronMeshFromEdgeLength(tetMesh.getAddress(), voxelEdgeLength, outVertices.getAddress(), outTetIndices.getAddress());
    }
    private static native boolean _createVoxelTetrahedronMeshFromEdgeLength(long tetMesh, float voxelEdgeLength, long outVertices, long outTetIndices);

    /**
     * Analyzes the triangle mesh to get a report about deficiencies. Some deficiencies can be handled by the tetmesher, others cannot.
     * @param triangleMesh The description of the triangle mesh including vertices and indices
     * @param minVolumeThreshold Minimum volume the mesh must have such that no volume warning is generated
     * @param minTriangleAngleRadians Minimum angle allowed for triangles such that no angle warning is generated
     * @return Flags that describe the triangle mesh's deficiencies
     */
    public static PxTriangleMeshAnalysisResults validateTriangleMesh(PxSimpleTriangleMesh triangleMesh, float minVolumeThreshold, float minTriangleAngleRadians) {
        return PxTriangleMeshAnalysisResults.wrapPointer(_validateTriangleMesh(triangleMesh.getAddress(), minVolumeThreshold, minTriangleAngleRadians));
    }
    private static native long _validateTriangleMesh(long triangleMesh, float minVolumeThreshold, float minTriangleAngleRadians);

    /**
     * Analyzes the tetrahedron mesh to get a report about deficiencies. Some deficiencies can be handled by the deformable volume cooker, others cannot.
     * @param points The mesh's points
     * @param tetrahedra The mesh's tetrahedra (index buffer)
     * @param minTetVolumeThreshold Minimum volume every tetrahedron in the mesh must have such that no volume warning is generated
     * @return Flags that describe the tetrahedron mesh's deficiencies
     */
    public static PxTetrahedronMeshAnalysisResults validateTetrahedronMesh(PxBoundedData points, PxBoundedData tetrahedra, float minTetVolumeThreshold) {
        return PxTetrahedronMeshAnalysisResults.wrapPointer(_validateTetrahedronMesh(points.getAddress(), tetrahedra.getAddress(), minTetVolumeThreshold));
    }
    private static native long _validateTetrahedronMesh(long points, long tetrahedra, float minTetVolumeThreshold);

    /**
     * Simplifies (decimates) a triangle mesh using quadric simplification.
     * @param inputVertices The vertices of the input triangle mesh
     * @param inputIndices The indices of the input triangle mesh of the form  (id0, id1, id2),  (id0, id1, id2), ..
     * @param targetTriangleCount Desired number of triangles in the output mesh
     * @param maximalEdgeLength Edges below this length will not be collapsed. A value of zero means there is no limit.
     * @param outputVertices The vertices of the output (decimated) triangle mesh
     * @param outputIndices The indices of the output (decimated) triangle mesh of the form  (id0, id1, id2),  (id0, id1, id2), ..
     */
    public static void simplifyTriangleMesh(PxArray_PxVec3 inputVertices, PxArray_PxU32 inputIndices, int targetTriangleCount, float maximalEdgeLength, PxArray_PxVec3 outputVertices, PxArray_PxU32 outputIndices) {
        _simplifyTriangleMesh(inputVertices.getAddress(), inputIndices.getAddress(), targetTriangleCount, maximalEdgeLength, outputVertices.getAddress(), outputIndices.getAddress());
    }
    private static native void _simplifyTriangleMesh(long inputVertices, long inputIndices, int targetTriangleCount, float maximalEdgeLength, long outputVertices, long outputIndices);

    /**
     * Simplifies (decimates) a triangle mesh using quadric simplification.
     * @param inputVertices The vertices of the input triangle mesh
     * @param inputIndices The indices of the input triangle mesh of the form  (id0, id1, id2),  (id0, id1, id2), ..
     * @param targetTriangleCount Desired number of triangles in the output mesh
     * @param maximalEdgeLength Edges below this length will not be collapsed. A value of zero means there is no limit.
     * @param outputVertices The vertices of the output (decimated) triangle mesh
     * @param outputIndices The indices of the output (decimated) triangle mesh of the form  (id0, id1, id2),  (id0, id1, id2), ..
     * @param vertexMap Optional parameter which returns the mapping from input to output vertices. Note that multiple input vertices are typically collapsed into the same output vertex.
     */
    public static void simplifyTriangleMesh(PxArray_PxVec3 inputVertices, PxArray_PxU32 inputIndices, int targetTriangleCount, float maximalEdgeLength, PxArray_PxVec3 outputVertices, PxArray_PxU32 outputIndices, PxArray_PxU32 vertexMap) {
        _simplifyTriangleMesh(inputVertices.getAddress(), inputIndices.getAddress(), targetTriangleCount, maximalEdgeLength, outputVertices.getAddress(), outputIndices.getAddress(), vertexMap.getAddress());
    }
    private static native void _simplifyTriangleMesh(long inputVertices, long inputIndices, int targetTriangleCount, float maximalEdgeLength, long outputVertices, long outputIndices, long vertexMap);

    /**
     * Simplifies (decimates) a triangle mesh using quadric simplification.
     * @param inputVertices The vertices of the input triangle mesh
     * @param inputIndices The indices of the input triangle mesh of the form  (id0, id1, id2),  (id0, id1, id2), ..
     * @param targetTriangleCount Desired number of triangles in the output mesh
     * @param maximalEdgeLength Edges below this length will not be collapsed. A value of zero means there is no limit.
     * @param outputVertices The vertices of the output (decimated) triangle mesh
     * @param outputIndices The indices of the output (decimated) triangle mesh of the form  (id0, id1, id2),  (id0, id1, id2), ..
     * @param vertexMap Optional parameter which returns the mapping from input to output vertices. Note that multiple input vertices are typically collapsed into the same output vertex.
     * @param edgeLengthCostWeight Factor to scale influence of edge length when prioritizing edge collapses. Has no effect if set to zero.
     */
    public static void simplifyTriangleMesh(PxArray_PxVec3 inputVertices, PxArray_PxU32 inputIndices, int targetTriangleCount, float maximalEdgeLength, PxArray_PxVec3 outputVertices, PxArray_PxU32 outputIndices, PxArray_PxU32 vertexMap, float edgeLengthCostWeight) {
        _simplifyTriangleMesh(inputVertices.getAddress(), inputIndices.getAddress(), targetTriangleCount, maximalEdgeLength, outputVertices.getAddress(), outputIndices.getAddress(), vertexMap.getAddress(), edgeLengthCostWeight);
    }
    private static native void _simplifyTriangleMesh(long inputVertices, long inputIndices, int targetTriangleCount, float maximalEdgeLength, long outputVertices, long outputIndices, long vertexMap, float edgeLengthCostWeight);

    /**
     * Simplifies (decimates) a triangle mesh using quadric simplification.
     * @param inputVertices The vertices of the input triangle mesh
     * @param inputIndices The indices of the input triangle mesh of the form  (id0, id1, id2),  (id0, id1, id2), ..
     * @param targetTriangleCount Desired number of triangles in the output mesh
     * @param maximalEdgeLength Edges below this length will not be collapsed. A value of zero means there is no limit.
     * @param outputVertices The vertices of the output (decimated) triangle mesh
     * @param outputIndices The indices of the output (decimated) triangle mesh of the form  (id0, id1, id2),  (id0, id1, id2), ..
     * @param vertexMap Optional parameter which returns the mapping from input to output vertices. Note that multiple input vertices are typically collapsed into the same output vertex.
     * @param edgeLengthCostWeight Factor to scale influence of edge length when prioritizing edge collapses. Has no effect if set to zero.
     * @param flatnessDetectionThreshold Threshold used to detect edges in flat regions and to improve the placement of the collapsed point. If set to a large value it will have no effect.
     */
    public static void simplifyTriangleMesh(PxArray_PxVec3 inputVertices, PxArray_PxU32 inputIndices, int targetTriangleCount, float maximalEdgeLength, PxArray_PxVec3 outputVertices, PxArray_PxU32 outputIndices, PxArray_PxU32 vertexMap, float edgeLengthCostWeight, float flatnessDetectionThreshold) {
        _simplifyTriangleMesh(inputVertices.getAddress(), inputIndices.getAddress(), targetTriangleCount, maximalEdgeLength, outputVertices.getAddress(), outputIndices.getAddress(), vertexMap.getAddress(), edgeLengthCostWeight, flatnessDetectionThreshold);
    }
    private static native void _simplifyTriangleMesh(long inputVertices, long inputIndices, int targetTriangleCount, float maximalEdgeLength, long outputVertices, long outputIndices, long vertexMap, float edgeLengthCostWeight, float flatnessDetectionThreshold);

    /**
     * Simplifies (decimates) a triangle mesh using quadric simplification.
     * @param inputVertices The vertices of the input triangle mesh
     * @param inputIndices The indices of the input triangle mesh of the form  (id0, id1, id2),  (id0, id1, id2), ..
     * @param targetTriangleCount Desired number of triangles in the output mesh
     * @param maximalEdgeLength Edges below this length will not be collapsed. A value of zero means there is no limit.
     * @param outputVertices The vertices of the output (decimated) triangle mesh
     * @param outputIndices The indices of the output (decimated) triangle mesh of the form  (id0, id1, id2),  (id0, id1, id2), ..
     * @param vertexMap Optional parameter which returns the mapping from input to output vertices. Note that multiple input vertices are typically collapsed into the same output vertex.
     * @param edgeLengthCostWeight Factor to scale influence of edge length when prioritizing edge collapses. Has no effect if set to zero.
     * @param flatnessDetectionThreshold Threshold used to detect edges in flat regions and to improve the placement of the collapsed point. If set to a large value it will have no effect.
     * @param projectSimplifiedPointsOnInputMeshSurface If set to true, the simplified points will lie exactly on the original surface. 
     */
    public static void simplifyTriangleMesh(PxArray_PxVec3 inputVertices, PxArray_PxU32 inputIndices, int targetTriangleCount, float maximalEdgeLength, PxArray_PxVec3 outputVertices, PxArray_PxU32 outputIndices, PxArray_PxU32 vertexMap, float edgeLengthCostWeight, float flatnessDetectionThreshold, boolean projectSimplifiedPointsOnInputMeshSurface) {
        _simplifyTriangleMesh(inputVertices.getAddress(), inputIndices.getAddress(), targetTriangleCount, maximalEdgeLength, outputVertices.getAddress(), outputIndices.getAddress(), vertexMap.getAddress(), edgeLengthCostWeight, flatnessDetectionThreshold, projectSimplifiedPointsOnInputMeshSurface);
    }
    private static native void _simplifyTriangleMesh(long inputVertices, long inputIndices, int targetTriangleCount, float maximalEdgeLength, long outputVertices, long outputIndices, long vertexMap, float edgeLengthCostWeight, float flatnessDetectionThreshold, boolean projectSimplifiedPointsOnInputMeshSurface);

    /**
     * Simplifies (decimates) a triangle mesh using quadric simplification.
     * @param inputVertices The vertices of the input triangle mesh
     * @param inputIndices The indices of the input triangle mesh of the form  (id0, id1, id2),  (id0, id1, id2), ..
     * @param targetTriangleCount Desired number of triangles in the output mesh
     * @param maximalEdgeLength Edges below this length will not be collapsed. A value of zero means there is no limit.
     * @param outputVertices The vertices of the output (decimated) triangle mesh
     * @param outputIndices The indices of the output (decimated) triangle mesh of the form  (id0, id1, id2),  (id0, id1, id2), ..
     * @param vertexMap Optional parameter which returns the mapping from input to output vertices. Note that multiple input vertices are typically collapsed into the same output vertex.
     * @param edgeLengthCostWeight Factor to scale influence of edge length when prioritizing edge collapses. Has no effect if set to zero.
     * @param flatnessDetectionThreshold Threshold used to detect edges in flat regions and to improve the placement of the collapsed point. If set to a large value it will have no effect.
     * @param projectSimplifiedPointsOnInputMeshSurface If set to true, the simplified points will lie exactly on the original surface. 
     * @param outputVertexToInputTriangle Optional indices providing the triangle index per resulting vertex. Only available when projectSimplifiedPointsOnInputMeshSurface is set to true
     */
    public static void simplifyTriangleMesh(PxArray_PxVec3 inputVertices, PxArray_PxU32 inputIndices, int targetTriangleCount, float maximalEdgeLength, PxArray_PxVec3 outputVertices, PxArray_PxU32 outputIndices, PxArray_PxU32 vertexMap, float edgeLengthCostWeight, float flatnessDetectionThreshold, boolean projectSimplifiedPointsOnInputMeshSurface, PxArray_PxU32 outputVertexToInputTriangle) {
        _simplifyTriangleMesh(inputVertices.getAddress(), inputIndices.getAddress(), targetTriangleCount, maximalEdgeLength, outputVertices.getAddress(), outputIndices.getAddress(), vertexMap.getAddress(), edgeLengthCostWeight, flatnessDetectionThreshold, projectSimplifiedPointsOnInputMeshSurface, outputVertexToInputTriangle.getAddress());
    }
    private static native void _simplifyTriangleMesh(long inputVertices, long inputIndices, int targetTriangleCount, float maximalEdgeLength, long outputVertices, long outputIndices, long vertexMap, float edgeLengthCostWeight, float flatnessDetectionThreshold, boolean projectSimplifiedPointsOnInputMeshSurface, long outputVertexToInputTriangle);

    /**
     * Simplifies (decimates) a triangle mesh using quadric simplification.
     * @param inputVertices The vertices of the input triangle mesh
     * @param inputIndices The indices of the input triangle mesh of the form  (id0, id1, id2),  (id0, id1, id2), ..
     * @param targetTriangleCount Desired number of triangles in the output mesh
     * @param maximalEdgeLength Edges below this length will not be collapsed. A value of zero means there is no limit.
     * @param outputVertices The vertices of the output (decimated) triangle mesh
     * @param outputIndices The indices of the output (decimated) triangle mesh of the form  (id0, id1, id2),  (id0, id1, id2), ..
     * @param vertexMap Optional parameter which returns the mapping from input to output vertices. Note that multiple input vertices are typically collapsed into the same output vertex.
     * @param edgeLengthCostWeight Factor to scale influence of edge length when prioritizing edge collapses. Has no effect if set to zero.
     * @param flatnessDetectionThreshold Threshold used to detect edges in flat regions and to improve the placement of the collapsed point. If set to a large value it will have no effect.
     * @param projectSimplifiedPointsOnInputMeshSurface If set to true, the simplified points will lie exactly on the original surface. 
     * @param outputVertexToInputTriangle Optional indices providing the triangle index per resulting vertex. Only available when projectSimplifiedPointsOnInputMeshSurface is set to true
     * @param removeDisconnectedPatches Enables the optional removal of disconnected triangles in the mesh. Only the largest connected set/patch will be kept
     */
    public static void simplifyTriangleMesh(PxArray_PxVec3 inputVertices, PxArray_PxU32 inputIndices, int targetTriangleCount, float maximalEdgeLength, PxArray_PxVec3 outputVertices, PxArray_PxU32 outputIndices, PxArray_PxU32 vertexMap, float edgeLengthCostWeight, float flatnessDetectionThreshold, boolean projectSimplifiedPointsOnInputMeshSurface, PxArray_PxU32 outputVertexToInputTriangle, boolean removeDisconnectedPatches) {
        _simplifyTriangleMesh(inputVertices.getAddress(), inputIndices.getAddress(), targetTriangleCount, maximalEdgeLength, outputVertices.getAddress(), outputIndices.getAddress(), vertexMap.getAddress(), edgeLengthCostWeight, flatnessDetectionThreshold, projectSimplifiedPointsOnInputMeshSurface, outputVertexToInputTriangle.getAddress(), removeDisconnectedPatches);
    }
    private static native void _simplifyTriangleMesh(long inputVertices, long inputIndices, int targetTriangleCount, float maximalEdgeLength, long outputVertices, long outputIndices, long vertexMap, float edgeLengthCostWeight, float flatnessDetectionThreshold, boolean projectSimplifiedPointsOnInputMeshSurface, long outputVertexToInputTriangle, boolean removeDisconnectedPatches);

    /**
     * Creates a new mesh from a given mesh. The input mesh is first voxelized. The new surface is created from the voxel surface and subsequent projection to the original mesh.
     * @param inputVertices The vertices of the input triangle mesh
     * @param inputIndices The indices of the input triangle mesh of the form  (id0, id1, id2),  (id0, id1, id2), ..
     * @param gridResolution Size of the voxel grid (number of voxels along the longest dimension)
     * @param outputVertices The vertices of the output (decimated) triangle mesh
     * @param outputIndices The indices of the output (decimated) triangle mesh of the form  (id0, id1, id2),  (id0, id1, id2), ..
     */
    public static void remeshTriangleMesh(PxArray_PxVec3 inputVertices, PxArray_PxU32 inputIndices, int gridResolution, PxArray_PxVec3 outputVertices, PxArray_PxU32 outputIndices) {
        _remeshTriangleMesh(inputVertices.getAddress(), inputIndices.getAddress(), gridResolution, outputVertices.getAddress(), outputIndices.getAddress());
    }
    private static native void _remeshTriangleMesh(long inputVertices, long inputIndices, int gridResolution, long outputVertices, long outputIndices);

    /**
     * Creates a new mesh from a given mesh. The input mesh is first voxelized. The new surface is created from the voxel surface and subsequent projection to the original mesh.
     * @param inputVertices The vertices of the input triangle mesh
     * @param inputIndices The indices of the input triangle mesh of the form  (id0, id1, id2),  (id0, id1, id2), ..
     * @param gridResolution Size of the voxel grid (number of voxels along the longest dimension)
     * @param outputVertices The vertices of the output (decimated) triangle mesh
     * @param outputIndices The indices of the output (decimated) triangle mesh of the form  (id0, id1, id2),  (id0, id1, id2), ..
     * @param vertexMap Optional parameter which returns a mapping from input to output vertices. Since the meshes are independent, the mapping returns an output vertex that is topologically close to the input vertex.
     */
    public static void remeshTriangleMesh(PxArray_PxVec3 inputVertices, PxArray_PxU32 inputIndices, int gridResolution, PxArray_PxVec3 outputVertices, PxArray_PxU32 outputIndices, PxArray_PxU32 vertexMap) {
        _remeshTriangleMesh(inputVertices.getAddress(), inputIndices.getAddress(), gridResolution, outputVertices.getAddress(), outputIndices.getAddress(), vertexMap.getAddress());
    }
    private static native void _remeshTriangleMesh(long inputVertices, long inputIndices, int gridResolution, long outputVertices, long outputIndices, long vertexMap);

    /**
     * Creates a tetrahedral mesh using an octree. 
     * @param inputVertices The vertices of the input triangle mesh
     * @param inputIndices The indices of the input triangle mesh of the form  (id0, id1, id2),  (id0, id1, id2), ..
     * @param useTreeNodes Using the nodes of the octree as tetrahedral vertices
     * @param outputVertices The vertices of the output tetrahedral mesh
     * @param outputIndices The indices of the output tetrahedral mesh of the form  (id0, id1, id2, id3),  (id0, id1, id2, id3), ..
     * @param volumeThreshold Tetrahedra with a volume smaller than the specified threshold will be removed from the mesh
     */
    public static void createTreeBasedTetrahedralMesh(PxArray_PxVec3 inputVertices, PxArray_PxU32 inputIndices, boolean useTreeNodes, PxArray_PxVec3 outputVertices, PxArray_PxU32 outputIndices, float volumeThreshold) {
        _createTreeBasedTetrahedralMesh(inputVertices.getAddress(), inputIndices.getAddress(), useTreeNodes, outputVertices.getAddress(), outputIndices.getAddress(), volumeThreshold);
    }
    private static native void _createTreeBasedTetrahedralMesh(long inputVertices, long inputIndices, boolean useTreeNodes, long outputVertices, long outputIndices, float volumeThreshold);

    /**
     * Creates a tetrahedral mesh by relaxing a voxel mesh around the input mesh
     * @param inputVertices The vertices of the input triangle mesh
     * @param inputIndices The indices of the input triangle mesh of the form  (id0, id1, id2),  (id0, id1, id2), ..
     * @param outputVertices The vertices of the output tetrahedral mesh
     * @param outputIndices The indices of the output tetrahedral mesh of the form  (id0, id1, id2, id3),  (id0, id1, id2, id3), ..
     * @param resolution The grid spacing is computed as the diagonal of the bounding box of the input mesh divided by the resolution.
     */
    public static void createRelaxedVoxelTetrahedralMesh(PxArray_PxVec3 inputVertices, PxArray_PxU32 inputIndices, PxArray_PxVec3 outputVertices, PxArray_PxU32 outputIndices, int resolution) {
        _createRelaxedVoxelTetrahedralMesh(inputVertices.getAddress(), inputIndices.getAddress(), outputVertices.getAddress(), outputIndices.getAddress(), resolution);
    }
    private static native void _createRelaxedVoxelTetrahedralMesh(long inputVertices, long inputIndices, long outputVertices, long outputIndices, int resolution);

    /**
     * Creates a tetrahedral mesh by relaxing a voxel mesh around the input mesh
     * @param inputVertices The vertices of the input triangle mesh
     * @param inputIndices The indices of the input triangle mesh of the form  (id0, id1, id2),  (id0, id1, id2), ..
     * @param outputVertices The vertices of the output tetrahedral mesh
     * @param outputIndices The indices of the output tetrahedral mesh of the form  (id0, id1, id2, id3),  (id0, id1, id2, id3), ..
     * @param resolution The grid spacing is computed as the diagonal of the bounding box of the input mesh divided by the resolution.
     * @param numRelaxationIterations Number of iterations to pull the tetrahedral mesh towards the input mesh
     */
    public static void createRelaxedVoxelTetrahedralMesh(PxArray_PxVec3 inputVertices, PxArray_PxU32 inputIndices, PxArray_PxVec3 outputVertices, PxArray_PxU32 outputIndices, int resolution, int numRelaxationIterations) {
        _createRelaxedVoxelTetrahedralMesh(inputVertices.getAddress(), inputIndices.getAddress(), outputVertices.getAddress(), outputIndices.getAddress(), resolution, numRelaxationIterations);
    }
    private static native void _createRelaxedVoxelTetrahedralMesh(long inputVertices, long inputIndices, long outputVertices, long outputIndices, int resolution, int numRelaxationIterations);

    /**
     * Creates a tetrahedral mesh by relaxing a voxel mesh around the input mesh
     * @param inputVertices The vertices of the input triangle mesh
     * @param inputIndices The indices of the input triangle mesh of the form  (id0, id1, id2),  (id0, id1, id2), ..
     * @param outputVertices The vertices of the output tetrahedral mesh
     * @param outputIndices The indices of the output tetrahedral mesh of the form  (id0, id1, id2, id3),  (id0, id1, id2, id3), ..
     * @param resolution The grid spacing is computed as the diagonal of the bounding box of the input mesh divided by the resolution.
     * @param numRelaxationIterations Number of iterations to pull the tetrahedral mesh towards the input mesh
     * @param relMinTetVolume Constrains the volumes of the tetrahedra to stay abobe relMinTetvolume times the tetrahedron's rest volume.
     */
    public static void createRelaxedVoxelTetrahedralMesh(PxArray_PxVec3 inputVertices, PxArray_PxU32 inputIndices, PxArray_PxVec3 outputVertices, PxArray_PxU32 outputIndices, int resolution, int numRelaxationIterations, float relMinTetVolume) {
        _createRelaxedVoxelTetrahedralMesh(inputVertices.getAddress(), inputIndices.getAddress(), outputVertices.getAddress(), outputIndices.getAddress(), resolution, numRelaxationIterations, relMinTetVolume);
    }
    private static native void _createRelaxedVoxelTetrahedralMesh(long inputVertices, long inputIndices, long outputVertices, long outputIndices, int resolution, int numRelaxationIterations, float relMinTetVolume);

    /**
     * Creates a tetrahedral mesh by relaxing a voxel mesh around the input mesh
     * @param triangles The indices of the input triangle mesh of the form  (id0, id1, id2),  (id0, id1, id2), ..
     * @param numTriangles The number of triangles
     * @param islandIndexPerTriangle Every triangle gets an island index assigned. Triangles with the same island index belong to the same patch of connected triangles.
     */
    public static void detectTriangleIslands(PxI32ConstPtr triangles, int numTriangles, PxArray_PxU32 islandIndexPerTriangle) {
        _detectTriangleIslands(triangles.getAddress(), numTriangles, islandIndexPerTriangle.getAddress());
    }
    private static native void _detectTriangleIslands(long triangles, int numTriangles, long islandIndexPerTriangle);

    /**
     * Creates a tetrahedral mesh by relaxing a voxel mesh around the input mesh
     * @param islandIndexPerTriangle An island marker per triangles. All triangles with the same marker belong to an island. Can becomputed using the method detectTriangleIslands.
     * @param numTriangles The number of triangles
     * @return The marker value of the island that contains the most triangles
     */
    public static int findLargestIslandId(PxU32ConstPtr islandIndexPerTriangle, int numTriangles) {
        return _findLargestIslandId(islandIndexPerTriangle.getAddress(), numTriangles);
    }
    private static native int _findLargestIslandId(long islandIndexPerTriangle, int numTriangles);

}
