package physx.geometry;

import physx.NativeObject;

/**
 * Heightfield sample format.
 * <p>
 * This format corresponds to the #PxHeightFieldFormat member PxHeightFieldFormat::eS16_TM.
 * <p>
 * An array of heightfield samples are used when creating a PxHeightField to specify
 * the elevation of the heightfield points. In addition the material and tessellation of the adjacent 
 * triangles are specified.
 * @see PxHeightField
 * @see PxHeightFieldDesc
 */
public class PxHeightFieldSample extends NativeObject {

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxHeightFieldSample wrapPointer(long address) {
        return address != 0L ? new PxHeightFieldSample(address) : null;
    }
    
    public static PxHeightFieldSample arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxHeightFieldSample(long address) {
        super(address);
    }

    // Placed Constructors

    /**
     * @param address Pre-allocated memory, where the object is created.
     * @return Stack allocated object of PxHeightFieldSample
     */
    public static PxHeightFieldSample createAt(long address) {
        __placement_new_PxHeightFieldSample(address);
        PxHeightFieldSample createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    /**
     * @param <T>       Allocator class, e.g. LWJGL's MemoryStack.
     * @param allocator Object to use for allocation, e.g. an instance of LWJGL's MemoryStack.
     * @param allocate  Method to call on allocator to obtain the target address, e.g. MemoryStack::nmalloc.
     * @return Stack allocated object of PxHeightFieldSample
     */
    public static <T> PxHeightFieldSample createAt(T allocator, Allocator<T> allocate) {
        long address = allocate.on(allocator, ALIGNOF, SIZEOF); 
        __placement_new_PxHeightFieldSample(address);
        PxHeightFieldSample createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    private static native void __placement_new_PxHeightFieldSample(long address);

    // Constructors

    public PxHeightFieldSample() {
        address = _PxHeightFieldSample();
    }
    private static native long _PxHeightFieldSample();

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * The height of the heightfield sample
     * <p>
     * This value is scaled by PxHeightFieldGeometry::heightScale.
     * @see PxHeightFieldGeometry
     */
    public short getHeight() {
        checkNotNull();
        return _getHeight(address);
    }
    private static native short _getHeight(long address);

    /**
     * The height of the heightfield sample
     * <p>
     * This value is scaled by PxHeightFieldGeometry::heightScale.
     * @see PxHeightFieldGeometry
     */
    public void setHeight(short value) {
        checkNotNull();
        _setHeight(address, value);
    }
    private static native void _setHeight(long address, short value);

    /**
     * The triangle material index of the quad's lower triangle + tesselation flag
     * <p>
     * An index pointing into the material table of the shape which instantiates the heightfield.
     * This index determines the material of the lower of the quad's two triangles (i.e. the quad whose 
     * upper-left corner is this sample, see the Guide for illustrations).
     * <p>
     * Special values of the 7 data bits are defined by PxHeightFieldMaterial
     * <p>
     * The tesselation flag specifies which way the quad is split whose upper left corner is this sample.
     * If the flag is set, the diagonal of the quad will run from this sample to the opposite vertex; if not,
     * it will run between the other two vertices (see the Guide for illustrations).
     * @see PxHeightFieldGeometry
     * @see physx.physics.PxShape#getMaterials
     */
    public byte getMaterialIndex0() {
        checkNotNull();
        return _getMaterialIndex0(address);
    }
    private static native byte _getMaterialIndex0(long address);

    /**
     * The triangle material index of the quad's lower triangle + tesselation flag
     * <p>
     * An index pointing into the material table of the shape which instantiates the heightfield.
     * This index determines the material of the lower of the quad's two triangles (i.e. the quad whose 
     * upper-left corner is this sample, see the Guide for illustrations).
     * <p>
     * Special values of the 7 data bits are defined by PxHeightFieldMaterial
     * <p>
     * The tesselation flag specifies which way the quad is split whose upper left corner is this sample.
     * If the flag is set, the diagonal of the quad will run from this sample to the opposite vertex; if not,
     * it will run between the other two vertices (see the Guide for illustrations).
     * @see PxHeightFieldGeometry
     * @see physx.physics.PxShape#getMaterials
     */
    public void setMaterialIndex0(byte value) {
        checkNotNull();
        _setMaterialIndex0(address, value);
    }
    private static native void _setMaterialIndex0(long address, byte value);

    /**
     * The triangle material index of the quad's upper triangle + reserved flag
     * <p>
     * An index pointing into the material table of the shape which instantiates the heightfield.
     * This index determines the material of the upper of the quad's two triangles (i.e. the quad whose 
     * upper-left corner is this sample, see the Guide for illustrations).
     * @see PxHeightFieldGeometry
     * @see physx.physics.PxShape#getMaterials
     */
    public byte getMaterialIndex1() {
        checkNotNull();
        return _getMaterialIndex1(address);
    }
    private static native byte _getMaterialIndex1(long address);

    /**
     * The triangle material index of the quad's upper triangle + reserved flag
     * <p>
     * An index pointing into the material table of the shape which instantiates the heightfield.
     * This index determines the material of the upper of the quad's two triangles (i.e. the quad whose 
     * upper-left corner is this sample, see the Guide for illustrations).
     * @see PxHeightFieldGeometry
     * @see physx.physics.PxShape#getMaterials
     */
    public void setMaterialIndex1(byte value) {
        checkNotNull();
        _setMaterialIndex1(address, value);
    }
    private static native void _setMaterialIndex1(long address, byte value);

    // Functions

    /**
     * @return WebIDL type: octet
     */
    public byte tessFlag() {
        checkNotNull();
        return _tessFlag(address);
    }
    private static native byte _tessFlag(long address);

    public void clearTessFlag() {
        checkNotNull();
        _clearTessFlag(address);
    }
    private static native void _clearTessFlag(long address);

    public void setTessFlag() {
        checkNotNull();
        _setTessFlag(address);
    }
    private static native void _setTessFlag(long address);

}
