package physx.extensions;

import physx.NativeObject;
import physx.common.PxTransform;
import physx.common.PxVec3;
import physx.physics.PxForceModeEnum;
import physx.physics.PxRigidBody;

/**
 * utility functions for use with PxRigidBody and subclasses
 * @see physx.physics.PxRigidBody
 * @see physx.physics.PxRigidDynamic
 * @see physx.physics.PxArticulationLink
 */
public class PxRigidBodyExt extends NativeObject {

    protected PxRigidBodyExt() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxRigidBodyExt wrapPointer(long address) {
        return address != 0L ? new PxRigidBodyExt(address) : null;
    }
    
    public static PxRigidBodyExt arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxRigidBodyExt(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Computation of mass properties for a rigid body actor
     * <p>
     * See previous method for details.
     * <p>
     * \param[in,out] body The rigid body.
     * @param density The density of the body. Used to compute the mass of the body. The density must be greater than 0. 
     * @return Boolean. True on success else false.
     */
    public static boolean updateMassAndInertia(PxRigidBody body, float density) {
        return _updateMassAndInertia(body.getAddress(), density);
    }
    private static native boolean _updateMassAndInertia(long body, float density);

    /**
     * Computation of mass properties for a rigid body actor
     * <p>
     * See previous method for details.
     * <p>
     * \param[in,out] body The rigid body.
     * @param density The density of the body. Used to compute the mass of the body. The density must be greater than 0. 
     * @param massLocalPose The center of mass relative to the actor frame.  If set to null then (0,0,0) is assumed.
     * @return Boolean. True on success else false.
     */
    public static boolean updateMassAndInertia(PxRigidBody body, float density, PxVec3 massLocalPose) {
        return _updateMassAndInertia(body.getAddress(), density, massLocalPose.getAddress());
    }
    private static native boolean _updateMassAndInertia(long body, float density, long massLocalPose);

    /**
     * Computation of mass properties for a rigid body actor
     * <p>
     * See previous method for details.
     * <p>
     * \param[in,out] body The rigid body.
     * @param density The density of the body. Used to compute the mass of the body. The density must be greater than 0. 
     * @param massLocalPose The center of mass relative to the actor frame.  If set to null then (0,0,0) is assumed.
     * @param includeNonSimShapes True if all kind of shapes (PxShapeFlag::eSCENE_QUERY_SHAPE, PxShapeFlag::eTRIGGER_SHAPE) should be taken into account.
     * @return Boolean. True on success else false.
     */
    public static boolean updateMassAndInertia(PxRigidBody body, float density, PxVec3 massLocalPose, boolean includeNonSimShapes) {
        return _updateMassAndInertia(body.getAddress(), density, massLocalPose.getAddress(), includeNonSimShapes);
    }
    private static native boolean _updateMassAndInertia(long body, float density, long massLocalPose, boolean includeNonSimShapes);

    /**
     * Computation of mass properties for a rigid body actor
     * <p>
     * This method sets the mass, inertia and center of mass of a rigid body. The mass is set to the user-supplied
     * value, and the inertia and center of mass are computed according to the rigid body's shapes and the input mass.
     * <p>
     * If no collision shapes are found, the inertia tensor is set to (1,1,1)
     * @see #updateMassAndInertia
     * <p>
     * \param[in,out] body The rigid body for which to set the mass and centre of mass local pose properties.
     * @param mass The mass of the body. Must be greater than 0.
     * @return Boolean. True on success else false.
     */
    public static boolean setMassAndUpdateInertia(PxRigidBody body, float mass) {
        return _setMassAndUpdateInertia(body.getAddress(), mass);
    }
    private static native boolean _setMassAndUpdateInertia(long body, float mass);

    /**
     * Computation of mass properties for a rigid body actor
     * <p>
     * This method sets the mass, inertia and center of mass of a rigid body. The mass is set to the user-supplied
     * value, and the inertia and center of mass are computed according to the rigid body's shapes and the input mass.
     * <p>
     * If no collision shapes are found, the inertia tensor is set to (1,1,1)
     * @see #updateMassAndInertia
     * <p>
     * \param[in,out] body The rigid body for which to set the mass and centre of mass local pose properties.
     * @param mass The mass of the body. Must be greater than 0.
     * @param massLocalPose The center of mass relative to the actor frame. If set to null then (0,0,0) is assumed.
     * @return Boolean. True on success else false.
     */
    public static boolean setMassAndUpdateInertia(PxRigidBody body, float mass, PxVec3 massLocalPose) {
        return _setMassAndUpdateInertia(body.getAddress(), mass, massLocalPose.getAddress());
    }
    private static native boolean _setMassAndUpdateInertia(long body, float mass, long massLocalPose);

    /**
     * Computation of mass properties for a rigid body actor
     * <p>
     * This method sets the mass, inertia and center of mass of a rigid body. The mass is set to the user-supplied
     * value, and the inertia and center of mass are computed according to the rigid body's shapes and the input mass.
     * <p>
     * If no collision shapes are found, the inertia tensor is set to (1,1,1)
     * @see #updateMassAndInertia
     * <p>
     * \param[in,out] body The rigid body for which to set the mass and centre of mass local pose properties.
     * @param mass The mass of the body. Must be greater than 0.
     * @param massLocalPose The center of mass relative to the actor frame. If set to null then (0,0,0) is assumed.
     * @param includeNonSimShapes True if all kind of shapes (PxShapeFlag::eSCENE_QUERY_SHAPE, PxShapeFlag::eTRIGGER_SHAPE) should be taken into account.
     * @return Boolean. True on success else false.
     */
    public static boolean setMassAndUpdateInertia(PxRigidBody body, float mass, PxVec3 massLocalPose, boolean includeNonSimShapes) {
        return _setMassAndUpdateInertia(body.getAddress(), mass, massLocalPose.getAddress(), includeNonSimShapes);
    }
    private static native boolean _setMassAndUpdateInertia(long body, float mass, long massLocalPose, boolean includeNonSimShapes);

    /**
     * Applies a force (or impulse) defined in the global coordinate frame, acting at a particular 
     * point in global coordinates, to the actor. 
     * <p>
     * Note that if the force does not act along the center of mass of the actor, this
     * will also add the corresponding torque. Because forces are reset at the end of every timestep, 
     * you can maintain a total external force on an object by calling this once every frame.
     * <p>
     * <b>Note:</b> if this call is used to apply a force or impulse to an articulation link, only the link is updated, not the entire
     * articulation
     * <p>
     * ::PxForceMode determines if the force is to be conventional or impulsive. Only eFORCE and eIMPULSE are supported, as the 
     * force required to produce a given velocity change or acceleration is underdetermined given only the desired change at a
     * given point.
     * <p>
     * <b>Sleeping:</b> This call wakes the actor if it is sleeping and the wakeup parameter is true (default).
     * @param body The rigid body to apply the force to.
     * @param force Force/impulse to add, defined in the global frame. <b>Range:</b> force vector
     * @param pos Position in the global frame to add the force at. <b>Range:</b> position vector
     * @see #addForceAtLocalPos
     * @see #addLocalForceAtPos
     * @see #addLocalForceAtLocalPos
     */
    public static void addForceAtPos(PxRigidBody body, PxVec3 force, PxVec3 pos) {
        _addForceAtPos(body.getAddress(), force.getAddress(), pos.getAddress());
    }
    private static native void _addForceAtPos(long body, long force, long pos);

    /**
     * Applies a force (or impulse) defined in the global coordinate frame, acting at a particular 
     * point in global coordinates, to the actor. 
     * <p>
     * Note that if the force does not act along the center of mass of the actor, this
     * will also add the corresponding torque. Because forces are reset at the end of every timestep, 
     * you can maintain a total external force on an object by calling this once every frame.
     * <p>
     * <b>Note:</b> if this call is used to apply a force or impulse to an articulation link, only the link is updated, not the entire
     * articulation
     * <p>
     * ::PxForceMode determines if the force is to be conventional or impulsive. Only eFORCE and eIMPULSE are supported, as the 
     * force required to produce a given velocity change or acceleration is underdetermined given only the desired change at a
     * given point.
     * <p>
     * <b>Sleeping:</b> This call wakes the actor if it is sleeping and the wakeup parameter is true (default).
     * @param body The rigid body to apply the force to.
     * @param force Force/impulse to add, defined in the global frame. <b>Range:</b> force vector
     * @param pos Position in the global frame to add the force at. <b>Range:</b> position vector
     * @param mode The mode to use when applying the force/impulse(see #PxForceMode). 
     * @see #addForceAtLocalPos
     * @see #addLocalForceAtPos
     * @see #addLocalForceAtLocalPos
     */
    public static void addForceAtPos(PxRigidBody body, PxVec3 force, PxVec3 pos, PxForceModeEnum mode) {
        _addForceAtPos(body.getAddress(), force.getAddress(), pos.getAddress(), mode.value);
    }
    private static native void _addForceAtPos(long body, long force, long pos, int mode);

    /**
     * Applies a force (or impulse) defined in the global coordinate frame, acting at a particular 
     * point in global coordinates, to the actor. 
     * <p>
     * Note that if the force does not act along the center of mass of the actor, this
     * will also add the corresponding torque. Because forces are reset at the end of every timestep, 
     * you can maintain a total external force on an object by calling this once every frame.
     * <p>
     * <b>Note:</b> if this call is used to apply a force or impulse to an articulation link, only the link is updated, not the entire
     * articulation
     * <p>
     * ::PxForceMode determines if the force is to be conventional or impulsive. Only eFORCE and eIMPULSE are supported, as the 
     * force required to produce a given velocity change or acceleration is underdetermined given only the desired change at a
     * given point.
     * <p>
     * <b>Sleeping:</b> This call wakes the actor if it is sleeping and the wakeup parameter is true (default).
     * @param body The rigid body to apply the force to.
     * @param force Force/impulse to add, defined in the global frame. <b>Range:</b> force vector
     * @param pos Position in the global frame to add the force at. <b>Range:</b> position vector
     * @param mode The mode to use when applying the force/impulse(see #PxForceMode). 
     * @param wakeup Specify if the call should wake up the actor.
     * @see #addForceAtLocalPos
     * @see #addLocalForceAtPos
     * @see #addLocalForceAtLocalPos
     */
    public static void addForceAtPos(PxRigidBody body, PxVec3 force, PxVec3 pos, PxForceModeEnum mode, boolean wakeup) {
        _addForceAtPos(body.getAddress(), force.getAddress(), pos.getAddress(), mode.value, wakeup);
    }
    private static native void _addForceAtPos(long body, long force, long pos, int mode, boolean wakeup);

    /**
     * Applies a force (or impulse) defined in the global coordinate frame, acting at a particular 
     * point in local coordinates, to the actor. 
     * <p>
     * Note that if the force does not act along the center of mass of the actor, this
     * will also add the corresponding torque. Because forces are reset at the end of every timestep, you can maintain a
     * total external force on an object by calling this once every frame.
     * <p>
     * <b>Note:</b> if this call is used to apply a force or impulse to an articulation link, only the link is updated, not the entire
     * articulation
     * <p>
     * ::PxForceMode determines if the force is to be conventional or impulsive. Only eFORCE and eIMPULSE are supported, as the 
     * force required to produce a given velocity change or acceleration is underdetermined given only the desired change at a
     * given point.
     * <p>
     * <b>Sleeping:</b> This call wakes the actor if it is sleeping and the wakeup parameter is true (default).
     * @param body The rigid body to apply the force to.
     * @param force Force/impulse to add, defined in the global frame. <b>Range:</b> force vector
     * @param pos Position in the local frame to add the force at. <b>Range:</b> position vector
     * @see #addForceAtPos
     * @see #addLocalForceAtPos
     * @see #addLocalForceAtLocalPos
     */
    public static void addForceAtLocalPos(PxRigidBody body, PxVec3 force, PxVec3 pos) {
        _addForceAtLocalPos(body.getAddress(), force.getAddress(), pos.getAddress());
    }
    private static native void _addForceAtLocalPos(long body, long force, long pos);

    /**
     * Applies a force (or impulse) defined in the global coordinate frame, acting at a particular 
     * point in local coordinates, to the actor. 
     * <p>
     * Note that if the force does not act along the center of mass of the actor, this
     * will also add the corresponding torque. Because forces are reset at the end of every timestep, you can maintain a
     * total external force on an object by calling this once every frame.
     * <p>
     * <b>Note:</b> if this call is used to apply a force or impulse to an articulation link, only the link is updated, not the entire
     * articulation
     * <p>
     * ::PxForceMode determines if the force is to be conventional or impulsive. Only eFORCE and eIMPULSE are supported, as the 
     * force required to produce a given velocity change or acceleration is underdetermined given only the desired change at a
     * given point.
     * <p>
     * <b>Sleeping:</b> This call wakes the actor if it is sleeping and the wakeup parameter is true (default).
     * @param body The rigid body to apply the force to.
     * @param force Force/impulse to add, defined in the global frame. <b>Range:</b> force vector
     * @param pos Position in the local frame to add the force at. <b>Range:</b> position vector
     * @param mode The mode to use when applying the force/impulse(see #PxForceMode). 
     * @see #addForceAtPos
     * @see #addLocalForceAtPos
     * @see #addLocalForceAtLocalPos
     */
    public static void addForceAtLocalPos(PxRigidBody body, PxVec3 force, PxVec3 pos, PxForceModeEnum mode) {
        _addForceAtLocalPos(body.getAddress(), force.getAddress(), pos.getAddress(), mode.value);
    }
    private static native void _addForceAtLocalPos(long body, long force, long pos, int mode);

    /**
     * Applies a force (or impulse) defined in the global coordinate frame, acting at a particular 
     * point in local coordinates, to the actor. 
     * <p>
     * Note that if the force does not act along the center of mass of the actor, this
     * will also add the corresponding torque. Because forces are reset at the end of every timestep, you can maintain a
     * total external force on an object by calling this once every frame.
     * <p>
     * <b>Note:</b> if this call is used to apply a force or impulse to an articulation link, only the link is updated, not the entire
     * articulation
     * <p>
     * ::PxForceMode determines if the force is to be conventional or impulsive. Only eFORCE and eIMPULSE are supported, as the 
     * force required to produce a given velocity change or acceleration is underdetermined given only the desired change at a
     * given point.
     * <p>
     * <b>Sleeping:</b> This call wakes the actor if it is sleeping and the wakeup parameter is true (default).
     * @param body The rigid body to apply the force to.
     * @param force Force/impulse to add, defined in the global frame. <b>Range:</b> force vector
     * @param pos Position in the local frame to add the force at. <b>Range:</b> position vector
     * @param mode The mode to use when applying the force/impulse(see #PxForceMode). 
     * @param wakeup Specify if the call should wake up the actor.
     * @see #addForceAtPos
     * @see #addLocalForceAtPos
     * @see #addLocalForceAtLocalPos
     */
    public static void addForceAtLocalPos(PxRigidBody body, PxVec3 force, PxVec3 pos, PxForceModeEnum mode, boolean wakeup) {
        _addForceAtLocalPos(body.getAddress(), force.getAddress(), pos.getAddress(), mode.value, wakeup);
    }
    private static native void _addForceAtLocalPos(long body, long force, long pos, int mode, boolean wakeup);

    /**
     * Applies a force (or impulse) defined in the actor local coordinate frame, acting at a 
     * particular point in global coordinates, to the actor. 
     * <p>
     * Note that if the force does not act along the center of mass of the actor, this
     * will also add the corresponding torque. Because forces are reset at the end of every timestep, you can maintain a
     * total external force on an object by calling this once every frame.
     * <p>
     * <b>Note:</b> if this call is used to apply a force or impulse to an articulation link, only the link is updated, not the entire
     * articulation
     * <p>
     * ::PxForceMode determines if the force is to be conventional or impulsive. Only eFORCE and eIMPULSE are supported, as the 
     * force required to produce a given velocity change or acceleration is underdetermined given only the desired change at a
     * given point.
     * <p>
     * <b>Sleeping:</b> This call wakes the actor if it is sleeping and the wakeup parameter is true (default).
     * @param body The rigid body to apply the force to.
     * @param force Force/impulse to add, defined in the local frame. <b>Range:</b> force vector
     * @param pos Position in the global frame to add the force at. <b>Range:</b> position vector
     * @see #addForceAtPos
     * @see #addForceAtLocalPos
     * @see #addLocalForceAtLocalPos
     */
    public static void addLocalForceAtPos(PxRigidBody body, PxVec3 force, PxVec3 pos) {
        _addLocalForceAtPos(body.getAddress(), force.getAddress(), pos.getAddress());
    }
    private static native void _addLocalForceAtPos(long body, long force, long pos);

    /**
     * Applies a force (or impulse) defined in the actor local coordinate frame, acting at a 
     * particular point in global coordinates, to the actor. 
     * <p>
     * Note that if the force does not act along the center of mass of the actor, this
     * will also add the corresponding torque. Because forces are reset at the end of every timestep, you can maintain a
     * total external force on an object by calling this once every frame.
     * <p>
     * <b>Note:</b> if this call is used to apply a force or impulse to an articulation link, only the link is updated, not the entire
     * articulation
     * <p>
     * ::PxForceMode determines if the force is to be conventional or impulsive. Only eFORCE and eIMPULSE are supported, as the 
     * force required to produce a given velocity change or acceleration is underdetermined given only the desired change at a
     * given point.
     * <p>
     * <b>Sleeping:</b> This call wakes the actor if it is sleeping and the wakeup parameter is true (default).
     * @param body The rigid body to apply the force to.
     * @param force Force/impulse to add, defined in the local frame. <b>Range:</b> force vector
     * @param pos Position in the global frame to add the force at. <b>Range:</b> position vector
     * @param mode The mode to use when applying the force/impulse(see #PxForceMode). 
     * @see #addForceAtPos
     * @see #addForceAtLocalPos
     * @see #addLocalForceAtLocalPos
     */
    public static void addLocalForceAtPos(PxRigidBody body, PxVec3 force, PxVec3 pos, PxForceModeEnum mode) {
        _addLocalForceAtPos(body.getAddress(), force.getAddress(), pos.getAddress(), mode.value);
    }
    private static native void _addLocalForceAtPos(long body, long force, long pos, int mode);

    /**
     * Applies a force (or impulse) defined in the actor local coordinate frame, acting at a 
     * particular point in global coordinates, to the actor. 
     * <p>
     * Note that if the force does not act along the center of mass of the actor, this
     * will also add the corresponding torque. Because forces are reset at the end of every timestep, you can maintain a
     * total external force on an object by calling this once every frame.
     * <p>
     * <b>Note:</b> if this call is used to apply a force or impulse to an articulation link, only the link is updated, not the entire
     * articulation
     * <p>
     * ::PxForceMode determines if the force is to be conventional or impulsive. Only eFORCE and eIMPULSE are supported, as the 
     * force required to produce a given velocity change or acceleration is underdetermined given only the desired change at a
     * given point.
     * <p>
     * <b>Sleeping:</b> This call wakes the actor if it is sleeping and the wakeup parameter is true (default).
     * @param body The rigid body to apply the force to.
     * @param force Force/impulse to add, defined in the local frame. <b>Range:</b> force vector
     * @param pos Position in the global frame to add the force at. <b>Range:</b> position vector
     * @param mode The mode to use when applying the force/impulse(see #PxForceMode). 
     * @param wakeup Specify if the call should wake up the actor.
     * @see #addForceAtPos
     * @see #addForceAtLocalPos
     * @see #addLocalForceAtLocalPos
     */
    public static void addLocalForceAtPos(PxRigidBody body, PxVec3 force, PxVec3 pos, PxForceModeEnum mode, boolean wakeup) {
        _addLocalForceAtPos(body.getAddress(), force.getAddress(), pos.getAddress(), mode.value, wakeup);
    }
    private static native void _addLocalForceAtPos(long body, long force, long pos, int mode, boolean wakeup);

    /**
     * Applies a force (or impulse) defined in the actor local coordinate frame, acting at a 
     * particular point in local coordinates, to the actor. 
     * <p>
     * Note that if the force does not act along the center of mass of the actor, this
     * will also add the corresponding torque. Because forces are reset at the end of every timestep, you can maintain a
     * total external force on an object by calling this once every frame.
     * <p>
     * <b>Note:</b> if this call is used to apply a force or impulse to an articulation link, only the link is updated, not the entire
     * articulation
     * <p>
     * ::PxForceMode determines if the force is to be conventional or impulsive. Only eFORCE and eIMPULSE are supported, as the 
     * force required to produce a given velocity change or acceleration is underdetermined given only the desired change at a
     * given point.
     * <p>
     * <b>Sleeping:</b> This call wakes the actor if it is sleeping and the wakeup parameter is true (default).
     * @param body The rigid body to apply the force to.
     * @param force Force/impulse to add, defined in the local frame. <b>Range:</b> force vector
     * @param pos Position in the local frame to add the force at. <b>Range:</b> position vector
     * @see #addForceAtPos
     * @see #addForceAtLocalPos
     * @see #addLocalForceAtPos
     */
    public static void addLocalForceAtLocalPos(PxRigidBody body, PxVec3 force, PxVec3 pos) {
        _addLocalForceAtLocalPos(body.getAddress(), force.getAddress(), pos.getAddress());
    }
    private static native void _addLocalForceAtLocalPos(long body, long force, long pos);

    /**
     * Applies a force (or impulse) defined in the actor local coordinate frame, acting at a 
     * particular point in local coordinates, to the actor. 
     * <p>
     * Note that if the force does not act along the center of mass of the actor, this
     * will also add the corresponding torque. Because forces are reset at the end of every timestep, you can maintain a
     * total external force on an object by calling this once every frame.
     * <p>
     * <b>Note:</b> if this call is used to apply a force or impulse to an articulation link, only the link is updated, not the entire
     * articulation
     * <p>
     * ::PxForceMode determines if the force is to be conventional or impulsive. Only eFORCE and eIMPULSE are supported, as the 
     * force required to produce a given velocity change or acceleration is underdetermined given only the desired change at a
     * given point.
     * <p>
     * <b>Sleeping:</b> This call wakes the actor if it is sleeping and the wakeup parameter is true (default).
     * @param body The rigid body to apply the force to.
     * @param force Force/impulse to add, defined in the local frame. <b>Range:</b> force vector
     * @param pos Position in the local frame to add the force at. <b>Range:</b> position vector
     * @param mode The mode to use when applying the force/impulse(see #PxForceMode). 
     * @see #addForceAtPos
     * @see #addForceAtLocalPos
     * @see #addLocalForceAtPos
     */
    public static void addLocalForceAtLocalPos(PxRigidBody body, PxVec3 force, PxVec3 pos, PxForceModeEnum mode) {
        _addLocalForceAtLocalPos(body.getAddress(), force.getAddress(), pos.getAddress(), mode.value);
    }
    private static native void _addLocalForceAtLocalPos(long body, long force, long pos, int mode);

    /**
     * Applies a force (or impulse) defined in the actor local coordinate frame, acting at a 
     * particular point in local coordinates, to the actor. 
     * <p>
     * Note that if the force does not act along the center of mass of the actor, this
     * will also add the corresponding torque. Because forces are reset at the end of every timestep, you can maintain a
     * total external force on an object by calling this once every frame.
     * <p>
     * <b>Note:</b> if this call is used to apply a force or impulse to an articulation link, only the link is updated, not the entire
     * articulation
     * <p>
     * ::PxForceMode determines if the force is to be conventional or impulsive. Only eFORCE and eIMPULSE are supported, as the 
     * force required to produce a given velocity change or acceleration is underdetermined given only the desired change at a
     * given point.
     * <p>
     * <b>Sleeping:</b> This call wakes the actor if it is sleeping and the wakeup parameter is true (default).
     * @param body The rigid body to apply the force to.
     * @param force Force/impulse to add, defined in the local frame. <b>Range:</b> force vector
     * @param pos Position in the local frame to add the force at. <b>Range:</b> position vector
     * @param mode The mode to use when applying the force/impulse(see #PxForceMode). 
     * @param wakeup Specify if the call should wake up the actor.
     * @see #addForceAtPos
     * @see #addForceAtLocalPos
     * @see #addLocalForceAtPos
     */
    public static void addLocalForceAtLocalPos(PxRigidBody body, PxVec3 force, PxVec3 pos, PxForceModeEnum mode, boolean wakeup) {
        _addLocalForceAtLocalPos(body.getAddress(), force.getAddress(), pos.getAddress(), mode.value, wakeup);
    }
    private static native void _addLocalForceAtLocalPos(long body, long force, long pos, int mode, boolean wakeup);

    /**
     * Computes the velocity of a point given in world coordinates if it were attached to the 
     * specified body and moving with it.
     * @param body The rigid body the point is attached to.
     * @param pos Position we wish to determine the velocity for, defined in the global frame. <b>Range:</b> position vector
     * @return The velocity of point in the global frame.
     */
    public static PxVec3 getVelocityAtPos(PxRigidBody body, PxVec3 pos) {
        return PxVec3.wrapPointer(_getVelocityAtPos(body.getAddress(), pos.getAddress()));
    }
    private static native long _getVelocityAtPos(long body, long pos);

    /**
     * Computes the velocity of a point given in local coordinates if it were attached to the 
     * specified body and moving with it.
     * @param body The rigid body the point is attached to.
     * @param pos Position we wish to determine the velocity for, defined in the local frame. <b>Range:</b> position vector
     * @return The velocity of point in the local frame.
     */
    public static PxVec3 getLocalVelocityAtLocalPos(PxRigidBody body, PxVec3 pos) {
        return PxVec3.wrapPointer(_getLocalVelocityAtLocalPos(body.getAddress(), pos.getAddress()));
    }
    private static native long _getLocalVelocityAtLocalPos(long body, long pos);

    /**
     * Computes the velocity of a point (offset from the origin of the body) given in world coordinates if it were attached to the 
     * specified body and moving with it.
     * @param body The rigid body the point is attached to.
     * @param pos Position (offset from the origin of the body) we wish to determine the velocity for, defined in the global frame. <b>Range:</b> position vector
     * @return The velocity of point (offset from the origin of the body) in the global frame.
     */
    public static PxVec3 getVelocityAtOffset(PxRigidBody body, PxVec3 pos) {
        return PxVec3.wrapPointer(_getVelocityAtOffset(body.getAddress(), pos.getAddress()));
    }
    private static native long _getVelocityAtOffset(long body, long pos);

    /**
     * Compute the change to linear and angular velocity that would occur if an impulsive force and torque were to be applied to a specified rigid body. 
     * <p>
     * The rigid body is left unaffected unless a subsequent independent call is executed that actually applies the computed changes to velocity and angular velocity.
     * <p>
     * <b>Note:</b> if this call is used to determine the velocity delta for an articulation link, only the mass properties of the link are taken into account.
     * @param body The body under consideration.
     * @param impulsiveForce The impulsive force that would be applied to the specified rigid body.
     * @param impulsiveTorque The impulsive torque that would be applied to the specified rigid body.
     * @param deltaLinearVelocity The change in linear velocity that would arise if impulsiveForce was to be applied to the specified rigid body.
     * @param deltaAngularVelocity The change in angular velocity that would arise if impulsiveTorque was to be applied to the specified rigid body.
     */
    public static void computeVelocityDeltaFromImpulse(PxRigidBody body, PxVec3 impulsiveForce, PxVec3 impulsiveTorque, PxVec3 deltaLinearVelocity, PxVec3 deltaAngularVelocity) {
        _computeVelocityDeltaFromImpulse(body.getAddress(), impulsiveForce.getAddress(), impulsiveTorque.getAddress(), deltaLinearVelocity.getAddress(), deltaAngularVelocity.getAddress());
    }
    private static native void _computeVelocityDeltaFromImpulse(long body, long impulsiveForce, long impulsiveTorque, long deltaLinearVelocity, long deltaAngularVelocity);

    /**
     * Computes the linear and angular velocity change vectors for a given impulse at a world space position taking a mass and inertia scale into account
     * <p>
     * This function is useful for extracting the respective linear and angular velocity changes from a contact or joint when the mass/inertia ratios have been adjusted.
     * <p>
     * <b>Note:</b> if this call is used to determine the velocity delta for an articulation link, only the mass properties of the link are taken into account.
     * @param body The rigid body
     * @param globalPose The body's world space transform
     * @param point The point in world space where the impulse is applied
     * @param impulse The impulse vector in world space
     * @param invMassScale The inverse mass scale
     * @param invInertiaScale The inverse inertia scale
     * @param deltaLinearVelocity The linear velocity change
     * @param deltaAngularVelocity The angular velocity change
     */
    public static void computeVelocityDeltaFromImpulse(PxRigidBody body, PxTransform globalPose, PxVec3 point, PxVec3 impulse, float invMassScale, float invInertiaScale, PxVec3 deltaLinearVelocity, PxVec3 deltaAngularVelocity) {
        _computeVelocityDeltaFromImpulse(body.getAddress(), globalPose.getAddress(), point.getAddress(), impulse.getAddress(), invMassScale, invInertiaScale, deltaLinearVelocity.getAddress(), deltaAngularVelocity.getAddress());
    }
    private static native void _computeVelocityDeltaFromImpulse(long body, long globalPose, long point, long impulse, float invMassScale, float invInertiaScale, long deltaLinearVelocity, long deltaAngularVelocity);

    /**
     * Computes the linear and angular impulse vectors for a given impulse at a world space position taking a mass and inertia scale into account
     * <p>
     * This function is useful for extracting the respective linear and angular impulses from a contact or joint when the mass/inertia ratios have been adjusted.
     * @param body The rigid body
     * @param globalPose The body's world space transform
     * @param point The point in world space where the impulse is applied
     * @param impulse The impulse vector in world space
     * @param invMassScale The inverse mass scale
     * @param invInertiaScale The inverse inertia scale
     * @param linearImpulse The linear impulse
     * @param angularImpulse The angular impulse
     */
    public static void computeLinearAngularImpulse(PxRigidBody body, PxTransform globalPose, PxVec3 point, PxVec3 impulse, float invMassScale, float invInertiaScale, PxVec3 linearImpulse, PxVec3 angularImpulse) {
        _computeLinearAngularImpulse(body.getAddress(), globalPose.getAddress(), point.getAddress(), impulse.getAddress(), invMassScale, invInertiaScale, linearImpulse.getAddress(), angularImpulse.getAddress());
    }
    private static native void _computeLinearAngularImpulse(long body, long globalPose, long point, long impulse, float invMassScale, float invInertiaScale, long linearImpulse, long angularImpulse);

}
