package physx.extensions;

import physx.common.PxBase;

/**
 * A joint that connects two existing revolute joints and constrains their relative angular velocity and position with respect to each other.
 * @see PxJoint
 */
public class PxGearJoint extends PxJoint {

    protected PxGearJoint() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxGearJoint wrapPointer(long address) {
        return address != 0L ? new PxGearJoint(address) : null;
    }
    
    public static PxGearJoint arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxGearJoint(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Set the hinge/revolute joints connected by the gear joint.
     * <p>
     * The passed joints can be either PxRevoluteJoint, PxD6Joint or PxArticulationJointReducedCoordinate. 
     * The joints must define degrees of freedom around the twist axis.
     * <p>
     * Note that these joints are only used to compute the positional error correction term,
     * used to adjust potential drift between jointed actors. The gear joint can run without
     * calling this function, but in that case some visible overlap may develop over time between
     * the teeth of the gear meshes.
     * <p>
     * <b>Note:</b> Calling this function resets the internal positional error correction term.
     * @param hinge0  The first hinge joint
     * @param hinge1  The second hinge joint
     * @return  true if success
     */
    public boolean setHinges(PxBase hinge0, PxBase hinge1) {
        checkNotNull();
        return _setHinges(address, hinge0.getAddress(), hinge1.getAddress());
    }
    private static native boolean _setHinges(long address, long hinge0, long hinge1);

    /**
     * Set the desired gear ratio.
     * <p>
     * For two gears with n0 and n1 teeth respectively, the gear ratio is n0/n1.
     * <p>
     * <b>Note:</b> You may need to use a negative gear ratio if the joint frames of involved actors are not oriented in the same direction.
     * <p>
     * <b>Note:</b> Calling this function resets the internal positional error correction term.
     * @param ratio Desired ratio between the two hinges.
     */
    public void setGearRatio(float ratio) {
        checkNotNull();
        _setGearRatio(address, ratio);
    }
    private static native void _setGearRatio(long address, float ratio);

    /**
     * Get the gear ratio.
     * @return  Current ratio
     */
    public float getGearRatio() {
        checkNotNull();
        return _getGearRatio(address);
    }
    private static native float _getGearRatio(long address);

}
