package physx.extensions;


/**
 * parameters for configuring the drive model of a PxD6Joint
 * @see PxD6Joint
 */
public class PxD6JointDrive extends PxSpring {

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxD6JointDrive wrapPointer(long address) {
        return address != 0L ? new PxD6JointDrive(address) : null;
    }
    
    public static PxD6JointDrive arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxD6JointDrive(long address) {
        super(address);
    }

    // Constructors

    /**
     * default constructor for PxD6JointDrive.
     */
    public PxD6JointDrive() {
        address = _PxD6JointDrive();
    }
    private static native long _PxD6JointDrive();

    /**
     * constructor a PxD6JointDrive.
     * @param driveStiffness The stiffness of the drive spring.
     * @param driveDamping  The damping of the drive spring
     * @param driveForceLimit The maximum impulse or force that can be exerted by the drive
     */
    public PxD6JointDrive(float driveStiffness, float driveDamping, float driveForceLimit) {
        address = _PxD6JointDrive(driveStiffness, driveDamping, driveForceLimit);
    }
    private static native long _PxD6JointDrive(float driveStiffness, float driveDamping, float driveForceLimit);

    /**
     * constructor a PxD6JointDrive.
     * @param driveStiffness The stiffness of the drive spring.
     * @param driveDamping  The damping of the drive spring
     * @param driveForceLimit The maximum impulse or force that can be exerted by the drive
     * @param isAcceleration Whether the drive is an acceleration drive or a force drive
     */
    public PxD6JointDrive(float driveStiffness, float driveDamping, float driveForceLimit, boolean isAcceleration) {
        address = _PxD6JointDrive(driveStiffness, driveDamping, driveForceLimit, isAcceleration);
    }
    private static native long _PxD6JointDrive(float driveStiffness, float driveDamping, float driveForceLimit, boolean isAcceleration);

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * the force limit of the drive - may be an impulse or a force depending on PxConstraintFlag::eDRIVE_LIMITS_ARE_FORCES
     */
    public float getForceLimit() {
        checkNotNull();
        return _getForceLimit(address);
    }
    private static native float _getForceLimit(long address);

    /**
     * the force limit of the drive - may be an impulse or a force depending on PxConstraintFlag::eDRIVE_LIMITS_ARE_FORCES
     */
    public void setForceLimit(float value) {
        checkNotNull();
        _setForceLimit(address, value);
    }
    private static native void _setForceLimit(long address, float value);

    /**
     * the joint drive flags
     */
    public PxD6JointDriveFlags getFlags() {
        checkNotNull();
        return PxD6JointDriveFlags.wrapPointer(_getFlags(address));
    }
    private static native long _getFlags(long address);

    /**
     * the joint drive flags
     */
    public void setFlags(PxD6JointDriveFlags value) {
        checkNotNull();
        _setFlags(address, value.getAddress());
    }
    private static native void _setFlags(long address, long value);

}
