package physx.common;

import physx.NativeObject;

/**
 * Class to define the scale at which simulation runs. Most simulation tolerances are
 * calculated in terms of the values here. 
 * <p>
 * <b>Note:</b> if you change the simulation scale, you will probably also wish to change the scene's
 * default value of gravity, and stable simulation will probably require changes to the scene's 
 * bounceThreshold also.
 */
public class PxTolerancesScale extends NativeObject {

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxTolerancesScale wrapPointer(long address) {
        return address != 0L ? new PxTolerancesScale(address) : null;
    }
    
    public static PxTolerancesScale arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxTolerancesScale(long address) {
        super(address);
    }

    // Constructors

    /**
     * constructor sets to default 
     */
    public PxTolerancesScale() {
        address = _PxTolerancesScale();
    }
    private static native long _PxTolerancesScale();

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * The approximate size of objects in the simulation. 
     * <p>
     * For simulating roughly human-sized in metric units, 1 is a good choice.
     * If simulation is done in centimetres, use 100 instead. This is used to
     * estimate certain length-related tolerances.
     */
    public float getLength() {
        checkNotNull();
        return _getLength(address);
    }
    private static native float _getLength(long address);

    /**
     * The approximate size of objects in the simulation. 
     * <p>
     * For simulating roughly human-sized in metric units, 1 is a good choice.
     * If simulation is done in centimetres, use 100 instead. This is used to
     * estimate certain length-related tolerances.
     */
    public void setLength(float value) {
        checkNotNull();
        _setLength(address, value);
    }
    private static native void _setLength(long address, float value);

    /**
     * The typical magnitude of velocities of objects in simulation. This is used to estimate 
     * whether a contact should be treated as bouncing or resting based on its impact velocity,
     * and a kinetic energy threshold below which the simulation may put objects to sleep.
     * <p>
     * For normal physical environments, a good choice is the approximate speed of an object falling
     * under gravity for one second.
     */
    public float getSpeed() {
        checkNotNull();
        return _getSpeed(address);
    }
    private static native float _getSpeed(long address);

    /**
     * The typical magnitude of velocities of objects in simulation. This is used to estimate 
     * whether a contact should be treated as bouncing or resting based on its impact velocity,
     * and a kinetic energy threshold below which the simulation may put objects to sleep.
     * <p>
     * For normal physical environments, a good choice is the approximate speed of an object falling
     * under gravity for one second.
     */
    public void setSpeed(float value) {
        checkNotNull();
        _setSpeed(address, value);
    }
    private static native void _setSpeed(long address, float value);

    // Functions

    /**
     * Returns true if the descriptor is valid.
     * @return true if the current settings are valid (returns always true).
     */
    public boolean isValid() {
        checkNotNull();
        return _isValid(address);
    }
    private static native boolean _isValid(long address);

}
