package physx.common;

import physx.NativeObject;
import physx.PlatformChecks;

/**
 * Representation of a plane.
 * <p>
 *  Plane equation used: n.dot(v) + d = 0
 */
public class PxPlane extends NativeObject {

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxPlane wrapPointer(long address) {
        return address != 0L ? new PxPlane(address) : null;
    }
    
    public static PxPlane arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxPlane(long address) {
        super(address);
    }

    // Placed Constructors

    /**
     * @param address Pre-allocated memory, where the object is created.
     * @return Stack allocated object of PxPlane
     */
    public static PxPlane createAt(long address) {
        __placement_new_PxPlane(address);
        PxPlane createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    /**
     * @param <T>       Allocator class, e.g. LWJGL's MemoryStack.
     * @param allocator Object to use for allocation, e.g. an instance of LWJGL's MemoryStack.
     * @param allocate  Method to call on allocator to obtain the target address, e.g. MemoryStack::nmalloc.
     * @return Stack allocated object of PxPlane
     */
    public static <T> PxPlane createAt(T allocator, Allocator<T> allocate) {
        long address = allocate.on(allocator, ALIGNOF, SIZEOF); 
        __placement_new_PxPlane(address);
        PxPlane createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    private static native void __placement_new_PxPlane(long address);

    /**
     * @param address  Pre-allocated memory, where the object is created.
     * @param nx       WebIDL type: float
     * @param ny       WebIDL type: float
     * @param nz       WebIDL type: float
     * @param distance WebIDL type: float
     * @return Stack allocated object of PxPlane
     */
    public static PxPlane createAt(long address, float nx, float ny, float nz, float distance) {
        __placement_new_PxPlane(address, nx, ny, nz, distance);
        PxPlane createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    /**
     * @param <T>       Allocator class, e.g. LWJGL's MemoryStack.
     * @param allocator Object to use for allocation, e.g. an instance of LWJGL's MemoryStack.
     * @param allocate  Method to call on allocator to obtain the target address, e.g. MemoryStack::nmalloc.
     * @param nx        WebIDL type: float
     * @param ny        WebIDL type: float
     * @param nz        WebIDL type: float
     * @param distance  WebIDL type: float
     * @return Stack allocated object of PxPlane
     */
    public static <T> PxPlane createAt(T allocator, Allocator<T> allocate, float nx, float ny, float nz, float distance) {
        long address = allocate.on(allocator, ALIGNOF, SIZEOF); 
        __placement_new_PxPlane(address, nx, ny, nz, distance);
        PxPlane createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    private static native void __placement_new_PxPlane(long address, float nx, float ny, float nz, float distance);

    /**
     * @param address  Pre-allocated memory, where the object is created.
     * @param normal   WebIDL type: {@link PxVec3} [Const, Ref]
     * @param distance WebIDL type: float
     * @return Stack allocated object of PxPlane
     */
    public static PxPlane createAt(long address, PxVec3 normal, float distance) {
        __placement_new_PxPlane(address, normal.getAddress(), distance);
        PxPlane createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    /**
     * @param <T>       Allocator class, e.g. LWJGL's MemoryStack.
     * @param allocator Object to use for allocation, e.g. an instance of LWJGL's MemoryStack.
     * @param allocate  Method to call on allocator to obtain the target address, e.g. MemoryStack::nmalloc.
     * @param normal    WebIDL type: {@link PxVec3} [Const, Ref]
     * @param distance  WebIDL type: float
     * @return Stack allocated object of PxPlane
     */
    public static <T> PxPlane createAt(T allocator, Allocator<T> allocate, PxVec3 normal, float distance) {
        long address = allocate.on(allocator, ALIGNOF, SIZEOF); 
        __placement_new_PxPlane(address, normal.getAddress(), distance);
        PxPlane createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    private static native void __placement_new_PxPlane(long address, long normal, float distance);

    /**
     * @param address Pre-allocated memory, where the object is created.
     * @param p0      WebIDL type: {@link PxVec3} [Const, Ref]
     * @param p1      WebIDL type: {@link PxVec3} [Const, Ref]
     * @param p2      WebIDL type: {@link PxVec3} [Const, Ref]
     * @return Stack allocated object of PxPlane
     */
    public static PxPlane createAt(long address, PxVec3 p0, PxVec3 p1, PxVec3 p2) {
        __placement_new_PxPlane(address, p0.getAddress(), p1.getAddress(), p2.getAddress());
        PxPlane createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    /**
     * @param <T>       Allocator class, e.g. LWJGL's MemoryStack.
     * @param allocator Object to use for allocation, e.g. an instance of LWJGL's MemoryStack.
     * @param allocate  Method to call on allocator to obtain the target address, e.g. MemoryStack::nmalloc.
     * @param p0        WebIDL type: {@link PxVec3} [Const, Ref]
     * @param p1        WebIDL type: {@link PxVec3} [Const, Ref]
     * @param p2        WebIDL type: {@link PxVec3} [Const, Ref]
     * @return Stack allocated object of PxPlane
     */
    public static <T> PxPlane createAt(T allocator, Allocator<T> allocate, PxVec3 p0, PxVec3 p1, PxVec3 p2) {
        long address = allocate.on(allocator, ALIGNOF, SIZEOF); 
        __placement_new_PxPlane(address, p0.getAddress(), p1.getAddress(), p2.getAddress());
        PxPlane createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    private static native void __placement_new_PxPlane(long address, long p0, long p1, long p2);

    /**
     * @param address Pre-allocated memory, where the object is created.
     * @param point   WebIDL type: {@link PxVec3} [Const, Ref]
     * @param normal  WebIDL type: {@link PxVec3} [Const, Ref]
     * @return Stack allocated object of PxPlane
     */
    public static PxPlane createAt(long address, PxVec3 point, PxVec3 normal) {
    PlatformChecks.requirePlatform(15, "physx.common.PxPlane");
        __placement_new_PxPlane(address, point.getAddress(), normal.getAddress());
        PxPlane createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    /**
     * @param <T>       Allocator class, e.g. LWJGL's MemoryStack.
     * @param allocator Object to use for allocation, e.g. an instance of LWJGL's MemoryStack.
     * @param allocate  Method to call on allocator to obtain the target address, e.g. MemoryStack::nmalloc.
     * @param point     WebIDL type: {@link PxVec3} [Const, Ref]
     * @param normal    WebIDL type: {@link PxVec3} [Const, Ref]
     * @return Stack allocated object of PxPlane
     */
    public static <T> PxPlane createAt(T allocator, Allocator<T> allocate, PxVec3 point, PxVec3 normal) {
        long address = allocate.on(allocator, ALIGNOF, SIZEOF); 
        __placement_new_PxPlane(address, point.getAddress(), normal.getAddress());
        PxPlane createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    private static native void __placement_new_PxPlane(long address, long point, long normal);

    // Constructors

    /**
     * Constructor
     */
    public PxPlane() {
        address = _PxPlane();
    }
    private static native long _PxPlane();

    /**
     * Constructor from a normal and a distance
     */
    public PxPlane(float nx, float ny, float nz, float distance) {
        address = _PxPlane(nx, ny, nz, distance);
    }
    private static native long _PxPlane(float nx, float ny, float nz, float distance);

    /**
     * Constructor from a normal and a distance
     */
    public PxPlane(PxVec3 normal, float distance) {
        address = _PxPlane(normal.getAddress(), distance);
    }
    private static native long _PxPlane(long normal, float distance);

    /**
     * Constructor from three points
     */
    public PxPlane(PxVec3 p0, PxVec3 p1, PxVec3 p2) {
        address = _PxPlane(p0.getAddress(), p1.getAddress(), p2.getAddress());
    }
    private static native long _PxPlane(long p0, long p1, long p2);

    /**
     * Constructor from a normal and a distance
     */
    public PxPlane(PxVec3 point, PxVec3 normal) {
        PlatformChecks.requirePlatform(15, "physx.common.PxPlane");
        address = _PxPlane(point.getAddress(), normal.getAddress());
    }
    private static native long _PxPlane(long point, long normal);

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * The normal to the plane
     */
    public PxVec3 getN() {
        checkNotNull();
        return PxVec3.wrapPointer(_getN(address));
    }
    private static native long _getN(long address);

    /**
     * The normal to the plane
     */
    public void setN(PxVec3 value) {
        checkNotNull();
        _setN(address, value.getAddress());
    }
    private static native void _setN(long address, long value);

    /**
     * The distance from the origin
     */
    public float getD() {
        checkNotNull();
        return _getD(address);
    }
    private static native float _getD(long address);

    /**
     * The distance from the origin
     */
    public void setD(float value) {
        checkNotNull();
        _setD(address, value);
    }
    private static native void _setD(long address, float value);

    // Functions

    /**
     * @param p WebIDL type: {@link PxVec3} [Const, Ref]
     * @return WebIDL type: float
     */
    public float distance(PxVec3 p) {
        checkNotNull();
        return _distance(address, p.getAddress());
    }
    private static native float _distance(long address, long p);

    /**
     * @param p WebIDL type: {@link PxVec3} [Const, Ref]
     * @return WebIDL type: boolean
     */
    public boolean contains(PxVec3 p) {
        checkNotNull();
        return _contains(address, p.getAddress());
    }
    private static native boolean _contains(long address, long p);

    /**
     * projects p into the plane
     */
    public PxVec3 project(PxVec3 p) {
        checkNotNull();
        return PxVec3.wrapPointer(_project(address, p.getAddress()));
    }
    private static native long _project(long address, long p);

    /**
     * find an arbitrary point in the plane
     */
    public PxVec3 pointInPlane() {
        checkNotNull();
        return PxVec3.wrapPointer(_pointInPlane(address));
    }
    private static native long _pointInPlane(long address);

    /**
     * equivalent plane with unit normal
     */
    public void normalize() {
        checkNotNull();
        _normalize(address);
    }
    private static native void _normalize(long address);

    /**
     * transform plane
     */
    public PxPlane transform(PxTransform pose) {
        checkNotNull();
        return PxPlane.wrapPointer(_transform(address, pose.getAddress()));
    }
    private static native long _transform(long address, long pose);

    /**
     * inverse-transform plane
     */
    public PxPlane inverseTransform(PxTransform pose) {
        checkNotNull();
        return PxPlane.wrapPointer(_inverseTransform(address, pose.getAddress()));
    }
    private static native long _inverseTransform(long address, long pose);

}
