package physx.common;

import physx.NativeObject;
import physx.PlatformChecks;

/**
 * Sizes of pre-allocated buffers use for GPU dynamics
 */
public class PxGpuDynamicsMemoryConfig extends NativeObject {

    static {
        PlatformChecks.requirePlatform(3, "physx.common.PxGpuDynamicsMemoryConfig");
    }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxGpuDynamicsMemoryConfig wrapPointer(long address) {
        return address != 0L ? new PxGpuDynamicsMemoryConfig(address) : null;
    }
    
    public static PxGpuDynamicsMemoryConfig arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxGpuDynamicsMemoryConfig(long address) {
        super(address);
    }

    // Constructors

    public PxGpuDynamicsMemoryConfig() {
        address = _PxGpuDynamicsMemoryConfig();
    }
    private static native long _PxGpuDynamicsMemoryConfig();

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * Initial capacity of temp solver buffer allocated in pinned host memory. This buffer will grow if more memory is needed than specified here.
     */
    public int getTempBufferCapacity() {
        checkNotNull();
        return _getTempBufferCapacity(address);
    }
    private static native int _getTempBufferCapacity(long address);

    /**
     * Initial capacity of temp solver buffer allocated in pinned host memory. This buffer will grow if more memory is needed than specified here.
     */
    public void setTempBufferCapacity(int value) {
        checkNotNull();
        _setTempBufferCapacity(address, value);
    }
    private static native void _setTempBufferCapacity(long address, int value);

    /**
     * Size of contact stream buffer allocated in pinned host memory. This is double-buffered so total allocation size = 2* contactStreamCapacity * sizeof(PxContact).
     */
    public int getMaxRigidContactCount() {
        checkNotNull();
        return _getMaxRigidContactCount(address);
    }
    private static native int _getMaxRigidContactCount(long address);

    /**
     * Size of contact stream buffer allocated in pinned host memory. This is double-buffered so total allocation size = 2* contactStreamCapacity * sizeof(PxContact).
     */
    public void setMaxRigidContactCount(int value) {
        checkNotNull();
        _setMaxRigidContactCount(address, value);
    }
    private static native void _setMaxRigidContactCount(long address, int value);

    /**
     * Size of the contact patch stream buffer allocated in pinned host memory. This is double-buffered so total allocation size = 2 * patchStreamCapacity * sizeof(PxContactPatch).
     */
    public int getMaxRigidPatchCount() {
        checkNotNull();
        return _getMaxRigidPatchCount(address);
    }
    private static native int _getMaxRigidPatchCount(long address);

    /**
     * Size of the contact patch stream buffer allocated in pinned host memory. This is double-buffered so total allocation size = 2 * patchStreamCapacity * sizeof(PxContactPatch).
     */
    public void setMaxRigidPatchCount(int value) {
        checkNotNull();
        _setMaxRigidPatchCount(address, value);
    }
    private static native void _setMaxRigidPatchCount(long address, int value);

    /**
     * Initial capacity of the GPU and pinned host memory heaps. Additional memory will be allocated if more memory is required.
     */
    public int getHeapCapacity() {
        checkNotNull();
        return _getHeapCapacity(address);
    }
    private static native int _getHeapCapacity(long address);

    /**
     * Initial capacity of the GPU and pinned host memory heaps. Additional memory will be allocated if more memory is required.
     */
    public void setHeapCapacity(int value) {
        checkNotNull();
        _setHeapCapacity(address, value);
    }
    private static native void _setHeapCapacity(long address, int value);

    /**
     * Capacity of found and lost buffers allocated in GPU global memory. This is used for the found/lost pair reports in the BP.
     */
    public int getFoundLostPairsCapacity() {
        checkNotNull();
        return _getFoundLostPairsCapacity(address);
    }
    private static native int _getFoundLostPairsCapacity(long address);

    /**
     * Capacity of found and lost buffers allocated in GPU global memory. This is used for the found/lost pair reports in the BP.
     */
    public void setFoundLostPairsCapacity(int value) {
        checkNotNull();
        _setFoundLostPairsCapacity(address, value);
    }
    private static native void _setFoundLostPairsCapacity(long address, int value);

    /**
     * Capacity of found and lost buffers in aggregate system allocated in GPU global memory. This is used for the found/lost pair reports in AABB manager.
     */
    public int getFoundLostAggregatePairsCapacity() {
        checkNotNull();
        return _getFoundLostAggregatePairsCapacity(address);
    }
    private static native int _getFoundLostAggregatePairsCapacity(long address);

    /**
     * Capacity of found and lost buffers in aggregate system allocated in GPU global memory. This is used for the found/lost pair reports in AABB manager.
     */
    public void setFoundLostAggregatePairsCapacity(int value) {
        checkNotNull();
        _setFoundLostAggregatePairsCapacity(address, value);
    }
    private static native void _setFoundLostAggregatePairsCapacity(long address, int value);

    /**
     * Capacity of aggregate pair buffer allocated in GPU global memory.
     */
    public int getTotalAggregatePairsCapacity() {
        checkNotNull();
        return _getTotalAggregatePairsCapacity(address);
    }
    private static native int _getTotalAggregatePairsCapacity(long address);

    /**
     * Capacity of aggregate pair buffer allocated in GPU global memory.
     */
    public void setTotalAggregatePairsCapacity(int value) {
        checkNotNull();
        _setTotalAggregatePairsCapacity(address, value);
    }
    private static native void _setTotalAggregatePairsCapacity(long address, int value);

    /**
     * Capacity of deformable surface contact buffer allocated in GPU global memory.
     */
    public int getMaxDeformableSurfaceContacts() {
        checkNotNull();
        return _getMaxDeformableSurfaceContacts(address);
    }
    private static native int _getMaxDeformableSurfaceContacts(long address);

    /**
     * Capacity of deformable surface contact buffer allocated in GPU global memory.
     */
    public void setMaxDeformableSurfaceContacts(int value) {
        checkNotNull();
        _setMaxDeformableSurfaceContacts(address, value);
    }
    private static native void _setMaxDeformableSurfaceContacts(long address, int value);

    /**
     * Deprecated, replace with maxDeformableSurfaceContacts.
     */
    public int getMaxFemClothContacts() {
        checkNotNull();
        return _getMaxFemClothContacts(address);
    }
    private static native int _getMaxFemClothContacts(long address);

    /**
     * Deprecated, replace with maxDeformableSurfaceContacts.
     */
    public void setMaxFemClothContacts(int value) {
        checkNotNull();
        _setMaxFemClothContacts(address, value);
    }
    private static native void _setMaxFemClothContacts(long address, int value);

    /**
     * Capacity of deformable volume contact buffer allocated in GPU global memory.
     */
    public int getMaxDeformableVolumeContacts() {
        checkNotNull();
        return _getMaxDeformableVolumeContacts(address);
    }
    private static native int _getMaxDeformableVolumeContacts(long address);

    /**
     * Capacity of deformable volume contact buffer allocated in GPU global memory.
     */
    public void setMaxDeformableVolumeContacts(int value) {
        checkNotNull();
        _setMaxDeformableVolumeContacts(address, value);
    }
    private static native void _setMaxDeformableVolumeContacts(long address, int value);

    /**
     * Deprecated, replace with maxDeformableVolumeContacts.
     */
    public int getMaxSoftBodyContacts() {
        checkNotNull();
        return _getMaxSoftBodyContacts(address);
    }
    private static native int _getMaxSoftBodyContacts(long address);

    /**
     * Deprecated, replace with maxDeformableVolumeContacts.
     */
    public void setMaxSoftBodyContacts(int value) {
        checkNotNull();
        _setMaxSoftBodyContacts(address, value);
    }
    private static native void _setMaxSoftBodyContacts(long address, int value);

    /**
     * Capacity of particle contact buffer allocated in GPU global memory.
     */
    public int getMaxParticleContacts() {
        checkNotNull();
        return _getMaxParticleContacts(address);
    }
    private static native int _getMaxParticleContacts(long address);

    /**
     * Capacity of particle contact buffer allocated in GPU global memory.
     */
    public void setMaxParticleContacts(int value) {
        checkNotNull();
        _setMaxParticleContacts(address, value);
    }
    private static native void _setMaxParticleContacts(long address, int value);

    /**
     * Capacity of the collision stack buffer, used as scratch space during narrowphase collision detection.
     */
    public int getCollisionStackSize() {
        checkNotNull();
        return _getCollisionStackSize(address);
    }
    private static native int _getCollisionStackSize(long address);

    /**
     * Capacity of the collision stack buffer, used as scratch space during narrowphase collision detection.
     */
    public void setCollisionStackSize(int value) {
        checkNotNull();
        _setCollisionStackSize(address, value);
    }
    private static native void _setCollisionStackSize(long address, int value);

    // Functions

    /**
     * @return WebIDL type: boolean
     */
    public boolean isValid() {
        checkNotNull();
        return _isValid(address);
    }
    private static native boolean _isValid(long address);

}
