package physx.character;

import physx.NativeObject;

/**
 * User callback class for character controller events.
 * <p>
 * <b>Note:</b> Character controller hit reports are only generated when move is called.
 */
public class PxUserControllerHitReport extends NativeObject {

    protected PxUserControllerHitReport() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxUserControllerHitReport wrapPointer(long address) {
        return address != 0L ? new PxUserControllerHitReport(address) : null;
    }
    
    public static PxUserControllerHitReport arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxUserControllerHitReport(long address) {
        super(address);
    }

    // Functions

    /**
     * Called when current controller hits a shape.
     * <p>
     * This is called when the CCT moves and hits a shape. This will not be called when a moving shape hits a non-moving CCT.
     * @param hit Provides information about the hit.
     * @see PxControllerShapeHit
     */
    public void onShapeHit(PxControllerShapeHit hit) {
        checkNotNull();
        _onShapeHit(address, hit.getAddress());
    }
    private static native void _onShapeHit(long address, long hit);

    /**
     * Called when current controller hits another controller.
     * @param hit Provides information about the hit.
     * @see PxControllersHit
     */
    public void onControllerHit(PxControllersHit hit) {
        checkNotNull();
        _onControllerHit(address, hit.getAddress());
    }
    private static native void _onControllerHit(long address, long hit);

    /**
     * Called when current controller hits a user-defined obstacle.
     * @param hit Provides information about the hit.
     * @see PxControllerObstacleHit
     * @see PxObstacleContext
     */
    public void onObstacleHit(PxControllerObstacleHit hit) {
        checkNotNull();
        _onObstacleHit(address, hit.getAddress());
    }
    private static native void _onObstacleHit(long address, long hit);

}
