package physx.character;

import physx.NativeObject;
import physx.common.PxVec3;
import physx.physics.PxScene;

/**
 * Manages an array of character controllers.
 * @see PxController
 * @see PxBoxController
 * @see PxCapsuleController
 */
public class PxControllerManager extends NativeObject {

    protected PxControllerManager() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxControllerManager wrapPointer(long address) {
        return address != 0L ? new PxControllerManager(address) : null;
    }
    
    public static PxControllerManager arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxControllerManager(long address) {
        super(address);
    }

    // Functions

    /**
     * Releases the controller manager.
     * <p>
     * <b>Note:</b> This will release all associated controllers and obstacle contexts.
     * <p>
     * <b>Note:</b> This function is required to be called to release foundation usage.
     */
    public void release() {
        checkNotNull();
        _release(address);
    }
    private static native void _release(long address);

    /**
     * Returns the scene the manager is adding the controllers to.
     * @return The associated physics scene.
     */
    public PxScene getScene() {
        checkNotNull();
        return PxScene.wrapPointer(_getScene(address));
    }
    private static native long _getScene(long address);

    /**
     * Returns the number of controllers that are being managed.
     * @return The number of controllers.
     */
    public int getNbControllers() {
        checkNotNull();
        return _getNbControllers(address);
    }
    private static native int _getNbControllers(long address);

    /**
     * Retrieve one of the controllers in the manager.
     * <p>
     * \param index the index of the controller to return
     * @return The controller with the specified index.
     */
    public PxController getController(int index) {
        checkNotNull();
        return PxController.wrapPointer(_getController(address, index));
    }
    private static native long _getController(long address, int index);

    /**
     * Creates a new character controller.
     * @param desc The controllers descriptor
     * @return The new controller
     * @see PxController
     * @see PxController#release
     * @see PxControllerDesc
     */
    public PxController createController(PxControllerDesc desc) {
        checkNotNull();
        return PxController.wrapPointer(_createController(address, desc.getAddress()));
    }
    private static native long _createController(long address, long desc);

    /**
     * Releases all the controllers that are being managed.
     */
    public void purgeControllers() {
        checkNotNull();
        _purgeControllers(address);
    }
    private static native void _purgeControllers(long address);

    /**
     * Returns the number of obstacle contexts that are being managed.
     * @return The number of obstacle contexts.
     */
    public int getNbObstacleContexts() {
        checkNotNull();
        return _getNbObstacleContexts(address);
    }
    private static native int _getNbObstacleContexts(long address);

    /**
     * Retrieve one of the obstacle contexts in the manager.
     * <p>
     * \param index The index of the obstacle context to retrieve.
     * @return The obstacle context with the specified index.
     */
    public PxObstacleContext getObstacleContext(int index) {
        checkNotNull();
        return PxObstacleContext.wrapPointer(_getObstacleContext(address, index));
    }
    private static native long _getObstacleContext(long address, int index);

    /**
     * Creates an obstacle context.
     * @return New obstacle context
     * @see PxObstacleContext
     */
    public PxObstacleContext createObstacleContext() {
        checkNotNull();
        return PxObstacleContext.wrapPointer(_createObstacleContext(address));
    }
    private static native long _createObstacleContext(long address);

    /**
     * Computes character-character interactions.
     * <p>
     * This function is an optional helper to properly resolve interactions between characters, in case they overlap (which can happen for gameplay reasons, etc).
     * <p>
     * You should call this once per frame, before your PxController::move() calls. The function will not move the characters directly, but it will
     * compute overlap information for each character that will be used in the next move() call.
     * <p>
     * You need to provide a proper time value here so that interactions are resolved in a way that do not depend on the framerate.
     * <p>
     * If you only have one character in the scene, or if you can guarantee your characters will never overlap, then you do not need to call this function.
     * <p>
     * <b>Note:</b> Releasing the manager will automatically release all the associated obstacle contexts.
     * @param elapsedTime Elapsed time since last call
     */
    public void computeInteractions(float elapsedTime) {
        checkNotNull();
        _computeInteractions(address, elapsedTime);
    }
    private static native void _computeInteractions(long address, float elapsedTime);

    /**
     * Enables or disables runtime tessellation.
     * <p>
     * Large triangles can create accuracy issues in the sweep code, which in turn can lead to characters not sliding smoothly
     * against geometries, or even penetrating them. This feature allows one to reduce those issues by tessellating large
     * triangles at runtime, before performing sweeps against them. The amount of tessellation is controlled by the 'maxEdgeLength' parameter.
     * Any triangle with at least one edge length greater than the maxEdgeLength will get recursively tessellated, until resulting triangles are small enough.
     * <p>
     * This features only applies to triangle meshes, convex meshes, heightfields and boxes.
     * @param flag    True/false to enable/disable runtime tessellation.
     * @param maxEdgeLength Max edge length allowed before tessellation kicks in.
     */
    public void setTessellation(boolean flag, float maxEdgeLength) {
        checkNotNull();
        _setTessellation(address, flag, maxEdgeLength);
    }
    private static native void _setTessellation(long address, boolean flag, float maxEdgeLength);

    /**
     * Enables or disables the overlap recovery module.
     * <p>
     * The overlap recovery module can be used to depenetrate CCTs from static objects when an overlap is detected. This can happen
     * in three main cases:
     *  - when the CCT is directly spawned or teleported in another object
     *  - when the CCT algorithm fails due to limited FPU accuracy
     *  - when the "up vector" is modified, making the rotated CCT shape overlap surrounding objects
     * <p>
     * When activated, the CCT module will automatically try to resolve the penetration, and move the CCT to a safe place where it does
     * not overlap other objects anymore. This only concerns static objects, dynamic objects are ignored by the recovery module.
     * <p>
     * When the recovery module is not activated, it is possible for the CCTs to go through static objects. By default, the recovery
     * module is enabled.
     * <p>
     * The recovery module currently works with all geometries except heightfields.
     * @param flag    True/false to enable/disable overlap recovery module.
     */
    public void setOverlapRecoveryModule(boolean flag) {
        checkNotNull();
        _setOverlapRecoveryModule(address, flag);
    }
    private static native void _setOverlapRecoveryModule(long address, boolean flag);

    /**
     * Enables or disables the precise sweeps.
     * <p>
     * Precise sweeps are more accurate, but also potentially slower than regular sweeps.
     * <p>
     * By default, precise sweeps are enabled.
     */
    public void setPreciseSweeps(boolean flags) {
        checkNotNull();
        _setPreciseSweeps(address, flags);
    }
    private static native void _setPreciseSweeps(long address, boolean flags);

    /**
     * Enables or disables vertical sliding against ceilings.
     * <p>
     * Geometry is seen as "ceilings" when the following condition is met:
     * <p>
     *  dot product(contact normal, up direction)&lt;0.0f
     * <p>
     * This flag controls whether characters should slide vertically along the geometry in that case.
     * <p>
     * By default, sliding is allowed.
     * @param flag    True/false to enable/disable sliding.
     */
    public void setPreventVerticalSlidingAgainstCeiling(boolean flag) {
        checkNotNull();
        _setPreventVerticalSlidingAgainstCeiling(address, flag);
    }
    private static native void _setPreventVerticalSlidingAgainstCeiling(long address, boolean flag);

    /**
     * Shift the origin of the character controllers and obstacle objects by the specified vector.
     * <p>
     * The positions of all character controllers, obstacle objects and the corresponding data structures will get adjusted to reflect the shifted origin location
     * (the shift vector will get subtracted from all character controller and obstacle object positions).
     * <p>
     * <b>Note:</b> It is the user's responsibility to keep track of the summed total origin shift and adjust all input/output to/from PhysXCharacterKinematic accordingly.
     * <p>
     * <b>Note:</b> This call will not automatically shift the PhysX scene and its objects. You need to call PxScene::shiftOrigin() separately to keep the systems in sync.
     * @param shift Translation vector to shift the origin by.
     */
    public void shiftOrigin(PxVec3 shift) {
        checkNotNull();
        _shiftOrigin(address, shift.getAddress());
    }
    private static native void _shiftOrigin(long address, long shift);

}
