package physx.character;

import physx.NativeObject;

/**
 * Dedicated filtering callback for CCT vs CCT.
 * <p>
 * This controls collisions between CCTs (one CCT vs anoter CCT).
 * <p>
 * To make each CCT collide against all other CCTs, just return true - or simply avoid defining a callback.
 * To make each CCT freely go through all other CCTs, just return false.
 * Otherwise create a custom filtering logic in this callback.
 * @see PxControllerFilters
 */
public class PxControllerFilterCallback extends NativeObject {

    protected PxControllerFilterCallback() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxControllerFilterCallback wrapPointer(long address) {
        return address != 0L ? new PxControllerFilterCallback(address) : null;
    }
    
    public static PxControllerFilterCallback arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxControllerFilterCallback(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Filtering method for CCT-vs-CCT.
     * @param a First CCT
     * @param b Second CCT
     * @return true to keep the pair, false to filter it out
     */
    public boolean filter(PxController a, PxController b) {
        checkNotNull();
        return _filter(address, a.getAddress(), b.getAddress());
    }
    private static native boolean _filter(long address, long a, long b);

}
