package physx.character;

import physx.NativeObject;
import physx.common.PxVec3;
import physx.physics.PxRigidDynamic;
import physx.physics.PxScene;

/**
 * Base class for character controllers.
 * @see PxCapsuleController
 * @see PxBoxController
 */
public class PxController extends NativeObject {

    protected PxController() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxController wrapPointer(long address) {
        return address != 0L ? new PxController(address) : null;
    }
    
    public static PxController arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxController(long address) {
        super(address);
    }

    // Functions

    /**
     * Return the type of controller
     */
    public PxControllerShapeTypeEnum getType() {
        checkNotNull();
        return PxControllerShapeTypeEnum.forValue(_getType(address));
    }
    private static native int _getType(long address);

    /**
     * Releases the controller.
     */
    public void release() {
        checkNotNull();
        _release(address);
    }
    private static native void _release(long address);

    /**
     * Moves the character using a "collide-and-slide" algorithm.
     * @param disp Displacement vector
     * @param minDist The minimum travelled distance to consider. If travelled distance is smaller, the character doesn't move.
     * This is used to stop the recursive motion algorithm when remaining distance to travel is small.
     * @param elapsedTime Time elapsed since last call
     * @param filters User-defined filters for this move
     * @return Collision flags, collection of ::PxControllerCollisionFlags
     */
    public PxControllerCollisionFlags move(PxVec3 disp, float minDist, float elapsedTime, PxControllerFilters filters) {
        checkNotNull();
        return PxControllerCollisionFlags.wrapPointer(_move(address, disp.getAddress(), minDist, elapsedTime, filters.getAddress()));
    }
    private static native long _move(long address, long disp, float minDist, float elapsedTime, long filters);

    /**
     * Moves the character using a "collide-and-slide" algorithm.
     * @param disp Displacement vector
     * @param minDist The minimum travelled distance to consider. If travelled distance is smaller, the character doesn't move.
     * This is used to stop the recursive motion algorithm when remaining distance to travel is small.
     * @param elapsedTime Time elapsed since last call
     * @param filters User-defined filters for this move
     * @param obstacles Potential additional obstacles the CCT should collide with.
     * @return Collision flags, collection of ::PxControllerCollisionFlags
     */
    public PxControllerCollisionFlags move(PxVec3 disp, float minDist, float elapsedTime, PxControllerFilters filters, PxObstacleContext obstacles) {
        checkNotNull();
        return PxControllerCollisionFlags.wrapPointer(_move(address, disp.getAddress(), minDist, elapsedTime, filters.getAddress(), obstacles.getAddress()));
    }
    private static native long _move(long address, long disp, float minDist, float elapsedTime, long filters, long obstacles);

    /**
     * Sets controller's position.
     * <p>
     * The position controlled by this function is the center of the collision shape.
     * <p>
     * \warning This is a 'teleport' function, it doesn't check for collisions.
     * \warning The character's position must be such that it does not overlap the static geometry.
     * <p>
     * To move the character under normal conditions use the #move() function.
     * @param position The new (center) positon for the controller.
     * @return Currently always returns true.
     * @see #getPosition
     * @see #getFootPosition
     * @see #setFootPosition
     * @see #move
     */
    public boolean setPosition(PxExtendedVec3 position) {
        checkNotNull();
        return _setPosition(address, position.getAddress());
    }
    private static native boolean _setPosition(long address, long position);

    /**
     * Retrieve the raw position of the controller.
     * <p>
     * The position retrieved by this function is the center of the collision shape. To retrieve the bottom position of the shape,
     * a.k.a. the foot position, use the getFootPosition() function.
     * <p>
     * The position is updated by calls to move(). Calling this method without calling
     * move() will return the last position or the initial position of the controller.
     * @return The controller's center position
     * @see #setPosition
     * @see #getFootPosition
     * @see #setFootPosition
     * @see #move
     */
    public PxExtendedVec3 getPosition() {
        checkNotNull();
        return PxExtendedVec3.wrapPointer(_getPosition(address));
    }
    private static native long _getPosition(long address);

    /**
     * Set controller's foot position.
     * <p>
     * The position controlled by this function is the bottom of the collision shape, a.k.a. the foot position.
     * <p>
     * <b>Note:</b> The foot position takes the contact offset into account
     * <p>
     * \warning This is a 'teleport' function, it doesn't check for collisions.
     * <p>
     * To move the character under normal conditions use the #move() function.
     * @param position The new (bottom) positon for the controller.
     * @return Currently always returns true.
     * @see #setPosition
     * @see #getPosition
     * @see #getFootPosition
     * @see #move
     */
    public boolean setFootPosition(PxExtendedVec3 position) {
        checkNotNull();
        return _setFootPosition(address, position.getAddress());
    }
    private static native boolean _setFootPosition(long address, long position);

    /**
     * Retrieve the "foot" position of the controller, i.e. the position of the bottom of the CCT's shape.
     * <p>
     * <b>Note:</b> The foot position takes the contact offset into account
     * @return The controller's foot position
     * @see #setPosition
     * @see #getPosition
     * @see #setFootPosition
     * @see #move
     */
    public PxExtendedVec3 getFootPosition() {
        checkNotNull();
        return PxExtendedVec3.wrapPointer(_getFootPosition(address));
    }
    private static native long _getFootPosition(long address);

    /**
     * Get the rigid body actor associated with this controller (see PhysX documentation). 
     * The behavior upon manually altering this actor is undefined, you should primarily 
     * use it for reading const properties.
     * @return the actor associated with the controller.
     */
    public PxRigidDynamic getActor() {
        checkNotNull();
        return PxRigidDynamic.wrapPointer(_getActor(address));
    }
    private static native long _getActor(long address);

    /**
     * The step height.
     * @param offset The new step offset for the controller.
     */
    public void setStepOffset(float offset) {
        checkNotNull();
        _setStepOffset(address, offset);
    }
    private static native void _setStepOffset(long address, float offset);

    /**
     * Retrieve the step height.
     * @return The step offset for the controller.
     * @see #setStepOffset
     */
    public float getStepOffset() {
        checkNotNull();
        return _getStepOffset(address);
    }
    private static native float _getStepOffset(long address);

    /**
     * Sets the non-walkable mode for the CCT.
     * @param flag The new value of the non-walkable mode.
     */
    public void setNonWalkableMode(PxControllerNonWalkableModeEnum flag) {
        checkNotNull();
        _setNonWalkableMode(address, flag.value);
    }
    private static native void _setNonWalkableMode(long address, int flag);

    /**
     * Retrieves the non-walkable mode for the CCT.
     * @return The current non-walkable mode.
     */
    public PxControllerNonWalkableModeEnum getNonWalkableMode() {
        checkNotNull();
        return PxControllerNonWalkableModeEnum.forValue(_getNonWalkableMode(address));
    }
    private static native int _getNonWalkableMode(long address);

    /**
     * Retrieve the contact offset.
     * @return The contact offset for the controller.
     */
    public float getContactOffset() {
        checkNotNull();
        return _getContactOffset(address);
    }
    private static native float _getContactOffset(long address);

    /**
     * Sets the contact offset.
     * @param offset The contact offset for the controller.
     */
    public void setContactOffset(float offset) {
        checkNotNull();
        _setContactOffset(address, offset);
    }
    private static native void _setContactOffset(long address, float offset);

    /**
     * Retrieve the 'up' direction.
     * @return The up direction for the controller.
     */
    public PxVec3 getUpDirection() {
        checkNotNull();
        return PxVec3.wrapPointer(_getUpDirection(address));
    }
    private static native long _getUpDirection(long address);

    /**
     * Sets the 'up' direction.
     * @param up The up direction for the controller.
     */
    public void setUpDirection(PxVec3 up) {
        checkNotNull();
        _setUpDirection(address, up.getAddress());
    }
    private static native void _setUpDirection(long address, long up);

    /**
     * Retrieve the slope limit.
     * @return The slope limit for the controller.
     */
    public float getSlopeLimit() {
        checkNotNull();
        return _getSlopeLimit(address);
    }
    private static native float _getSlopeLimit(long address);

    /**
     * Sets the slope limit.
     * <p>
     * <b>Note:</b> This feature can not be enabled at runtime, i.e. if the slope limit is zero when creating the CCT
     * (which disables the feature) then changing the slope limit at runtime will not have any effect, and the call
     * will be ignored.
     * @param slopeLimit The slope limit for the controller.
     */
    public void setSlopeLimit(float slopeLimit) {
        checkNotNull();
        _setSlopeLimit(address, slopeLimit);
    }
    private static native void _setSlopeLimit(long address, float slopeLimit);

    /**
     * Flushes internal geometry cache.
     * <p>
     * The character controller uses caching in order to speed up collision testing. The cache is
     * automatically flushed when a change to static objects is detected in the scene. For example when a
     * static shape is added, updated, or removed from the scene, the cache is automatically invalidated.
     * <p>
     * However there may be situations that cannot be automatically detected, and those require manual
     * invalidation of the cache. Currently the user must call this when the filtering behavior changes (the
     * PxControllerFilters parameter of the PxController::move call).  While the controller in principle 
     * could detect a change in these parameters, it cannot detect a change in the behavior of the filtering 
     * function.
     * @see PxController#move
     */
    public void invalidateCache() {
        checkNotNull();
        _invalidateCache(address);
    }
    private static native void _invalidateCache(long address);

    /**
     * Retrieve the scene associated with the controller.
     * @return The physics scene
     */
    public PxScene getScene() {
        checkNotNull();
        return PxScene.wrapPointer(_getScene(address));
    }
    private static native long _getScene(long address);

    /**
     * Returns the user data associated with this controller.
     * @return The user pointer associated with the controller.
     */
    public NativeObject getUserData() {
        checkNotNull();
        return NativeObject.wrapPointer(_getUserData(address));
    }
    private static native long _getUserData(long address);

    /**
     * Sets the user data associated with this controller.
     * @param userData The user pointer associated with the controller.
     */
    public void setUserData(NativeObject userData) {
        checkNotNull();
        _setUserData(address, userData.getAddress());
    }
    private static native void _setUserData(long address, long userData);

    /**
     * Returns information about the controller's internal state.
     * @param state The controller's internal state
     * @see PxControllerState
     */
    public void getState(PxControllerState state) {
        checkNotNull();
        _getState(address, state.getAddress());
    }
    private static native void _getState(long address, long state);

    /**
     * Returns the controller's internal statistics.
     * @param stats The controller's internal statistics
     * @see PxControllerStats
     */
    public void getStats(PxControllerStats stats) {
        checkNotNull();
        _getStats(address, stats.getAddress());
    }
    private static native void _getStats(long address, long stats);

    /**
     * Resizes the controller.
     * <p>
     * This function attempts to resize the controller to a given size, while making sure the bottom
     * position of the controller remains constant. In other words the function modifies both the
     * height and the (center) position of the controller. This is a helper function that can be used
     * to implement a 'crouch' functionality for example.
     * @param height Desired controller's height
     */
    public void resize(float height) {
        checkNotNull();
        _resize(address, height);
    }
    private static native void _resize(long address, float height);

}
