package physx.physics;

import physx.common.PxVec3;

/**
 * A spatial tendon that attaches to an articulation.
 * <p>
 * A spatial tendon attaches to multiple links in an articulation using a set of PxArticulationAttachments.
 * The tendon is defined as a tree of attachment points, where each attachment can have an arbitrary number of children.
 * Each leaf of the attachment tree defines a subtendon between itself and the root attachment. The subtendon then
 * applies forces at the leaf, and an equal but opposing force at the root, in order to satisfy the spring-damper and limit
 * constraints that the user sets up. Attachments in between the root and leaf do not exert any force on the articulation,
 * but define the geometry of the tendon from which the length is computed together with the attachment coefficients.
 */
public class PxArticulationSpatialTendon extends PxArticulationTendon {

    protected PxArticulationSpatialTendon() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxArticulationSpatialTendon wrapPointer(long address) {
        return address != 0L ? new PxArticulationSpatialTendon(address) : null;
    }
    
    public static PxArticulationSpatialTendon arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxArticulationSpatialTendon(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Creates an articulation attachment and adds it to the list of children in the parent attachment.
     * <p>
     * Creating an attachment is not allowed while the articulation is in a scene. In order to
     * add the attachment, remove and then re-add the articulation to the scene.
     * @param parent The parent attachment. Can be NULL for the root attachment of a tendon.
     * @param coefficient A user-defined scale that the accumulated length is scaled by.
     * @param relativeOffset An offset vector in the link's actor frame to the point where the tendon attachment is attached to the link.
     * @param link The link that this attachment is associated with.
     * @return The newly-created attachment if creation was successful, otherwise a null pointer.
     */
    public PxArticulationAttachment createAttachment(PxArticulationAttachment parent, float coefficient, PxVec3 relativeOffset, PxArticulationLink link) {
        checkNotNull();
        return PxArticulationAttachment.wrapPointer(_createAttachment(address, parent.getAddress(), coefficient, relativeOffset.getAddress(), link.getAddress()));
    }
    private static native long _createAttachment(long address, long parent, float coefficient, long relativeOffset, long link);

    /**
     * Returns the number of attachments in the tendon.
     * @return The number of attachments.
     */
    public int getNbAttachments() {
        checkNotNull();
        return _getNbAttachments(address);
    }
    private static native int _getNbAttachments(long address);

}
