package physx.particles;

import physx.PlatformChecks;
import physx.common.PxVec3;

/**
 * A particle system that uses the position based dynamics(PBD) solver.
 * <p>
 * The position based dynamics solver for particle systems supports behaviors like
 * fluid, cloth, inflatables etc. 
 */
public class PxPBDParticleSystem extends PxParticleSystem {

    static {
        PlatformChecks.requirePlatform(3, "physx.particles.PxPBDParticleSystem");
    }

    protected PxPBDParticleSystem() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxPBDParticleSystem wrapPointer(long address) {
        return address != 0L ? new PxPBDParticleSystem(address) : null;
    }
    
    public static PxPBDParticleSystem arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxPBDParticleSystem(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Set wind direction and intensity
     * @param wind The wind direction and intensity
     */
    public void setWind(PxVec3 wind) {
        checkNotNull();
        _setWind(address, wind.getAddress());
    }
    private static native void _setWind(long address, long wind);

    /**
     * Retrieves the wind direction and intensity.
     * @return The wind direction and intensity
     */
    public PxVec3 getWind() {
        checkNotNull();
        return PxVec3.wrapPointer(_getWind(address));
    }
    private static native long _getWind(long address);

    /**
     * Set the fluid boundary density scale
     * <p>
     * Defines how strong of a contribution the boundary (typically a rigid surface) should have on a fluid particle's density.
     * @param fluidBoundaryDensityScale  <b>Range:</b> (0.0, 1.0)
     */
    public void setFluidBoundaryDensityScale(float fluidBoundaryDensityScale) {
        checkNotNull();
        _setFluidBoundaryDensityScale(address, fluidBoundaryDensityScale);
    }
    private static native void _setFluidBoundaryDensityScale(long address, float fluidBoundaryDensityScale);

    /**
     * Return the fluid boundary density scale
     * @return the fluid boundary density scale
     * <p>
     * See #setFluidBoundaryDensityScale()
     */
    public float getFluidBoundaryDensityScale() {
        checkNotNull();
        return _getFluidBoundaryDensityScale(address);
    }
    private static native float _getFluidBoundaryDensityScale(long address);

    /**
     * Set the fluid rest offset
     * <p>
     * Two fluid particles will come to rest at a distance equal to twice the fluidRestOffset value.
     * @param fluidRestOffset  <b>Range:</b> (0, particleContactOffset)
     */
    public void setFluidRestOffset(float fluidRestOffset) {
        checkNotNull();
        _setFluidRestOffset(address, fluidRestOffset);
    }
    private static native void _setFluidRestOffset(long address, float fluidRestOffset);

    /**
     * Return the fluid rest offset
     * @return the fluid rest offset
     * <p>
     * See #setFluidRestOffset()
     */
    public float getFluidRestOffset() {
        checkNotNull();
        return _getFluidRestOffset(address);
    }
    private static native float _getFluidRestOffset(long address);

    /**
     * Set the particle system grid size x dimension
     * @param gridSizeX x dimension in the particle grid
     */
    public void setGridSizeX(int gridSizeX) {
        checkNotNull();
        _setGridSizeX(address, gridSizeX);
    }
    private static native void _setGridSizeX(long address, int gridSizeX);

    /**
     * Set the particle system grid size y dimension
     * @param gridSizeY y dimension in the particle grid
     */
    public void setGridSizeY(int gridSizeY) {
        checkNotNull();
        _setGridSizeY(address, gridSizeY);
    }
    private static native void _setGridSizeY(long address, int gridSizeY);

    /**
     * Set the particle system grid size z dimension
     * @param gridSizeZ z dimension in the particle grid
     */
    public void setGridSizeZ(int gridSizeZ) {
        checkNotNull();
        _setGridSizeZ(address, gridSizeZ);
    }
    private static native void _setGridSizeZ(long address, int gridSizeZ);

    /**
     * @return WebIDL type: DOMString
     */
    public String getConcreteTypeName() {
        checkNotNull();
        return _getConcreteTypeName(address);
    }
    private static native String _getConcreteTypeName(long address);

}
