package physx.extensions;

import physx.NativeObject;
import physx.common.PxVec3;

/**
 * Abstract interface for a user defined shape GJK mapping support.
 * A user defined shape consists of a core shape and a margin. If the distance
 * between two shapes' cores is equal to the sum of their margins, these shapes are
 * considered touching.
 */
public class Support extends NativeObject {

    protected Support() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static Support wrapPointer(long address) {
        return address != 0L ? new Support(address) : null;
    }
    
    public static Support arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected Support(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Return the user defined shape margin. Margin should be greater than or equal to 0
     * @return  Margin.
     */
    public float getMargin() {
        checkNotNull();
        return _getMargin(address);
    }
    private static native float _getMargin(long address);

    /**
     * Return the farthest point on the user defined shape's core in given direction.
     * @param dir  Direction
     * @return    Farthest point in given direction.
     */
    public PxVec3 supportLocal(PxVec3 dir) {
        checkNotNull();
        return PxVec3.wrapPointer(_supportLocal(address, dir.getAddress()));
    }
    private static native long _supportLocal(long address, long dir);

}
