package physx.extensions;

import physx.NativeObject;
import physx.common.PxTransform;
import physx.geometry.PxGeometry;
import physx.geometry.PxTriangleMeshGeometry;
import physx.support.PxU32ConstPtr;

/**
 * Utility class to find mesh triangles touched by a specified geometry object.
 * <p>
 * This class is a helper calling PxMeshQuery::findOverlapTriangleMesh or PxMeshQuery::findOverlapHeightField under the hood,
 * while taking care of necessary memory management issues.
 * <p>
 * PxMeshQuery::findOverlapTriangleMesh and PxMeshQuery::findOverlapHeightField are the "raw" functions operating on user-provided fixed-size
 * buffers. These functions abort with an error code in case of buffer overflow. PxMeshOverlapUtil is a convenient helper function checking
 * this error code, and resizing buffers appropriately, until the desired call succeeds.
 * <p>
 * Returned triangle indices are stored within the class, and can be used with PxMeshQuery::getTriangle() to retrieve the triangle properties.
 */
public class PxMeshOverlapUtil extends NativeObject {

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxMeshOverlapUtil wrapPointer(long address) {
        return address != 0L ? new PxMeshOverlapUtil(address) : null;
    }
    
    public static PxMeshOverlapUtil arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxMeshOverlapUtil(long address) {
        super(address);
    }

    // Constructors

    public PxMeshOverlapUtil() {
        address = _PxMeshOverlapUtil();
    }
    private static native long _PxMeshOverlapUtil();

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Find the mesh triangles which touch the specified geometry object.
     * @param geom The geometry object to test for mesh triangle overlaps. Supported geometries are #PxSphereGeometry, #PxCapsuleGeometry and #PxBoxGeometry
     * @param geomPose Pose of the geometry object
     * @param meshGeom The triangle mesh geometry to check overlap against
     * @param meshPose Pose of the triangle mesh
     * @return Number of overlaps found. Triangle indices can then be accessed through the #getResults() function.
     * @see physx.geometry.PxGeometry
     * @see physx.common.PxTransform
     * @see physx.geometry.PxTriangleMeshGeometry
     */
    public int findOverlap(PxGeometry geom, PxTransform geomPose, PxTriangleMeshGeometry meshGeom, PxTransform meshPose) {
        checkNotNull();
        return _findOverlap(address, geom.getAddress(), geomPose.getAddress(), meshGeom.getAddress(), meshPose.getAddress());
    }
    private static native int _findOverlap(long address, long geom, long geomPose, long meshGeom, long meshPose);

    /**
     * Retrieves array of triangle indices after a findOverlap call.
     * @return Indices of touched triangles
     */
    public PxU32ConstPtr getResults() {
        checkNotNull();
        return PxU32ConstPtr.wrapPointer(_getResults(address));
    }
    private static native long _getResults(long address);

    /**
     * Retrieves number of triangle indices after a findOverlap call.
     * @return Number of touched triangles
     */
    public int getNbResults() {
        checkNotNull();
        return _getNbResults(address);
    }
    private static native int _getNbResults(long address);

}
