package physx.extensions;

import physx.common.PxVec3;

/**
 * Pre-made support mapping for a box
 */
public class BoxSupport extends Support {

    protected BoxSupport() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static BoxSupport wrapPointer(long address) {
        return address != 0L ? new BoxSupport(address) : null;
    }
    
    public static BoxSupport arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected BoxSupport(long address) {
        super(address);
    }

    // Placed Constructors

    /**
     * @param address     Pre-allocated memory, where the object is created.
     * @param halfExtents WebIDL type: {@link PxVec3} [Const, Ref]
     * @return Stack allocated object of BoxSupport
     */
    public static BoxSupport createAt(long address, PxVec3 halfExtents) {
        __placement_new_BoxSupport(address, halfExtents.getAddress());
        BoxSupport createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    /**
     * @param <T>         Allocator class, e.g. LWJGL's MemoryStack.
     * @param allocator   Object to use for allocation, e.g. an instance of LWJGL's MemoryStack.
     * @param allocate    Method to call on allocator to obtain the target address, e.g. MemoryStack::nmalloc.
     * @param halfExtents WebIDL type: {@link PxVec3} [Const, Ref]
     * @return Stack allocated object of BoxSupport
     */
    public static <T> BoxSupport createAt(T allocator, Allocator<T> allocate, PxVec3 halfExtents) {
        long address = allocate.on(allocator, ALIGNOF, SIZEOF); 
        __placement_new_BoxSupport(address, halfExtents.getAddress());
        BoxSupport createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    private static native void __placement_new_BoxSupport(long address, long halfExtents);

    /**
     * @param address     Pre-allocated memory, where the object is created.
     * @param halfExtents WebIDL type: {@link PxVec3} [Const, Ref]
     * @param margin      WebIDL type: float
     * @return Stack allocated object of BoxSupport
     */
    public static BoxSupport createAt(long address, PxVec3 halfExtents, float margin) {
        __placement_new_BoxSupport(address, halfExtents.getAddress(), margin);
        BoxSupport createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    /**
     * @param <T>         Allocator class, e.g. LWJGL's MemoryStack.
     * @param allocator   Object to use for allocation, e.g. an instance of LWJGL's MemoryStack.
     * @param allocate    Method to call on allocator to obtain the target address, e.g. MemoryStack::nmalloc.
     * @param halfExtents WebIDL type: {@link PxVec3} [Const, Ref]
     * @param margin      WebIDL type: float
     * @return Stack allocated object of BoxSupport
     */
    public static <T> BoxSupport createAt(T allocator, Allocator<T> allocate, PxVec3 halfExtents, float margin) {
        long address = allocate.on(allocator, ALIGNOF, SIZEOF); 
        __placement_new_BoxSupport(address, halfExtents.getAddress(), margin);
        BoxSupport createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    private static native void __placement_new_BoxSupport(long address, long halfExtents, float margin);

    // Constructors

    /**
     * Constructs a BoxSupport for a box halfExtents with optional margin
     */
    public BoxSupport(PxVec3 halfExtents) {
        address = _BoxSupport(halfExtents.getAddress());
    }
    private static native long _BoxSupport(long halfExtents);

    /**
     * Constructs a BoxSupport for a box halfExtents with optional margin
     */
    public BoxSupport(PxVec3 halfExtents, float margin) {
        address = _BoxSupport(halfExtents.getAddress(), margin);
    }
    private static native long _BoxSupport(long halfExtents, float margin);

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     */
    public PxVec3 getHalfExtents() {
        checkNotNull();
        return PxVec3.wrapPointer(_getHalfExtents(address));
    }
    private static native long _getHalfExtents(long address);

    /**
     */
    public void setHalfExtents(PxVec3 value) {
        checkNotNull();
        _setHalfExtents(address, value.getAddress());
    }
    private static native void _setHalfExtents(long address, long value);

    /**
     */
    public float getMargin() {
        checkNotNull();
        return _getMargin(address);
    }
    private static native float _getMargin(long address);

    /**
     */
    public void setMargin(float value) {
        checkNotNull();
        _setMargin(address, value);
    }
    private static native void _setMargin(long address, float value);

}
