package physx.cooking;

import physx.NativeObject;
import physx.common.PxTolerancesScale;

/**
 * Structure describing parameters affecting mesh cooking.
 */
public class PxCookingParams extends NativeObject {

    protected PxCookingParams() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxCookingParams wrapPointer(long address) {
        return address != 0L ? new PxCookingParams(address) : null;
    }
    
    public static PxCookingParams arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxCookingParams(long address) {
        super(address);
    }

    // Constructors

    /**
     * @param sc WebIDL type: {@link PxTolerancesScale} [Const, Ref]
     */
    public PxCookingParams(PxTolerancesScale sc) {
        address = _PxCookingParams(sc.getAddress());
    }
    private static native long _PxCookingParams(long sc);

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * Zero-size area epsilon used in convex hull computation.
     * <p>
     * If the area of a triangle of the hull is below this value, the triangle will be rejected. This test
     * is done only if PxConvexFlag::eCHECK_ZERO_AREA_TRIANGLES is used.
     * <p>
     * <b>Default value:</b> 0.06f*PxTolerancesScale.length*PxTolerancesScale.length
     * <p>
     * <b>Range:</b> (0.0f, PX_MAX_F32)
     */
    public float getAreaTestEpsilon() {
        checkNotNull();
        return _getAreaTestEpsilon(address);
    }
    private static native float _getAreaTestEpsilon(long address);

    /**
     * Zero-size area epsilon used in convex hull computation.
     * <p>
     * If the area of a triangle of the hull is below this value, the triangle will be rejected. This test
     * is done only if PxConvexFlag::eCHECK_ZERO_AREA_TRIANGLES is used.
     * <p>
     * <b>Default value:</b> 0.06f*PxTolerancesScale.length*PxTolerancesScale.length
     * <p>
     * <b>Range:</b> (0.0f, PX_MAX_F32)
     */
    public void setAreaTestEpsilon(float value) {
        checkNotNull();
        _setAreaTestEpsilon(address, value);
    }
    private static native void _setAreaTestEpsilon(long address, float value);

    /**
     * Plane tolerance used in convex hull computation.
     * <p>
     * The value is used during hull construction. When a new point is about to be added to the hull it
     * gets dropped when the point is closer to the hull than the planeTolerance. The planeTolerance
     * is increased according to the hull size. 
     * <p>
     * If 0.0f is set all points are accepted when the convex hull is created. This may lead to edge cases
     * where the new points may be merged into an existing polygon and the polygons plane equation might 
     * slightly change therefore. This might lead to failures during polygon merging phase in the hull computation.
     * <p>
     * It is recommended to use the default value, however if it is required that all points needs to be
     * accepted or huge thin convexes are created, it might be required to lower the default value.
     * <p>
     * <b>Note:</b> The plane tolerance is used only within PxConvexMeshCookingType::eQUICKHULL algorithm.
     * <p>
     * <b>Default value:</b> 0.0007f
     * <p>
     * <b>Range:</b> &lt;0.0f, PX_MAX_F32)
     */
    public float getPlaneTolerance() {
        checkNotNull();
        return _getPlaneTolerance(address);
    }
    private static native float _getPlaneTolerance(long address);

    /**
     * Plane tolerance used in convex hull computation.
     * <p>
     * The value is used during hull construction. When a new point is about to be added to the hull it
     * gets dropped when the point is closer to the hull than the planeTolerance. The planeTolerance
     * is increased according to the hull size. 
     * <p>
     * If 0.0f is set all points are accepted when the convex hull is created. This may lead to edge cases
     * where the new points may be merged into an existing polygon and the polygons plane equation might 
     * slightly change therefore. This might lead to failures during polygon merging phase in the hull computation.
     * <p>
     * It is recommended to use the default value, however if it is required that all points needs to be
     * accepted or huge thin convexes are created, it might be required to lower the default value.
     * <p>
     * <b>Note:</b> The plane tolerance is used only within PxConvexMeshCookingType::eQUICKHULL algorithm.
     * <p>
     * <b>Default value:</b> 0.0007f
     * <p>
     * <b>Range:</b> &lt;0.0f, PX_MAX_F32)
     */
    public void setPlaneTolerance(float value) {
        checkNotNull();
        _setPlaneTolerance(address, value);
    }
    private static native void _setPlaneTolerance(long address, float value);

    /**
     * Convex hull creation algorithm.
     * <p>
     * <b>Default value:</b> PxConvexMeshCookingType::eQUICKHULL
     */
    public PxConvexMeshCookingTypeEnum getConvexMeshCookingType() {
        checkNotNull();
        return PxConvexMeshCookingTypeEnum.forValue(_getConvexMeshCookingType(address));
    }
    private static native int _getConvexMeshCookingType(long address);

    /**
     * Convex hull creation algorithm.
     * <p>
     * <b>Default value:</b> PxConvexMeshCookingType::eQUICKHULL
     */
    public void setConvexMeshCookingType(PxConvexMeshCookingTypeEnum value) {
        checkNotNull();
        _setConvexMeshCookingType(address, value.value);
    }
    private static native void _setConvexMeshCookingType(long address, int value);

    /**
     * When true, the face remap table is not created.  This saves a significant amount of memory, but the SDK will
     *  not be able to provide the remap information for internal mesh triangles returned by collisions, 
     *  sweeps or raycasts hits.
     * <p>
     * <b>Default value:</b> false
     */
    public boolean getSuppressTriangleMeshRemapTable() {
        checkNotNull();
        return _getSuppressTriangleMeshRemapTable(address);
    }
    private static native boolean _getSuppressTriangleMeshRemapTable(long address);

    /**
     * When true, the face remap table is not created.  This saves a significant amount of memory, but the SDK will
     *  not be able to provide the remap information for internal mesh triangles returned by collisions, 
     *  sweeps or raycasts hits.
     * <p>
     * <b>Default value:</b> false
     */
    public void setSuppressTriangleMeshRemapTable(boolean value) {
        checkNotNull();
        _setSuppressTriangleMeshRemapTable(address, value);
    }
    private static native void _setSuppressTriangleMeshRemapTable(long address, boolean value);

    /**
     * When true, the triangle adjacency information is created. You can get the adjacency triangles
     * for a given triangle from getTriangle.
     * <p>
     * <b>Default value:</b> false
     */
    public boolean getBuildTriangleAdjacencies() {
        checkNotNull();
        return _getBuildTriangleAdjacencies(address);
    }
    private static native boolean _getBuildTriangleAdjacencies(long address);

    /**
     * When true, the triangle adjacency information is created. You can get the adjacency triangles
     * for a given triangle from getTriangle.
     * <p>
     * <b>Default value:</b> false
     */
    public void setBuildTriangleAdjacencies(boolean value) {
        checkNotNull();
        _setBuildTriangleAdjacencies(address, value);
    }
    private static native void _setBuildTriangleAdjacencies(long address, boolean value);

    /**
     * When true, additional information required for GPU-accelerated rigid body simulation is created. This can increase memory usage and cooking times for convex meshes and triangle meshes.
     * <p>
     * <b>Default value:</b> false
     */
    public boolean getBuildGPUData() {
        checkNotNull();
        return _getBuildGPUData(address);
    }
    private static native boolean _getBuildGPUData(long address);

    /**
     * When true, additional information required for GPU-accelerated rigid body simulation is created. This can increase memory usage and cooking times for convex meshes and triangle meshes.
     * <p>
     * <b>Default value:</b> false
     */
    public void setBuildGPUData(boolean value) {
        checkNotNull();
        _setBuildGPUData(address, value);
    }
    private static native void _setBuildGPUData(long address, boolean value);

    /**
     * Tolerance scale is used to check if cooked triangles are not too huge. This check will help with simulation stability.
     * <p>
     * <b>Note:</b> The PxTolerancesScale values have to match the values used when creating a PxPhysics or PxScene instance.
     * @see physx.common.PxTolerancesScale
     */
    public PxTolerancesScale getScale() {
        checkNotNull();
        return PxTolerancesScale.wrapPointer(_getScale(address));
    }
    private static native long _getScale(long address);

    /**
     * Tolerance scale is used to check if cooked triangles are not too huge. This check will help with simulation stability.
     * <p>
     * <b>Note:</b> The PxTolerancesScale values have to match the values used when creating a PxPhysics or PxScene instance.
     * @see physx.common.PxTolerancesScale
     */
    public void setScale(PxTolerancesScale value) {
        checkNotNull();
        _setScale(address, value.getAddress());
    }
    private static native void _setScale(long address, long value);

    /**
     * Mesh pre-processing parameters. Used to control options like whether the mesh cooking performs vertex welding before cooking.
     * <p>
     * <b>Default value:</b> 0
     */
    public PxMeshPreprocessingFlags getMeshPreprocessParams() {
        checkNotNull();
        return PxMeshPreprocessingFlags.wrapPointer(_getMeshPreprocessParams(address));
    }
    private static native long _getMeshPreprocessParams(long address);

    /**
     * Mesh pre-processing parameters. Used to control options like whether the mesh cooking performs vertex welding before cooking.
     * <p>
     * <b>Default value:</b> 0
     */
    public void setMeshPreprocessParams(PxMeshPreprocessingFlags value) {
        checkNotNull();
        _setMeshPreprocessParams(address, value.getAddress());
    }
    private static native void _setMeshPreprocessParams(long address, long value);

    /**
     * Mesh weld tolerance. If mesh welding is enabled, this controls the distance at which vertices are welded.
     * If mesh welding is not enabled, this value defines the acceptance distance for mesh validation. Provided no two vertices are within this distance, the mesh is considered to be
     * clean. If not, a warning will be emitted. Having a clean, welded mesh is required to achieve the best possible performance.
     * <p>
     * The default vertex welding uses a snap-to-grid approach. This approach effectively truncates each vertex to integer values using meshWeldTolerance.
     * Once these snapped vertices are produced, all vertices that snap to a given vertex on the grid are remapped to reference a single vertex. Following this,
     * all triangles' indices are remapped to reference this subset of clean vertices. It should be noted that the vertices that we do not alter the
     * position of the vertices; the snap-to-grid is only performed to identify nearby vertices.
     * <p>
     * The mesh validation approach also uses the same snap-to-grid approach to identify nearby vertices. If more than one vertex snaps to a given grid coordinate,
     * we ensure that the distance between the vertices is at least meshWeldTolerance. If this is not the case, a warning is emitted.
     * <p>
     * <b>Default value:</b> 0.0
     */
    public float getMeshWeldTolerance() {
        checkNotNull();
        return _getMeshWeldTolerance(address);
    }
    private static native float _getMeshWeldTolerance(long address);

    /**
     * Mesh weld tolerance. If mesh welding is enabled, this controls the distance at which vertices are welded.
     * If mesh welding is not enabled, this value defines the acceptance distance for mesh validation. Provided no two vertices are within this distance, the mesh is considered to be
     * clean. If not, a warning will be emitted. Having a clean, welded mesh is required to achieve the best possible performance.
     * <p>
     * The default vertex welding uses a snap-to-grid approach. This approach effectively truncates each vertex to integer values using meshWeldTolerance.
     * Once these snapped vertices are produced, all vertices that snap to a given vertex on the grid are remapped to reference a single vertex. Following this,
     * all triangles' indices are remapped to reference this subset of clean vertices. It should be noted that the vertices that we do not alter the
     * position of the vertices; the snap-to-grid is only performed to identify nearby vertices.
     * <p>
     * The mesh validation approach also uses the same snap-to-grid approach to identify nearby vertices. If more than one vertex snaps to a given grid coordinate,
     * we ensure that the distance between the vertices is at least meshWeldTolerance. If this is not the case, a warning is emitted.
     * <p>
     * <b>Default value:</b> 0.0
     */
    public void setMeshWeldTolerance(float value) {
        checkNotNull();
        _setMeshWeldTolerance(address, value);
    }
    private static native void _setMeshWeldTolerance(long address, float value);

    /**
     * Controls the desired midphase desc structure for triangle meshes.
     * @see PxMidphaseDesc
     * <p>
     * <b>Default value:</b> PxMeshMidPhase::eBVH34
     */
    public PxMidphaseDesc getMidphaseDesc() {
        checkNotNull();
        return PxMidphaseDesc.wrapPointer(_getMidphaseDesc(address));
    }
    private static native long _getMidphaseDesc(long address);

    /**
     * Controls the desired midphase desc structure for triangle meshes.
     * @see PxMidphaseDesc
     * <p>
     * <b>Default value:</b> PxMeshMidPhase::eBVH34
     */
    public void setMidphaseDesc(PxMidphaseDesc value) {
        checkNotNull();
        _setMidphaseDesc(address, value.getAddress());
    }
    private static native void _setMidphaseDesc(long address, long value);

    /**
     * Vertex limit beyond which additional acceleration structures are computed for each convex mesh. Increase that limit to reduce memory usage.
     * Computing the extra structures all the time does not guarantee optimal performance. There is a per-platform break-even point below which the
     * extra structures actually hurt performance.
     * <p>
     * <b>Default value:</b> 32
     */
    public int getGaussMapLimit() {
        checkNotNull();
        return _getGaussMapLimit(address);
    }
    private static native int _getGaussMapLimit(long address);

    /**
     * Vertex limit beyond which additional acceleration structures are computed for each convex mesh. Increase that limit to reduce memory usage.
     * Computing the extra structures all the time does not guarantee optimal performance. There is a per-platform break-even point below which the
     * extra structures actually hurt performance.
     * <p>
     * <b>Default value:</b> 32
     */
    public void setGaussMapLimit(int value) {
        checkNotNull();
        _setGaussMapLimit(address, value);
    }
    private static native void _setGaussMapLimit(long address, int value);

}
