package physx.support;

/**
 * Debug visualization parameters.
 * <p>
 * #PxVisualizationParameter::eSCALE is the master switch for enabling visualization, please read the corresponding documentation
 * for further details.
 * <p>
 * <b>See also:</b> PxScene.setVisualizationParameter() PxScene.getVisualizationParameter() PxScene.getRenderBuffer()
 */
public enum PxVisualizationParameterEnum {

    /**
     * This overall visualization scale gets multiplied with the individual scales. Setting to zero ignores all visualizations. Default is 0.
     * <p>
     * The below settings permit the debug visualization of various simulation properties. 
     * The setting is either zero, in which case the property is not drawn. Otherwise it is a scaling factor
     * that determines the size of the visualization widgets.
     * <p>
     * Only objects for which visualization is turned on using setFlag(eVISUALIZATION) are visualized (see #PxActorFlag::eVISUALIZATION, #PxShapeFlag::eVISUALIZATION, ...).
     * Contacts are visualized if they involve a body which is being visualized.
     * Default is 0.
     * <p>
     * Notes:
     * - to see any visualization, you have to set PxVisualizationParameter::eSCALE to nonzero first.
     * - the scale factor has been introduced because it's difficult (if not impossible) to come up with a
     * good scale for 3D vectors. Normals are normalized and their length is always 1. But it doesn't mean
     * we should render a line of length 1. Depending on your objects/scene, this might be completely invisible
     * or extremely huge. That's why the scale factor is here, to let you tune the length until it's ok in
     * your scene.
     * - however, things like collision shapes aren't ambiguous. They are clearly defined for example by the
     * triangles &amp; polygons themselves, and there's no point in scaling that. So the visualization widgets
     * are only scaled when it makes sense.
     * <p>
     * <b>Range:</b> [0, PX_MAX_F32)<br>
     * <b>Default:</b> 0
     */
    eSCALE(geteSCALE()),
    /**
     * Visualize the world axes.
     */
    eWORLD_AXES(geteWORLD_AXES()),
    /**
     * Visualize a bodies axes.
     * <p>
     * <b>See also:</b> PxActor.globalPose PxActor
     */
    eBODY_AXES(geteBODY_AXES()),
    /**
     * Visualize a body's mass axes.
     * <p>
     * This visualization is also useful for visualizing the sleep state of bodies. Sleeping bodies are drawn in
     * black, while awake bodies are drawn in white. If the body is sleeping and part of a sleeping group, it is
     * drawn in red.
     * <p>
     * <b>See also:</b> PxBodyDesc.massLocalPose PxActor
     */
    eBODY_MASS_AXES(geteBODY_MASS_AXES()),
    /**
     * Visualize the bodies linear velocity.
     * <p>
     * <b>See also:</b> PxBodyDesc.linearVelocity PxActor
     */
    eBODY_LIN_VELOCITY(geteBODY_LIN_VELOCITY()),
    /**
     * Visualize the bodies angular velocity.
     * <p>
     * <b>See also:</b> PxBodyDesc.angularVelocity PxActor
     */
    eBODY_ANG_VELOCITY(geteBODY_ANG_VELOCITY()),
    /**
     *  Visualize contact points. Will enable contact information.
     */
    eCONTACT_POINT(geteCONTACT_POINT()),
    /**
     * Visualize contact normals. Will enable contact information.
     */
    eCONTACT_NORMAL(geteCONTACT_NORMAL()),
    /**
     *  Visualize contact errors. Will enable contact information.
     */
    eCONTACT_ERROR(geteCONTACT_ERROR()),
    /**
     * Visualize Contact forces. Will enable contact information.
     */
    eCONTACT_FORCE(geteCONTACT_FORCE()),
    /**
     * Visualize actor axes.
     * <p>
     * <b>See also:</b> PxRigidStatic PxRigidDynamic PxArticulationLink
     */
    eACTOR_AXES(geteACTOR_AXES()),
    /**
     * Visualize bounds (AABBs in world space)
     */
    eCOLLISION_AABBS(geteCOLLISION_AABBS()),
    /**
     * Shape visualization
     * <p>
     * <b>See also:</b> PxShape
     */
    eCOLLISION_SHAPES(geteCOLLISION_SHAPES()),
    /**
     * Shape axis visualization
     * <p>
     * <b>See also:</b> PxShape
     */
    eCOLLISION_AXES(geteCOLLISION_AXES()),
    /**
     * Compound visualization (compound AABBs in world space)
     */
    eCOLLISION_COMPOUNDS(geteCOLLISION_COMPOUNDS()),
    /**
     * Mesh &amp; convex face normals
     * <p>
     * <b>See also:</b> PxTriangleMesh PxConvexMesh
     */
    eCOLLISION_FNORMALS(geteCOLLISION_FNORMALS()),
    /**
     * Active edges for meshes
     * <p>
     * <b>See also:</b> PxTriangleMesh
     */
    eCOLLISION_EDGES(geteCOLLISION_EDGES()),
    /**
     * Static pruning structures
     */
    eCOLLISION_STATIC(geteCOLLISION_STATIC()),
    /**
     * Dynamic pruning structures
     */
    eCOLLISION_DYNAMIC(geteCOLLISION_DYNAMIC()),
    /**
     * Joint local axes
     */
    eJOINT_LOCAL_FRAMES(geteJOINT_LOCAL_FRAMES()),
    /**
     * Joint limits
     */
    eJOINT_LIMITS(geteJOINT_LIMITS()),
    /**
     * Visualize culling box
     */
    eCULL_BOX(geteCULL_BOX()),
    /**
     * MBP regions
     */
    eMBP_REGIONS(geteMBP_REGIONS()),
    /**
     * Renders the simulation mesh instead of the collision mesh (only available for tetmeshes)
     */
    eSIMULATION_MESH(geteSIMULATION_MESH()),
    /**
     * Renders the SDF of a mesh instead of the collision mesh (only available for triangle meshes with SDFs)
     */
    eSDF(geteSDF()),
    /**
     * This is not a parameter, it just records the current number of parameters (as maximum(PxVisualizationParameter)+1) for use in loops.
     */
    eNUM_VALUES(geteNUM_VALUES()),
    eFORCE_DWORD(geteFORCE_DWORD());
    public final int value;
    
    PxVisualizationParameterEnum(int value) {
        this.value = value;
    }

    private static native int _geteSCALE();
    private static int geteSCALE() {
        de.fabmax.physxjni.Loader.load();
        return _geteSCALE();
    }

    private static native int _geteWORLD_AXES();
    private static int geteWORLD_AXES() {
        de.fabmax.physxjni.Loader.load();
        return _geteWORLD_AXES();
    }

    private static native int _geteBODY_AXES();
    private static int geteBODY_AXES() {
        de.fabmax.physxjni.Loader.load();
        return _geteBODY_AXES();
    }

    private static native int _geteBODY_MASS_AXES();
    private static int geteBODY_MASS_AXES() {
        de.fabmax.physxjni.Loader.load();
        return _geteBODY_MASS_AXES();
    }

    private static native int _geteBODY_LIN_VELOCITY();
    private static int geteBODY_LIN_VELOCITY() {
        de.fabmax.physxjni.Loader.load();
        return _geteBODY_LIN_VELOCITY();
    }

    private static native int _geteBODY_ANG_VELOCITY();
    private static int geteBODY_ANG_VELOCITY() {
        de.fabmax.physxjni.Loader.load();
        return _geteBODY_ANG_VELOCITY();
    }

    private static native int _geteCONTACT_POINT();
    private static int geteCONTACT_POINT() {
        de.fabmax.physxjni.Loader.load();
        return _geteCONTACT_POINT();
    }

    private static native int _geteCONTACT_NORMAL();
    private static int geteCONTACT_NORMAL() {
        de.fabmax.physxjni.Loader.load();
        return _geteCONTACT_NORMAL();
    }

    private static native int _geteCONTACT_ERROR();
    private static int geteCONTACT_ERROR() {
        de.fabmax.physxjni.Loader.load();
        return _geteCONTACT_ERROR();
    }

    private static native int _geteCONTACT_FORCE();
    private static int geteCONTACT_FORCE() {
        de.fabmax.physxjni.Loader.load();
        return _geteCONTACT_FORCE();
    }

    private static native int _geteACTOR_AXES();
    private static int geteACTOR_AXES() {
        de.fabmax.physxjni.Loader.load();
        return _geteACTOR_AXES();
    }

    private static native int _geteCOLLISION_AABBS();
    private static int geteCOLLISION_AABBS() {
        de.fabmax.physxjni.Loader.load();
        return _geteCOLLISION_AABBS();
    }

    private static native int _geteCOLLISION_SHAPES();
    private static int geteCOLLISION_SHAPES() {
        de.fabmax.physxjni.Loader.load();
        return _geteCOLLISION_SHAPES();
    }

    private static native int _geteCOLLISION_AXES();
    private static int geteCOLLISION_AXES() {
        de.fabmax.physxjni.Loader.load();
        return _geteCOLLISION_AXES();
    }

    private static native int _geteCOLLISION_COMPOUNDS();
    private static int geteCOLLISION_COMPOUNDS() {
        de.fabmax.physxjni.Loader.load();
        return _geteCOLLISION_COMPOUNDS();
    }

    private static native int _geteCOLLISION_FNORMALS();
    private static int geteCOLLISION_FNORMALS() {
        de.fabmax.physxjni.Loader.load();
        return _geteCOLLISION_FNORMALS();
    }

    private static native int _geteCOLLISION_EDGES();
    private static int geteCOLLISION_EDGES() {
        de.fabmax.physxjni.Loader.load();
        return _geteCOLLISION_EDGES();
    }

    private static native int _geteCOLLISION_STATIC();
    private static int geteCOLLISION_STATIC() {
        de.fabmax.physxjni.Loader.load();
        return _geteCOLLISION_STATIC();
    }

    private static native int _geteCOLLISION_DYNAMIC();
    private static int geteCOLLISION_DYNAMIC() {
        de.fabmax.physxjni.Loader.load();
        return _geteCOLLISION_DYNAMIC();
    }

    private static native int _geteJOINT_LOCAL_FRAMES();
    private static int geteJOINT_LOCAL_FRAMES() {
        de.fabmax.physxjni.Loader.load();
        return _geteJOINT_LOCAL_FRAMES();
    }

    private static native int _geteJOINT_LIMITS();
    private static int geteJOINT_LIMITS() {
        de.fabmax.physxjni.Loader.load();
        return _geteJOINT_LIMITS();
    }

    private static native int _geteCULL_BOX();
    private static int geteCULL_BOX() {
        de.fabmax.physxjni.Loader.load();
        return _geteCULL_BOX();
    }

    private static native int _geteMBP_REGIONS();
    private static int geteMBP_REGIONS() {
        de.fabmax.physxjni.Loader.load();
        return _geteMBP_REGIONS();
    }

    private static native int _geteSIMULATION_MESH();
    private static int geteSIMULATION_MESH() {
        de.fabmax.physxjni.Loader.load();
        return _geteSIMULATION_MESH();
    }

    private static native int _geteSDF();
    private static int geteSDF() {
        de.fabmax.physxjni.Loader.load();
        return _geteSDF();
    }

    private static native int _geteNUM_VALUES();
    private static int geteNUM_VALUES() {
        de.fabmax.physxjni.Loader.load();
        return _geteNUM_VALUES();
    }

    private static native int _geteFORCE_DWORD();
    private static int geteFORCE_DWORD() {
        de.fabmax.physxjni.Loader.load();
        return _geteFORCE_DWORD();
    }

    public static PxVisualizationParameterEnum forValue(int value) {
        for (int i = 0; i < values().length; i++) {
            if (values()[i].value == value) {
                return values()[i];
            }
        }
        throw new IllegalArgumentException("Unknown value for enum PxVisualizationParameterEnum: " + value);
    }

}
