package physx.physics;

import physx.NativeObject;

/**
 * Descriptor for a trigger pair.
 * <p>
 * An array of these structs gets passed to the PxSimulationEventCallback::onTrigger() report.
 * <p>
 * <b>Note:</b> The shape pointers might reference deleted shapes. This will be the case if #PxPairFlag::eNOTIFY_TOUCH_LOST
 *       events were requested for the pair and one of the involved shapes gets deleted. Check the #flags member to see
 *    whether that is the case. Do not dereference a pointer to a deleted shape. The pointer to a deleted shape is 
 *    only provided such that user data structures which might depend on the pointer value can be updated.
 */
public class PxTriggerPair extends NativeObject {

    protected PxTriggerPair() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxTriggerPair wrapPointer(long address) {
        return address != 0L ? new PxTriggerPair(address) : null;
    }
    
    public static PxTriggerPair arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxTriggerPair(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * The shape that has been marked as a trigger.
     */
    public PxShape getTriggerShape() {
        checkNotNull();
        return PxShape.wrapPointer(_getTriggerShape(address));
    }
    private static native long _getTriggerShape(long address);

    /**
     * The shape that has been marked as a trigger.
     */
    public void setTriggerShape(PxShape value) {
        checkNotNull();
        _setTriggerShape(address, value.getAddress());
    }
    private static native void _setTriggerShape(long address, long value);

    /**
     * The actor to which triggerShape is attached
     */
    public PxActor getTriggerActor() {
        checkNotNull();
        return PxActor.wrapPointer(_getTriggerActor(address));
    }
    private static native long _getTriggerActor(long address);

    /**
     * The actor to which triggerShape is attached
     */
    public void setTriggerActor(PxActor value) {
        checkNotNull();
        _setTriggerActor(address, value.getAddress());
    }
    private static native void _setTriggerActor(long address, long value);

    /**
     * The shape causing the trigger event. @deprecated  (see #PxSimulationEventCallback::onTrigger()) If collision between trigger shapes is enabled, then this member might point to a trigger shape as well.
     */
    @Deprecated
    public PxShape getOtherShape() {
        checkNotNull();
        return PxShape.wrapPointer(_getOtherShape(address));
    }
    private static native long _getOtherShape(long address);

    /**
     * The shape causing the trigger event. @deprecated  (see #PxSimulationEventCallback::onTrigger()) If collision between trigger shapes is enabled, then this member might point to a trigger shape as well.
     */
    @Deprecated
    public void setOtherShape(PxShape value) {
        checkNotNull();
        _setOtherShape(address, value.getAddress());
    }
    private static native void _setOtherShape(long address, long value);

    /**
     * The actor to which otherShape is attached
     */
    public PxActor getOtherActor() {
        checkNotNull();
        return PxActor.wrapPointer(_getOtherActor(address));
    }
    private static native long _getOtherActor(long address);

    /**
     * The actor to which otherShape is attached
     */
    public void setOtherActor(PxActor value) {
        checkNotNull();
        _setOtherActor(address, value.getAddress());
    }
    private static native void _setOtherActor(long address, long value);

    /**
     * Type of trigger event (eNOTIFY_TOUCH_FOUND or eNOTIFY_TOUCH_LOST). eNOTIFY_TOUCH_PERSISTS events are not supported.
     */
    public PxPairFlagEnum getStatus() {
        checkNotNull();
        return PxPairFlagEnum.forValue(_getStatus(address));
    }
    private static native int _getStatus(long address);

    /**
     * Type of trigger event (eNOTIFY_TOUCH_FOUND or eNOTIFY_TOUCH_LOST). eNOTIFY_TOUCH_PERSISTS events are not supported.
     */
    public void setStatus(PxPairFlagEnum value) {
        checkNotNull();
        _setStatus(address, value.value);
    }
    private static native void _setStatus(long address, int value);

    /**
     * Additional information on the pair (see #PxTriggerPairFlag)
     */
    public PxTriggerPairFlags getFlags() {
        checkNotNull();
        return PxTriggerPairFlags.wrapPointer(_getFlags(address));
    }
    private static native long _getFlags(long address);

    /**
     * Additional information on the pair (see #PxTriggerPairFlag)
     */
    public void setFlags(PxTriggerPairFlags value) {
        checkNotNull();
        _setFlags(address, value.getAddress());
    }
    private static native void _setFlags(long address, long value);

}
