package physx.physics;

import physx.NativeObject;
import physx.common.PxTransform;
import physx.common.PxVec3;
import physx.geometry.PxGeometry;

/**
 * Base class for the scene-query system.
 * <p>
 * Methods defined here are common to both the traditional PxScene API and the PxSceneQuerySystem API.
 * @see PxScene
 */
public class PxSceneQuerySystemBase extends NativeObject {

    protected PxSceneQuerySystemBase() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxSceneQuerySystemBase wrapPointer(long address) {
        return address != 0L ? new PxSceneQuerySystemBase(address) : null;
    }
    
    public static PxSceneQuerySystemBase arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxSceneQuerySystemBase(long address) {
        super(address);
    }

    // Functions

    /**
     * Sets the rebuild rate of the dynamic tree pruning structures.
     * @param dynamicTreeRebuildRateHint Rebuild rate of the dynamic tree pruning structures.
     * @see #getDynamicTreeRebuildRateHint
     * @see #forceRebuildDynamicTree
     */
    public void setDynamicTreeRebuildRateHint(int dynamicTreeRebuildRateHint) {
        checkNotNull();
        _setDynamicTreeRebuildRateHint(address, dynamicTreeRebuildRateHint);
    }
    private static native void _setDynamicTreeRebuildRateHint(long address, int dynamicTreeRebuildRateHint);

    /**
     * Retrieves the rebuild rate of the dynamic tree pruning structures.
     * @return The rebuild rate of the dynamic tree pruning structures.
     * @see #setDynamicTreeRebuildRateHint
     * @see #forceRebuildDynamicTree
     */
    public int getDynamicTreeRebuildRateHint() {
        checkNotNull();
        return _getDynamicTreeRebuildRateHint(address);
    }
    private static native int _getDynamicTreeRebuildRateHint(long address);

    /**
     * Forces dynamic trees to be immediately rebuilt.
     * @param prunerIndex Index of pruner containing the dynamic tree to rebuild
     * <p>
     * <b>Note:</b> PxScene will call this function with the PX_SCENE_PRUNER_STATIC or PX_SCENE_PRUNER_DYNAMIC value.
     * @see #setDynamicTreeRebuildRateHint
     * @see #getDynamicTreeRebuildRateHint
     */
    public void forceRebuildDynamicTree(int prunerIndex) {
        checkNotNull();
        _forceRebuildDynamicTree(address, prunerIndex);
    }
    private static native void _forceRebuildDynamicTree(long address, int prunerIndex);

    /**
     * Sets scene query update mode 
     * @param updateMode Scene query update mode.
     */
    public void setUpdateMode(PxSceneQueryUpdateModeEnum updateMode) {
        checkNotNull();
        _setUpdateMode(address, updateMode.value);
    }
    private static native void _setUpdateMode(long address, int updateMode);

    /**
     * Gets scene query update mode 
     * @return Current scene query update mode.
     */
    public PxSceneQueryUpdateModeEnum getUpdateMode() {
        checkNotNull();
        return PxSceneQueryUpdateModeEnum.forValue(_getUpdateMode(address));
    }
    private static native int _getUpdateMode(long address);

    /**
     * Retrieves the system's internal scene query timestamp, increased each time a change to the
     * static scene query structure is performed.
     * @return scene query static timestamp
     */
    public int getStaticTimestamp() {
        checkNotNull();
        return _getStaticTimestamp(address);
    }
    private static native int _getStaticTimestamp(long address);

    /**
     * Flushes any changes to the scene query representation.
     * <p>
     * This method updates the state of the scene query representation to match changes in the scene state.
     * <p>
     * By default, these changes are buffered until the next query is submitted. Calling this function will not change
     * the results from scene queries, but can be used to ensure that a query will not perform update work in the course of 
     * its execution.
     * <p>
     * A thread performing updates will hold a write lock on the query structure, and thus stall other querying threads. In multithread
     * scenarios it can be useful to explicitly schedule the period where this lock may be held for a significant period, so that
     * subsequent queries issued from multiple threads will not block.
     */
    public void flushUpdates() {
        checkNotNull();
        _flushUpdates(address);
    }
    private static native void _flushUpdates(long address);

    /**
     * Performs a raycast against objects in the scene, returns results in a PxRaycastBuffer object
     * or via a custom user callback implementation inheriting from PxRaycastCallback.
     * <p>
     * <b>Note:</b> Touching hits are not ordered.
     * <b>Note:</b> Shooting a ray from within an object leads to different results depending on the shape type. Please check the details in user guide article SceneQuery. User can ignore such objects by employing one of the provided filter mechanisms.
     * @param origin  Origin of the ray.
     * @param unitDir  Normalized direction of the ray.
     * @param distance  Length of the ray. Has to be in the [0, inf) range.
     * @param hitCall  Raycast hit buffer or callback object used to report raycast hits.
     *       Note: Filtering is not executed for a cached shape if supplied; instead, if a hit is found, it is assumed to be a blocking hit.
     *       Note: Using past touching hits as cache will produce incorrect behavior since the cached hit will always be treated as blocking.
     * @return True if any touching or blocking hits were found or any hit was found in case PxQueryFlag::eANY_HIT was specified.
     * @see PxRaycastCallback
     * @see PxQueryFilterData
     * @see PxQueryFilterCallback
     * @see PxRaycastHit
     */
    public boolean raycast(PxVec3 origin, PxVec3 unitDir, float distance, PxRaycastCallback hitCall) {
        checkNotNull();
        return _raycast(address, origin.getAddress(), unitDir.getAddress(), distance, hitCall.getAddress());
    }
    private static native boolean _raycast(long address, long origin, long unitDir, float distance, long hitCall);

    /**
     * Performs a raycast against objects in the scene, returns results in a PxRaycastBuffer object
     * or via a custom user callback implementation inheriting from PxRaycastCallback.
     * <p>
     * <b>Note:</b> Touching hits are not ordered.
     * <b>Note:</b> Shooting a ray from within an object leads to different results depending on the shape type. Please check the details in user guide article SceneQuery. User can ignore such objects by employing one of the provided filter mechanisms.
     * @param origin  Origin of the ray.
     * @param unitDir  Normalized direction of the ray.
     * @param distance  Length of the ray. Has to be in the [0, inf) range.
     * @param hitCall  Raycast hit buffer or callback object used to report raycast hits.
     * @param hitFlags  Specifies which properties per hit should be computed and returned via the hit callback.
     *       Note: Filtering is not executed for a cached shape if supplied; instead, if a hit is found, it is assumed to be a blocking hit.
     *       Note: Using past touching hits as cache will produce incorrect behavior since the cached hit will always be treated as blocking.
     * @return True if any touching or blocking hits were found or any hit was found in case PxQueryFlag::eANY_HIT was specified.
     * @see PxRaycastCallback
     * @see PxQueryFilterData
     * @see PxQueryFilterCallback
     * @see PxRaycastHit
     */
    public boolean raycast(PxVec3 origin, PxVec3 unitDir, float distance, PxRaycastCallback hitCall, PxHitFlags hitFlags) {
        checkNotNull();
        return _raycast(address, origin.getAddress(), unitDir.getAddress(), distance, hitCall.getAddress(), hitFlags.getAddress());
    }
    private static native boolean _raycast(long address, long origin, long unitDir, float distance, long hitCall, long hitFlags);

    /**
     * Performs a raycast against objects in the scene, returns results in a PxRaycastBuffer object
     * or via a custom user callback implementation inheriting from PxRaycastCallback.
     * <p>
     * <b>Note:</b> Touching hits are not ordered.
     * <b>Note:</b> Shooting a ray from within an object leads to different results depending on the shape type. Please check the details in user guide article SceneQuery. User can ignore such objects by employing one of the provided filter mechanisms.
     * @param origin  Origin of the ray.
     * @param unitDir  Normalized direction of the ray.
     * @param distance  Length of the ray. Has to be in the [0, inf) range.
     * @param hitCall  Raycast hit buffer or callback object used to report raycast hits.
     * @param hitFlags  Specifies which properties per hit should be computed and returned via the hit callback.
     * @param filterData Filtering data passed to the filter shader. 
     *       Note: Filtering is not executed for a cached shape if supplied; instead, if a hit is found, it is assumed to be a blocking hit.
     *       Note: Using past touching hits as cache will produce incorrect behavior since the cached hit will always be treated as blocking.
     * @return True if any touching or blocking hits were found or any hit was found in case PxQueryFlag::eANY_HIT was specified.
     * @see PxRaycastCallback
     * @see PxQueryFilterData
     * @see PxQueryFilterCallback
     * @see PxRaycastHit
     */
    public boolean raycast(PxVec3 origin, PxVec3 unitDir, float distance, PxRaycastCallback hitCall, PxHitFlags hitFlags, PxQueryFilterData filterData) {
        checkNotNull();
        return _raycast(address, origin.getAddress(), unitDir.getAddress(), distance, hitCall.getAddress(), hitFlags.getAddress(), filterData.getAddress());
    }
    private static native boolean _raycast(long address, long origin, long unitDir, float distance, long hitCall, long hitFlags, long filterData);

    /**
     * Performs a sweep test against objects in the scene, returns results in a PxSweepBuffer object
     * or via a custom user callback implementation inheriting from PxSweepCallback.
     * <p>
     * <b>Note:</b> Touching hits are not ordered.
     * <b>Note:</b> If a shape from the scene is already overlapping with the query shape in its starting position,
     *   the hit is returned unless eASSUME_NO_INITIAL_OVERLAP was specified.
     * @param geometry  Geometry of object to sweep (supported types are: box, sphere, capsule, convex).
     * @param pose   Pose of the sweep object.
     * @param unitDir  Normalized direction of the sweep.
     * @param distance  Sweep distance. Needs to be in [0, inf) range and &gt;0 if eASSUME_NO_INITIAL_OVERLAP was specified. Will be clamped to PX_MAX_SWEEP_DISTANCE.
     * @param hitCall  Sweep hit buffer or callback object used to report sweep hits.
     *       Note: Filtering is not executed for a cached shape if supplied; instead, if a hit is found, it is assumed to be a blocking hit.
     *       Note: Using past touching hits as cache will produce incorrect behavior since the cached hit will always be treated as blocking.
     *       Note: ePRECISE_SWEEP doesn't support inflation. Therefore the sweep will be performed with zero inflation. 
     * @return True if any touching or blocking hits were found or any hit was found in case PxQueryFlag::eANY_HIT was specified.
     * @see PxSweepCallback
     * @see PxQueryFilterData
     * @see PxQueryFilterCallback
     * @see PxSweepHit
     */
    public boolean sweep(PxGeometry geometry, PxTransform pose, PxVec3 unitDir, float distance, PxSweepCallback hitCall) {
        checkNotNull();
        return _sweep(address, geometry.getAddress(), pose.getAddress(), unitDir.getAddress(), distance, hitCall.getAddress());
    }
    private static native boolean _sweep(long address, long geometry, long pose, long unitDir, float distance, long hitCall);

    /**
     * Performs a sweep test against objects in the scene, returns results in a PxSweepBuffer object
     * or via a custom user callback implementation inheriting from PxSweepCallback.
     * <p>
     * <b>Note:</b> Touching hits are not ordered.
     * <b>Note:</b> If a shape from the scene is already overlapping with the query shape in its starting position,
     *   the hit is returned unless eASSUME_NO_INITIAL_OVERLAP was specified.
     * @param geometry  Geometry of object to sweep (supported types are: box, sphere, capsule, convex).
     * @param pose   Pose of the sweep object.
     * @param unitDir  Normalized direction of the sweep.
     * @param distance  Sweep distance. Needs to be in [0, inf) range and &gt;0 if eASSUME_NO_INITIAL_OVERLAP was specified. Will be clamped to PX_MAX_SWEEP_DISTANCE.
     * @param hitCall  Sweep hit buffer or callback object used to report sweep hits.
     * @param hitFlags  Specifies which properties per hit should be computed and returned via the hit callback.
     *       Note: Filtering is not executed for a cached shape if supplied; instead, if a hit is found, it is assumed to be a blocking hit.
     *       Note: Using past touching hits as cache will produce incorrect behavior since the cached hit will always be treated as blocking.
     *       Note: ePRECISE_SWEEP doesn't support inflation. Therefore the sweep will be performed with zero inflation. 
     * @return True if any touching or blocking hits were found or any hit was found in case PxQueryFlag::eANY_HIT was specified.
     * @see PxSweepCallback
     * @see PxQueryFilterData
     * @see PxQueryFilterCallback
     * @see PxSweepHit
     */
    public boolean sweep(PxGeometry geometry, PxTransform pose, PxVec3 unitDir, float distance, PxSweepCallback hitCall, PxHitFlags hitFlags) {
        checkNotNull();
        return _sweep(address, geometry.getAddress(), pose.getAddress(), unitDir.getAddress(), distance, hitCall.getAddress(), hitFlags.getAddress());
    }
    private static native boolean _sweep(long address, long geometry, long pose, long unitDir, float distance, long hitCall, long hitFlags);

    /**
     * Performs a sweep test against objects in the scene, returns results in a PxSweepBuffer object
     * or via a custom user callback implementation inheriting from PxSweepCallback.
     * <p>
     * <b>Note:</b> Touching hits are not ordered.
     * <b>Note:</b> If a shape from the scene is already overlapping with the query shape in its starting position,
     *   the hit is returned unless eASSUME_NO_INITIAL_OVERLAP was specified.
     * @param geometry  Geometry of object to sweep (supported types are: box, sphere, capsule, convex).
     * @param pose   Pose of the sweep object.
     * @param unitDir  Normalized direction of the sweep.
     * @param distance  Sweep distance. Needs to be in [0, inf) range and &gt;0 if eASSUME_NO_INITIAL_OVERLAP was specified. Will be clamped to PX_MAX_SWEEP_DISTANCE.
     * @param hitCall  Sweep hit buffer or callback object used to report sweep hits.
     * @param hitFlags  Specifies which properties per hit should be computed and returned via the hit callback.
     * @param filterData Filtering data and simple logic.
     *       Note: Filtering is not executed for a cached shape if supplied; instead, if a hit is found, it is assumed to be a blocking hit.
     *       Note: Using past touching hits as cache will produce incorrect behavior since the cached hit will always be treated as blocking.
     *       Note: ePRECISE_SWEEP doesn't support inflation. Therefore the sweep will be performed with zero inflation. 
     * @return True if any touching or blocking hits were found or any hit was found in case PxQueryFlag::eANY_HIT was specified.
     * @see PxSweepCallback
     * @see PxQueryFilterData
     * @see PxQueryFilterCallback
     * @see PxSweepHit
     */
    public boolean sweep(PxGeometry geometry, PxTransform pose, PxVec3 unitDir, float distance, PxSweepCallback hitCall, PxHitFlags hitFlags, PxQueryFilterData filterData) {
        checkNotNull();
        return _sweep(address, geometry.getAddress(), pose.getAddress(), unitDir.getAddress(), distance, hitCall.getAddress(), hitFlags.getAddress(), filterData.getAddress());
    }
    private static native boolean _sweep(long address, long geometry, long pose, long unitDir, float distance, long hitCall, long hitFlags, long filterData);

    /**
     * Performs an overlap test of a given geometry against objects in the scene, returns results in a PxOverlapBuffer object
     * or via a custom user callback implementation inheriting from PxOverlapCallback.
     * <p>
     * <b>Note:</b> Filtering: returning eBLOCK from user filter for overlap queries will cause a warning (see #PxQueryHitType).
     * @param geometry  Geometry of object to check for overlap (supported types are: box, sphere, capsule, convex).
     * @param pose   Pose of the object.
     * @param hitCall  Overlap hit buffer or callback object used to report overlap hits.
     * Note: Filtering is not executed for a cached shape if supplied; instead, if a hit is found, it is assumed to be a blocking hit.
     * Note: Using past touching hits as cache will produce incorrect behavior since the cached hit will always be treated as blocking.
     * @return True if any touching or blocking hits were found or any hit was found in case PxQueryFlag::eANY_HIT was specified.
     * <p>
     * <b>Note:</b> eBLOCK should not be returned from user filters for overlap(). Doing so will result in undefined behavior, and a warning will be issued.
     * <b>Note:</b> If the PxQueryFlag::eNO_BLOCK flag is set, the eBLOCK will instead be automatically converted to an eTOUCH and the warning suppressed.
     * @see PxOverlapCallback
     * @see PxHitFlags
     * @see PxQueryFilterData
     * @see PxQueryFilterCallback
     */
    public boolean overlap(PxGeometry geometry, PxTransform pose, PxOverlapCallback hitCall) {
        checkNotNull();
        return _overlap(address, geometry.getAddress(), pose.getAddress(), hitCall.getAddress());
    }
    private static native boolean _overlap(long address, long geometry, long pose, long hitCall);

    /**
     * Performs an overlap test of a given geometry against objects in the scene, returns results in a PxOverlapBuffer object
     * or via a custom user callback implementation inheriting from PxOverlapCallback.
     * <p>
     * <b>Note:</b> Filtering: returning eBLOCK from user filter for overlap queries will cause a warning (see #PxQueryHitType).
     * @param geometry  Geometry of object to check for overlap (supported types are: box, sphere, capsule, convex).
     * @param pose   Pose of the object.
     * @param hitCall  Overlap hit buffer or callback object used to report overlap hits.
     * @param filterData Filtering data and simple logic. See #PxQueryFilterData #PxQueryFilterCallback
     * Note: Filtering is not executed for a cached shape if supplied; instead, if a hit is found, it is assumed to be a blocking hit.
     * Note: Using past touching hits as cache will produce incorrect behavior since the cached hit will always be treated as blocking.
     * @return True if any touching or blocking hits were found or any hit was found in case PxQueryFlag::eANY_HIT was specified.
     * <p>
     * <b>Note:</b> eBLOCK should not be returned from user filters for overlap(). Doing so will result in undefined behavior, and a warning will be issued.
     * <b>Note:</b> If the PxQueryFlag::eNO_BLOCK flag is set, the eBLOCK will instead be automatically converted to an eTOUCH and the warning suppressed.
     * @see PxOverlapCallback
     * @see PxHitFlags
     * @see PxQueryFilterData
     * @see PxQueryFilterCallback
     */
    public boolean overlap(PxGeometry geometry, PxTransform pose, PxOverlapCallback hitCall, PxQueryFilterData filterData) {
        checkNotNull();
        return _overlap(address, geometry.getAddress(), pose.getAddress(), hitCall.getAddress(), filterData.getAddress());
    }
    private static native boolean _overlap(long address, long geometry, long pose, long hitCall, long filterData);

}
