package physx.physics;

import physx.NativeObject;
import physx.common.PxBase;
import physx.common.PxBounds3;

/**
 * PxActor is the base class for the main simulation objects in the physics SDK.
 * <p>
 * The actor is owned by and contained in a PxScene.
 */
public class PxActor extends PxBase {

    protected PxActor() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxActor wrapPointer(long address) {
        return address != 0L ? new PxActor(address) : null;
    }
    
    public static PxActor arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxActor(long address) {
        super(address);
    }

    // Attributes

    /**
     * user can assign this to whatever, usually to create a 1:1 relationship with a user object.
     */
    public NativeObject getUserData() {
        checkNotNull();
        return NativeObject.wrapPointer(_getUserData(address));
    }
    private static native long _getUserData(long address);

    /**
     * user can assign this to whatever, usually to create a 1:1 relationship with a user object.
     */
    public void setUserData(NativeObject value) {
        checkNotNull();
        _setUserData(address, value.getAddress());
    }
    private static native void _setUserData(long address, long value);

    // Functions

    /**
     * Retrieves the type of actor.
     * @return The actor type of the actor.
     */
    public PxActorTypeEnum getType() {
        checkNotNull();
        return PxActorTypeEnum.forValue(_getType(address));
    }
    private static native int _getType(long address);

    /**
     * Retrieves the scene which this actor belongs to.
     * @return Owner Scene. NULL if not part of a scene.
     * @see PxScene
     */
    public PxScene getScene() {
        checkNotNull();
        return PxScene.wrapPointer(_getScene(address));
    }
    private static native long _getScene(long address);

    /**
     * Sets a name string for the object that can be retrieved with getName().
     * <p>
     * This is for debugging and is not used by the SDK. The string is not copied by the SDK, 
     * only the pointer is stored.
     * @param name String to set the objects name to.
     * <p>
     * <b>Default:</b> NULL
     * @see #getName
     */
    public void setName(String name) {
        checkNotNull();
        _setName(address, name);
    }
    private static native void _setName(long address, String name);

    /**
     * Retrieves the name string set with setName().
     * @return Name string associated with object.
     * @see #setName
     */
    public String getName() {
        checkNotNull();
        return _getName(address);
    }
    private static native String _getName(long address);

    /**
     * Retrieves the axis aligned bounding box enclosing the actor.
     * <p>
     * <b>Note:</b> It is not allowed to use this method while the simulation is running (except during PxScene::collide(),
     * in PxContactModifyCallback or in contact report callbacks).
     * @return The actor's bounding box.
     * @see physx.common.PxBounds3
     */
    public PxBounds3 getWorldBounds() {
        checkNotNull();
        return PxBounds3.wrapPointer(_getWorldBounds(address));
    }
    private static native long _getWorldBounds(long address);

    /**
     * Retrieves the axis aligned bounding box enclosing the actor.
     * <p>
     * <b>Note:</b> It is not allowed to use this method while the simulation is running (except during PxScene::collide(),
     * in PxContactModifyCallback or in contact report callbacks).
     * @param inflation  Scale factor for computed world bounds. Box extents are multiplied by this value.
     * @return The actor's bounding box.
     * @see physx.common.PxBounds3
     */
    public PxBounds3 getWorldBounds(float inflation) {
        checkNotNull();
        return PxBounds3.wrapPointer(_getWorldBounds(address, inflation));
    }
    private static native long _getWorldBounds(long address, float inflation);

    /**
     * Raises or clears a particular actor flag.
     * <p>
     * See the list of flags #PxActorFlag
     * <p>
     * <b>Sleeping:</b> Does <b>NOT</b> wake the actor up automatically.
     * @param flag  The PxActor flag to raise(set) or clear. See #PxActorFlag.
     * @param value The boolean value to assign to the flag.
     * @see #getActorFlags
     */
    public void setActorFlag(PxActorFlagEnum flag, boolean value) {
        checkNotNull();
        _setActorFlag(address, flag.value, value);
    }
    private static native void _setActorFlag(long address, int flag, boolean value);

    /**
     * Sets the actor flags.
     * <p>
     * See the list of flags #PxActorFlag
     * @see #setActorFlag
     */
    public void setActorFlags(PxActorFlags flags) {
        checkNotNull();
        _setActorFlags(address, flags.getAddress());
    }
    private static native void _setActorFlags(long address, long flags);

    /**
     * Reads the PxActor flags.
     * <p>
     * See the list of flags #PxActorFlag
     * @return The values of the PxActor flags.
     * @see #setActorFlag
     */
    public PxActorFlags getActorFlags() {
        checkNotNull();
        return PxActorFlags.wrapPointer(_getActorFlags(address));
    }
    private static native long _getActorFlags(long address);

    /**
     * Assigns dynamic actors a dominance group identifier.
     * <p>
     * PxDominanceGroup is a 5 bit group identifier (legal range from 0 to 31).
     * <p>
     * The PxScene::setDominanceGroupPair() lets you set certain behaviors for pairs of dominance groups.
     * By default every dynamic actor is created in group 0.
     * <p>
     * <b>Default:</b> 0
     * <p>
     * <b>Sleeping:</b> Changing the dominance group does <b>NOT</b> wake the actor up automatically.
     * @param dominanceGroup The dominance group identifier. <b>Range:</b> [0..31]
     * @see #getDominanceGroup
     */
    public void setDominanceGroup(byte dominanceGroup) {
        checkNotNull();
        _setDominanceGroup(address, dominanceGroup);
    }
    private static native void _setDominanceGroup(long address, byte dominanceGroup);

    /**
     * Retrieves the value set with setDominanceGroup().
     * @return The dominance group of this actor.
     * @see #setDominanceGroup
     */
    public byte getDominanceGroup() {
        checkNotNull();
        return _getDominanceGroup(address);
    }
    private static native byte _getDominanceGroup(long address);

    /**
     * Sets the owner client of an actor.
     * <p>
     * This cannot be done once the actor has been placed into a scene.
     * <p>
     * <b>Default:</b> PX_DEFAULT_CLIENT
     */
    public void setOwnerClient(byte inClient) {
        checkNotNull();
        _setOwnerClient(address, inClient);
    }
    private static native void _setOwnerClient(long address, byte inClient);

    /**
     * Returns the owner client that was specified at creation time.
     * <p>
     * This value cannot be changed once the object is placed into the scene.
     */
    public byte getOwnerClient() {
        checkNotNull();
        return _getOwnerClient(address);
    }
    private static native byte _getOwnerClient(long address);

}
