package physx.particles;

import physx.PlatformChecks;
import physx.physics.PxBaseMaterial;

/**
 * Material class to represent a set of particle material properties.
 */
public class PxParticleMaterial extends PxBaseMaterial {

    static {
        PlatformChecks.requirePlatform(3, "physx.particles.PxParticleMaterial");
    }

    protected PxParticleMaterial() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxParticleMaterial wrapPointer(long address) {
        return address != 0L ? new PxParticleMaterial(address) : null;
    }
    
    public static PxParticleMaterial arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxParticleMaterial(long address) {
        super(address);
    }

    // Functions

    /**
     * Sets friction
     * @param friction Friction. <b>Range:</b> [0, PX_MAX_F32)
     */
    public void setFriction(float friction) {
        checkNotNull();
        _setFriction(address, friction);
    }
    private static native void _setFriction(long address, float friction);

    /**
     * Retrieves the friction value.
     * @return The friction value.
     */
    public float getFriction() {
        checkNotNull();
        return _getFriction(address);
    }
    private static native float _getFriction(long address);

    /**
     * Sets velocity damping term
     * @param damping Velocity damping term. <b>Range:</b> [0, PX_MAX_F32)
     */
    public void setDamping(float damping) {
        checkNotNull();
        _setDamping(address, damping);
    }
    private static native void _setDamping(long address, float damping);

    /**
     * Retrieves the velocity damping term
     * @return The velocity damping term.
     */
    public float getDamping() {
        checkNotNull();
        return _getDamping(address);
    }
    private static native float _getDamping(long address);

    /**
     * Sets adhesion term
     * @param adhesion adhesion coefficient. <b>Range:</b> [0, PX_MAX_F32)
     */
    public void setAdhesion(float adhesion) {
        checkNotNull();
        _setAdhesion(address, adhesion);
    }
    private static native void _setAdhesion(long address, float adhesion);

    /**
     * Retrieves the adhesion term
     * @return The adhesion term.
     */
    public float getAdhesion() {
        checkNotNull();
        return _getAdhesion(address);
    }
    private static native float _getAdhesion(long address);

    /**
     * Sets gravity scale term
     * @param scale gravity scale coefficient. <b>Range:</b> (-PX_MAX_F32, PX_MAX_F32)
     */
    public void setGravityScale(float scale) {
        checkNotNull();
        _setGravityScale(address, scale);
    }
    private static native void _setGravityScale(long address, float scale);

    /**
     * Retrieves the gravity scale term
     * @return The gravity scale term.
     */
    public float getGravityScale() {
        checkNotNull();
        return _getGravityScale(address);
    }
    private static native float _getGravityScale(long address);

    /**
     * Sets material adhesion radius scale. This is multiplied by the particle rest offset to compute the fall-off distance
     * at which point adhesion ceases to operate.
     * @param scale Material adhesion radius scale. <b>Range:</b> [0, PX_MAX_F32)
     */
    public void setAdhesionRadiusScale(float scale) {
        checkNotNull();
        _setAdhesionRadiusScale(address, scale);
    }
    private static native void _setAdhesionRadiusScale(long address, float scale);

    /**
     * Retrieves the adhesion radius scale.
     * @return The adhesion radius scale.
     */
    public float getAdhesionRadiusScale() {
        checkNotNull();
        return _getAdhesionRadiusScale(address);
    }
    private static native float _getAdhesionRadiusScale(long address);

}
