package physx.particles;

import physx.NativeObject;
import physx.PlatformChecks;
import physx.common.PxVec4;

/**
 * A particle buffer used to simulate particle cloth.
 * <p>
 * See #PxPhysics::createParticleClothBuffer.
 */
public class PxParticleClothBuffer extends PxParticleBuffer {

    static {
        PlatformChecks.requirePlatform(3, "physx.particles.PxParticleClothBuffer");
    }

    protected PxParticleClothBuffer() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxParticleClothBuffer wrapPointer(long address) {
        return address != 0L ? new PxParticleClothBuffer(address) : null;
    }
    
    public static PxParticleClothBuffer arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxParticleClothBuffer(long address) {
        super(address);
    }

    // Functions

    /**
     * Get rest positions for this particle buffer.
     * @return A pointer to a device buffer containing the rest positions packed as PxVec4(pos.x, pos.y, pos.z, 0.0f).
     */
    public PxVec4 getRestPositions() {
        checkNotNull();
        return PxVec4.wrapPointer(_getRestPositions(address));
    }
    private static native long _getRestPositions(long address);

    /**
     * Get the triangle indices for this particle buffer.
     * @return A pointer to a device buffer containing the triangle indices for this cloth buffer.
     */
    public NativeObject getTriangles() {
        checkNotNull();
        return NativeObject.wrapPointer(_getTriangles(address));
    }
    private static native long _getTriangles(long address);

    /**
     * Set the number of triangles for this particle buffer.
     * @param nbTriangles The number of triangles for this particle cloth buffer.
     */
    public void setNbTriangles(int nbTriangles) {
        checkNotNull();
        _setNbTriangles(address, nbTriangles);
    }
    private static native void _setNbTriangles(long address, int nbTriangles);

    /**
     * Get the number of triangles for this particle buffer.
     * @return The number triangles for this cloth buffer.
     */
    public int getNbTriangles() {
        checkNotNull();
        return _getNbTriangles(address);
    }
    private static native int _getNbTriangles(long address);

    /**
     * Get the number of springs in this particle buffer.
     * @return The number of springs in this cloth buffer.
     */
    public int getNbSprings() {
        checkNotNull();
        return _getNbSprings(address);
    }
    private static native int _getNbSprings(long address);

    /**
     * Get the springs for this particle buffer.
     * @return A pointer to a device buffer containing the springs for this cloth buffer.
     */
    public PxParticleSpring getSprings() {
        checkNotNull();
        return PxParticleSpring.wrapPointer(_getSprings(address));
    }
    private static native long _getSprings(long address);

    /**
     * Set cloths for this particle buffer.
     * @param cloths A pointer to a PxPartitionedParticleCloth.
     * <p>
     * See #PxPartitionedParticleCloth, #PxParticleClothPreProcessor
     */
    public void setCloths(PxPartitionedParticleCloth cloths) {
        checkNotNull();
        _setCloths(address, cloths.getAddress());
    }
    private static native void _setCloths(long address, long cloths);

}
