package physx.geometry;

import physx.NativeObject;
import physx.common.PxBounds3;
import physx.common.PxRefCounted;
import physx.common.PxVec3;
import physx.support.PxU32ConstPtr;

/**
 * A triangle mesh, also called a 'polygon soup'.
 * <p>
 * It is represented as an indexed triangle list. There are no restrictions on the
 * triangle data. 
 * <p>
 * To avoid duplicating data when you have several instances of a particular 
 * mesh positioned differently, you do not use this class to represent a 
 * mesh object directly. Instead, you create an instance of this mesh via
 * the PxTriangleMeshGeometry and PxShape classes.
 * <p>
 * <h3>Creation</h3>
 * <p>
 * To create an instance of this class call PxPhysics::createTriangleMesh(),
 * and release() to delete it. This is only possible
 * once you have released all of its PxShape instances.
 * <p>
 * <h3>Visualizations:</h3>
 * \li #PxVisualizationParameter::eCOLLISION_AABBS
 * \li #PxVisualizationParameter::eCOLLISION_SHAPES
 * \li #PxVisualizationParameter::eCOLLISION_AXES
 * \li #PxVisualizationParameter::eCOLLISION_FNORMALS
 * \li #PxVisualizationParameter::eCOLLISION_EDGES
 * @see physx.cooking.PxTriangleMeshDesc
 * @see PxTriangleMeshGeometry
 * @see physx.physics.PxShape
 */
public class PxTriangleMesh extends PxRefCounted {

    protected PxTriangleMesh() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxTriangleMesh wrapPointer(long address) {
        return address != 0L ? new PxTriangleMesh(address) : null;
    }
    
    public static PxTriangleMesh arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxTriangleMesh(long address) {
        super(address);
    }

    // Functions

    /**
     * Returns the number of vertices.
     * @return number of vertices
     * @see #getVertices
     */
    public int getNbVertices() {
        checkNotNull();
        return _getNbVertices(address);
    }
    private static native int _getNbVertices(long address);

    /**
     * Returns the vertices.
     * @return array of vertices
     * @see #getNbVertices
     */
    public PxVec3 getVertices() {
        checkNotNull();
        return PxVec3.wrapPointer(_getVertices(address));
    }
    private static native long _getVertices(long address);

    /**
     * Returns all mesh vertices for modification.
     * <p>
     * This function will return the vertices of the mesh so that their positions can be changed in place.
     * After modifying the vertices you must call refitBVH for the refitting to actually take place.
     * This function maintains the old mesh topology (triangle indices). 
     * @return  inplace vertex coordinates for each existing mesh vertex.
     * <p>
     * <b>Note:</b> It is recommended to use this feature for scene queries only.
     * <b>Note:</b> Size of array returned is equal to the number returned by getNbVertices().
     * <b>Note:</b> This function operates on cooked vertex indices.
     * <b>Note:</b> This means the index mapping and vertex count can be different from what was provided as an input to the cooking routine.
     * <b>Note:</b> To achieve unchanged 1-to-1 index mapping with orignal mesh data (before cooking) please use the following cooking flags:
     * <b>Note:</b> eWELD_VERTICES = 0, eDISABLE_CLEAN_MESH = 1.
     * <b>Note:</b> It is also recommended to make sure that a call to validateTriangleMesh returns true if mesh cleaning is disabled.
     * @see #getNbVertices
     * @see #refitBVH
     */
    public PxVec3 getVerticesForModification() {
        checkNotNull();
        return PxVec3.wrapPointer(_getVerticesForModification(address));
    }
    private static native long _getVerticesForModification(long address);

    /**
     * Refits BVH for mesh vertices.
     * <p>
     * This function will refit the mesh BVH to correctly enclose the new positions updated by getVerticesForModification.
     * Mesh BVH will not be reoptimized by this function so significantly different new positions will cause significantly reduced performance. 
     * @return New bounds for the entire mesh.
     * <p>
     * <b>Note:</b> For PxMeshMidPhase::eBVH34 trees the refit operation is only available on non-quantized trees (see PxBVH34MidphaseDesc::quantized)
     * <b>Note:</b> PhysX does not keep a mapping from the mesh to mesh shapes that reference it.
     * <b>Note:</b> Call PxShape::setGeometry on each shape which references the mesh, to ensure that internal data structures are updated to reflect the new geometry.
     * <b>Note:</b> PxShape::setGeometry does not guarantee correct/continuous behavior when objects are resting on top of old or new geometry.
     * <b>Note:</b> It is also recommended to make sure that a call to validateTriangleMesh returns true if mesh cleaning is disabled.
     * <b>Note:</b> Active edges information will be lost during refit, the rigid body mesh contact generation might not perform as expected.
     * @see #getNbVertices
     * @see #getVerticesForModification
     */
    public PxBounds3 refitBVH() {
        checkNotNull();
        return PxBounds3.wrapPointer(_refitBVH(address));
    }
    private static native long _refitBVH(long address);

    /**
     * Returns the number of triangles.
     * @return number of triangles
     * @see #getTriangles
     * @see #getTrianglesRemap
     */
    public int getNbTriangles() {
        checkNotNull();
        return _getNbTriangles(address);
    }
    private static native int _getNbTriangles(long address);

    /**
     * Returns the triangle indices.
     * <p>
     * The indices can be 16 or 32bit depending on the number of triangles in the mesh.
     * Call getTriangleMeshFlags() to know if the indices are 16 or 32 bits.
     * <p>
     * The number of indices is the number of triangles * 3.
     * @return array of triangles
     * @see #getNbTriangles
     * @see #getTriangleMeshFlags
     * @see #getTrianglesRemap
     */
    public NativeObject getTriangles() {
        checkNotNull();
        return NativeObject.wrapPointer(_getTriangles(address));
    }
    private static native long _getTriangles(long address);

    /**
     * Reads the PxTriangleMesh flags.
     * <p>
     * See the list of flags #PxTriangleMeshFlag
     * @return The values of the PxTriangleMesh flags.
     * @see PxTriangleMesh
     */
    public PxTriangleMeshFlags getTriangleMeshFlags() {
        checkNotNull();
        return PxTriangleMeshFlags.wrapPointer(_getTriangleMeshFlags(address));
    }
    private static native long _getTriangleMeshFlags(long address);

    /**
     * Returns the triangle remapping table.
     * <p>
     * The triangles are internally sorted according to various criteria. Hence the internal triangle order
     * does not always match the original (user-defined) order. The remapping table helps finding the old
     * indices knowing the new ones:
     * <p>
     *  remapTable[ internalTriangleIndex ] = originalTriangleIndex
     * @return the remapping table (or NULL if 'PxCookingParams::suppressTriangleMeshRemapTable' has been used)
     * @see #getNbTriangles
     * @see #getTriangles
     */
    public PxU32ConstPtr getTrianglesRemap() {
        checkNotNull();
        return PxU32ConstPtr.wrapPointer(_getTrianglesRemap(address));
    }
    private static native long _getTrianglesRemap(long address);

    /**
     * Returns material table index of given triangle
     * <p>
     * This function takes a post cooking triangle index.
     * @param triangleIndex (internal) index of desired triangle
     * @return Material table index, or 0xffff if no per-triangle materials are used
     */
    public short getTriangleMaterialIndex(int triangleIndex) {
        checkNotNull();
        return _getTriangleMaterialIndex(address, triangleIndex);
    }
    private static native short _getTriangleMaterialIndex(long address, int triangleIndex);

    /**
     * Returns the local-space (vertex space) AABB from the triangle mesh.
     * @return local-space bounds
     */
    public PxBounds3 getLocalBounds() {
        checkNotNull();
        return PxBounds3.wrapPointer(_getLocalBounds(address));
    }
    private static native long _getLocalBounds(long address);

}
