package physx.extensions;


/**
 * Describes a one-sided linear limit.
 */
public class PxJointLinearLimit extends PxJointLimitParameters {

    protected PxJointLinearLimit() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxJointLinearLimit wrapPointer(long address) {
        return address != 0L ? new PxJointLinearLimit(address) : null;
    }
    
    public static PxJointLinearLimit arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxJointLinearLimit(long address) {
        super(address);
    }

    // Constructors

    /**
     * construct a linear soft limit 
     * @param extent the extent of the limit
     * @param spring the stiffness and damping parameters for the limit spring
     * @see PxJointLimitParameters
     * @see physx.common.PxTolerancesScale
     */
    public PxJointLinearLimit(float extent, PxSpring spring) {
        address = _PxJointLinearLimit(extent, spring.getAddress());
    }
    private static native long _PxJointLinearLimit(float extent, long spring);

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * the extent of the limit. 
     * <p>
     * <b>Range:</b> (0, PX_MAX_F32) <br>
     * <b>Default:</b> PX_MAX_F32
     */
    public float getValue() {
        checkNotNull();
        return _getValue(address);
    }
    private static native float _getValue(long address);

    /**
     * the extent of the limit. 
     * <p>
     * <b>Range:</b> (0, PX_MAX_F32) <br>
     * <b>Default:</b> PX_MAX_F32
     */
    public void setValue(float value) {
        checkNotNull();
        _setValue(address, value);
    }
    private static native void _setValue(long address, float value);

}
